/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef TABLE_EVAL_HH
#define TABLE_EVAL_HH

#include "calc_engine/engine.hh"
#include "calc_engine/str_stack.hh"
#include "table.hh"
#include <map>

namespace table {

    /**  The eval class is used to evaluate tablepgm symbolic expressions with
      *  calc_engine. 
      *  @memo Evaluate a table expression.
      *  @author John Zweizig (john.zweizig@ligo.org)
      *  @version $Id: eval.hh 6303 2010-09-16 02:39:45Z john.zweizig@LIGO.ORG $
      */
    class eval : public Engine {
    public:
	/**  Dictionary data type.
	  */
	typedef std::map<std::string, std::string> dict_type;

    public:
	/**  Construct an evaluator and parse the symbolic expression. 
	  *  @memo Constructor
	  *  @param t Reference to the table for which the expression is 
	  *           to be evaluated.
	  *  @param cmd The symbolic expression.
	  *  @param debug Debugging printout level.
	  */
	eval(const Table& t, const std::string& cmd, int debug=0);

	/**  Add a symbol to the dictionary.
	  *  @memo Add a symbol.
	  *  @param name Symbol name
	  *  @param value Symbol value.
	  */
	void addSymbol(const std::string& name, const std::string& value);

	/**  Evaluate the symbolic expression for the specified row and 
	  *  return a string value.
	  *  @memo Evaluate a row.
	  *  @param row Row number
	  *  @return Result string.
	  */ 
	std::string evalRow(int row);

	/**  Evaluate the symbolic expression for the specified row and 
	  *  return a boolean value.
	  *  @memo Evaluate a row.
	  *  @param row Row number
	  *  @return Boolean result.
	  */ 
	bool evalRow_bool(int row);

	/**  Evaluate the symbolic expression for the specified row and 
	  *  return a numeric value.
	  *  @memo Evaluate a row.
	  *  @param row Row number
	  *  @return Numeric result.
	  */ 
	double evalRow_numeric(int row);

	/**  Evaluate a specified function. The arguments are left on the 
	  *  \c calc_engine and the result is pushed onto the stack at exit.
	  *  @memo Evaluate a function.
	  *  @param sym Function name
	  *  @param N   Number of arguments.
	  */
	virtual void eval_funct(const std::string& sym, int N);

	/**  Push the value of a specified symbol onto the \c calc_engine
	  *  stack.
	  *  @memo Push a symbol value
	  *  @param sym Symbol name.
	  */
	virtual void push_symbol(const std::string& sym);

	/**  Preset the internal dictionary with the contents of an external
	  *  dictionary.
	  *  @memo Set the internal dictionary.
	  *  @param dict Constant reference to the external dictionary.
	  */
	void setDictionary(const dict_type& dict);

    private:
	const Table& mTable;
	dict_type mDictionary;
	Column::row_index mRow;
	str_stack mStack;
    };

} // namespace table

#endif // !defined(TABLE_EVAL_HH)
