/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//    Dump frames from shared memory partition.
//
#include "lsmp_con.hh"
#include "Interval.hh"
#include "Time.hh"
#include "SigFlag.hh"
#include <cstdlib>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <fstream>
#include <iostream>
#include <vector>
#include <string>
#include <cmath>
#include <unistd.h>

using namespace std;

//======================================  Parameter constants
const int def_stat_count = 100;

//======================================  Print leading digits
static int
put(string& s, int N, int nDig) {
    int r = 0;
    if (N < 10) {
	s += (N + '0');
    } else {
	r = put(s, N/10, nDig) + 1;
	if (!nDig || r < nDig) {
	    s += (N%10 + '0');
	}
    }
    return r;
}

//======================================  Main function.
int 
main(int argc, const char *argv[]) {
    bool syntax(false), debug(false), demonize(false);
    string partition;
    const char *name = getenv("LIGOSMPART");
    if (name) partition = name;
    Interval delta_t=1;
    string filetemp("X-R-%g-%d.gwf");
    string lastfile, logfile;
    int stats(0), nRing(0);
    for (int i=1; i<argc; ++i) {
	string argi = argv[i];
	if (argi == "--daemon") {
	    demonize = true;
	} else if (argi == "--debug") {
	    debug = true;
	} else if (argi == "-d" || argi == "--delta") {
	    delta_t = strtod(argv[++i], 0);
	} else if (argi == "-f" || argi == "--filename") {
	    filetemp = argv[++i];
	} else if (argi == "--help") {
	    syntax = true; // Just print out the command syntax.
	} else if (argi == "-l" || argi == "--last") {
	    lastfile = argv[++i];
	} else if (argi == "--log") {
	    logfile = argv[++i];
	} else if (argi == "-p" || argi == "--partition") {
	    partition = argv[++i];
	} else if (argi == "-r" || argi == "--ring") {
	    nRing = strtol(argv[++i], 0, 0);
	} else if (argi == "-s" || argi == "--statistics") {
	    ++i;
	    if (i >= argc || *argv[i] == '-') {
		stats = def_stat_count;
		--i;
	    } else {
		stats = strtol(argv[i], 0, 0);
	    }
	} else {
	    syntax = true;
	    cerr << "Invalid argument: " << argi << endl;
	}
    }

    if (partition.empty()) syntax = true;
    if (syntax) {
        cerr << "Syntax:" << endl;
        cerr << "Frame_Log [--delta <delta-t>] [--filename <file-name>] "
	     << "[--partition <part>] \\" << endl;
	cerr << "          [--last <time-stamp-file>] [--statistics [<nframe>]]"
	     << " [--ring <nbuf>] \\" << endl;
	cerr << "          [--daemon] [--debug] [--help]" << endl;
	cerr << "<file-name> A file path template with replacement sequences:" 
	     << endl;
	cerr << "              %d     duration [X-R-%g-%d.gwf]" << endl;
	cerr << "              %<n>g  first <n> digits of gps" << endl;
	cerr << "              %<n>r  gps / 10 ^ <n>" << endl;
	cerr << "            A file path is generated for each frame gps. Any"
	     << endl;
	cerr << "            missing directories are created [X-R-%g-%d.gwf]" 
	     << endl;
	cerr << "<delta-t>   The frame length (used to define" << endl;
	cerr << "            the output frame file name) [1]" << endl;
	cerr << "<part>      Input partition [$LIGOSMPART]" << endl;
	cerr << "<time-stamp-file> File to receive time stamp of "
	     << "most recent frame" << endl;
	cerr << "<nframe>    Number of frames to be averaged for statics"
	     << endl;
	cerr << "--daemon    Launch the process as a daemon." << endl;
	cerr << "--debug     Print debugging information." << endl;
	cerr << "--help      Print the command line syntax." << endl;
	return 1;
    }

    //-----------------------------------  Fork a daemon if requested
    if (demonize) {
	switch (fork()) {
	case -1:
	    exit(1);
	case 0:
	    break;
	default:
	    exit(0);
	}
    }

    //-----------------------------------  Optionally switch to a log file
    if (!logfile.empty()) {
	int flags = O_WRONLY | O_CREAT;
	int fd = open(logfile.c_str(), flags, 0664);
	close(1); // stdout
	if (dup(fd) != 1) {
	    cerr << "stdout not diverted!" << endl;
	}
	close(2); // stderr
	if (dup(fd) != 2) {
	    cout << "stderr not diverted!" << endl;
	}
	close(fd);
    }

    //----------------------------------  Open the partition for input
    LSMP_CON cons(partition.c_str(), 0);

    //----------------------------------  Handle signals
    SigFlag mTerm;
    mTerm.add(SIGTERM);
    mTerm.add(SIGHUP);
    mTerm.add(SIGINT);

    //----------------------------------  Open the output file
    double tAvg(0), tSig(0), tMin(0), tMax(0);
    int    nAvg(0), nErr(0), nBuf(0);
    vector<string> ring_file(nRing);

    //----------------------------------  Open the output file
    while (!mTerm) {

	//------------------------------  Get a buffer
	const char* b = cons.get_buffer(0);
	if (!b) {
	    cerr << "Unable to fetch frame from partition: " << partition 
		 << endl;
	    return 3;
	}

	//----------------------------------  get the length and gps time
	LSMP::eventid_type gps = cons.getEvtID();
	long length = cons.getLength();
	if (!length) {
	    std::cerr << "Zero length lsmp buffer skipped, gps=" << gps
		      << endl;
	    cons.free_buffer();
	    nErr++;
	    continue;
	}

	//----------------------------------  Get the ring buffer number
	if (nRing) {
	    ++nBuf;
	    if (nBuf == nRing) nBuf = 0;
	}

	//----------------------------------  Build a string
	int N = filetemp.size();
	string file;
	for (int i=0; i<N; ++i) {
	    int nDig = 0;
	    if (filetemp[i] == '%') {
		i++;
		while (filetemp[i] >= '0' && filetemp[i] < '9') {
		    nDig *= 10;
		    nDig += (filetemp[i++] -'0');
		}
		switch (filetemp[i]) {
		case 'b':
		    put(file, nBuf, nDig);
		    break;
		case 'd':
		    put(file, delta_t, nDig);
		    break;
		case 'g':
		    put(file, gps, nDig);
		    break;
		default:
		    file += filetemp[i];
		}
	    } else {
		file += filetemp[i];
	    }
	}

	//----------------------------------  Delete an output file
	if (nRing) {
	    if (!ring_file[nBuf].empty()) {
		int ercd = unlink(ring_file[nBuf].c_str());
		if (ercd) {
		    cerr << "Unable to erase: " << ring_file[nBuf] << endl;
		    perror("erase error");
		}
	    }
	    ring_file[nBuf] = file;
	}

	//----------------------------------  Get the directory name
	string::size_type inxdir = file.find_last_of('/');
	if (inxdir != string::npos) {
	    string dirname = file.substr(0, inxdir);
	    if (dirname != file) {
		if (access(dirname.c_str(), W_OK)) {
		    cout << "Creating directory: " << dirname << endl;
		    int rc = mkdir(dirname.c_str(), 0755);
		    if (rc) perror("Error creating directory");
		}
	    }
	}

	//----------------------------------  Generate a temp. output file name
	string tempfile, fileroot;
	if (inxdir != string::npos) {
	    inxdir++;
	    tempfile = file.substr(0,inxdir);
	    fileroot = file.substr(inxdir);
	} else {
	    fileroot = file;
	}
	tempfile += "."; 
	tempfile += fileroot.substr(0, fileroot.find_last_of(".")) + ".tmp";
	if (debug) cerr << "Frame_Log: Temp file name is " << tempfile << endl;

	//----------------------------------  Open the output file
	std::ofstream fout(tempfile.c_str(), ios::out);
	if (!fout.good()) {
	    std::cerr << "Unable to open file " << file << std::endl;
	    nErr++;
	    return 4;
	}

	//----------------------------------  Write a frame.
	fout.exceptions(ios::badbit | ios::failbit);
	try {
	    fout.write(b, length);
	} catch (std::exception& e) {
	    std::cerr << "Error writing " << tempfile << ": " << e.what()
		      << endl;
	    cons.free_buffer();
	    fout.close();
	    nErr++;
	    continue;
	}
	cons.free_buffer();
	fout.exceptions(ios::goodbit);
	fout.close();
	rename(tempfile.c_str(), file.c_str());	

	//----------------------------------  Update the last file time
	if (!lastfile.empty()) {
	    string tempname;
	    string::size_type slash = lastfile.find_last_of('/');
	    if (slash != string::npos) tempname = lastfile.substr(0, slash+1);
	    tempname += "latest_temp";
	    ofstream tstr(tempname.c_str(), ios_base::out);
	    if (!tstr.is_open()) {
		cerr << "Unable to write time-stamp temp file: " 
		     << tempname << endl;
		nErr++;
	    } else {
		tstr << gps << endl;
		tstr.close();
		if (tstr.fail()) {
		    cerr << "Stream failed when writing file: " 
			 << tempname << endl;
		}
		rename(tempname.c_str(), lastfile.c_str());
		nErr++;
	    }
	}

	//----------------------------------  Calculate statistics.
	if (stats > 0) {
	    double dt = double(Now() - Time(gps));
	    if (!nAvg) {
		tMin = dt;
		tMax = dt;
	    } else if (dt > tMax) {
		tMax = dt;
	    } else if (dt < tMin) {
		tMin = dt;
	    }
	    tAvg += dt;
	    tSig += dt * dt;
	    nAvg++;
	    if (nAvg >= stats) {
		tAvg /= nAvg;
		tSig  = tSig / nAvg - tAvg * tAvg;
		if (tSig > 0) tSig = sqrt(tSig);
		else          tSig = 0;
		cout << "Time statistics for " << nAvg << " frames: avg=" 
		     << tAvg<< " sigma=" << tSig << " min=" << tMin 
		     << " max=" << tMax << " lost="<< nErr << endl;
		nAvg = 0;
		tAvg = 0;
		tSig = 0;
		nErr = 0;
	    }
	}
    }
    return 0;
}
