/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//    Designer dataset writer version 1.2
//
#include "FrWrite.hh"
#include "Dacc.hh"
#include "Time.hh"
#include "ParseLine.hh"
#include "OperStateCondList.hh"
#include "framecpp/FrameH.hh"
#include "framecpp/FrAdcData.hh"
// #include "framecpp/Dimension.hh"
#include "framecpp/FrVect.hh"

#if LDAS_VERSION_NUMBER < 200000
#include "general/gpstime.hh"
using General::GPSTime;
#else
#include "framecpp/GPSTime.hh"
using FrameCPP::GPSTime;
#endif

#if LDAS_VERSION_NUMBER < 109360
#include "framecpp/OFrameStream.hh"
#else
#include "framecpp/Common/IOStream.hh"
using FrameCPP::Common::FrameBuffer;
#endif
#include <fstream>
#include <cstring>
#include <cstdio>
#include <unistd.h>
#include <sys/stat.h>

using namespace std;
using FrameCPP::FrameH;
using FrameCPP::FrRawData;
using FrameCPP::FrAdcData;
using FrameCPP::Dimension;
using FrameCPP::FrVect;

//======================================  Generate the main function.
EXECDMT(FrWrite)
static const char* trigName = "Trigger";

//======================================  Translate from General::GPSTime
inline Time
GetGPSTime(const GPSTime& T) {
    return Time(T.GetSeconds(), T.GetNanoseconds());
}

//======================================  Constructor
FrWrite::FrWrite(int argc, const char *argv[]) 
  : DMTBase(argc, argv), mFrOut(1), mFrDone(0), mFrTotal(0), mFileOut(0),
    mRepeat(1), mSequence(0), mMode(m_CopyAll), mResize(0), mTOC(false), 
    mWriter(0),
#if LDAS_VERSION_NUMBER < 109360
    mStream(0),
#else
    mFrameBuf(0),
#endif
    mOSC(0), mBackFrame(0), mTrigFrame(1), mBackWrite(0), mBackValid(0)
{
    const char* synText[] = {
      "FrWrite copies DMT online frame data to a file.",
      " ",
      "Syntax: ",
      "FrWrite [-ofile <file>] [-nframe <nframe>] [-cfile <conf>]",
      "        [-repeat <nfiles>] [+toc] [<dmt args>]",
      "<file> specifies an output file name template and may include any",
      "TimeStr() escape sequence. If <file> is not specified,",
      "'FrWrite-%Y.%m.%d-%H.%N.%S.F'is used. <nframe> frames are written to",
      "each of <nfiles> files. <nframe> and <nfiles> default to 1. If ",
      "'-repeat 0' is specified FrWrite will write files continuously until",
      "it is killed. <conf> names a configuration file listing channels to ",
      "be written. If unspecified, all channels are included. If +toc is",
      "specified, a table of contents will be written on all frame files.",
      0
    };

    const char* file = "FrWrite-%Y.%m.%d-%H.%N.%S.F";
    const char* cfile = 0;
    bool ra = false;

    //--------------------------------  Parse the arguments
    for (int i=1 ; i<argc ; i++) {
        string argi = argv[i];
        if (argi == "-ofile") {
	    file = argv[++i];
        } else if (argi == "-cfile") {
	    cfile = argv[++i];
        } else if (argi == "-nframe") {
	    mFrOut = strtol(argv[++i], 0, 0);
        } else if (argi == "-repeat") {
	    mRepeat = strtol(argv[++i], 0, 0);
        } else if (argi == "-resize") {
	    mResize = strtod(argv[++i], 0);
        } else if (argi == "-sequence") {
	    mSequence = strtol(argv[++i], 0, 0);
        } else if (argi == "-toc") {
	    mTOC = false;
        } else if (argi == "+toc") {
	    mTOC = true;
        } else if (argi == "+readall" || argi == "+ra") {
	    ra = true;
        } else if (argi == "-backframe" || argi == "-bf") {
	    mBackFrame = strtol(argv[++i], 0, 0);
        } else if (argi == "-trigframe" || argi == "-tf") {
	    mTrigFrame = strtol(argv[++i], 0, 0);
        } else if (argi == "-trigger") {
	    mOSC = new OperStateCondList(getDacc());
	    mOSC->readConfig(argv[++i]);
	    if (mOSC->find(trigName) == mOSC->end()) {
	        cerr << "No valid definition of '" << trigName 
		     << "' condition." << endl;
		finish();
		return;
	    }
        } else if (isEnvArg(argi.c_str())) {
	    i++;
	} else {
	    cerr << "Unrecognized argument: " << argi << endl;
	    for (const char** p=synText ; *p ; p++) cerr << *p << endl;
	    cerr << endl;
	    finish();
	    return;
	}
    }
    if (mFrOut <= 0) mFrOut = 1;

    //--------------------------------  Read the configuration file.
    ifstream cstream;
    if (!cfile) {
        cerr << "No configuration file specified. Copy all channels." << endl;
	mMode = m_CopyAll;
    } else {
        cstream.open(cfile, ios::in);
        if (cstream.bad()) {
            cerr << "Error opening configuration file " << cfile << endl;
	    finish();
	    return;
	} else {
            readConfig(cstream);
	    mMode = m_Select;
	    cstream.close();
	}
    }

    //--------------------------------  Set up data accessor
    getDacc().setTOCMode(false);

    //--------------------------------  Set up based on parameters
    mOutFile = file;
    if (ra) setBuffer(-1);
    if (!mOSC) {
        mBackFrame = 0;
	mTrigFrame = 1;
    } else if (mBackFrame > 16) {
        cout << "Backlog reset to maximum (16)." << endl;
        mBackFrame = 16;
    }
    return;
}

//======================================  Read the configuration file.
bool
FrWrite::readConfig(istream& fd) {
    ParseLine pl(fd);
    if (!pl.isOpen()) return true;

    //--------------------------------  Read in the configuration file
    while (pl.getLine() >= 0) {
        if (pl.getCount() > 0) mChannel.push_back(ChanDesc(pl[0]));
    }
    cerr << "Read " << mChannel.size() << " channel names from config" << endl;
    return false;
}

//======================================  Open an output writer.
bool
FrWrite::openWriter(const Time& gps) {
    char line[128], seqno[16];
    string format = mOutFile;

    if (Debug()) cout << "FrWrite: in openWriter." << endl;

    //--------------------------------  Generate an output file name
    unsigned int index = format.find("%q");
    if (index < format.size()) {
        int iseq = 0;
	if (mSequence) iseq = mFileOut/mSequence;
	sprintf(seqno, "%d", iseq);
	format.replace(index, 2, seqno);
    }
    TimeStr(gps, line, format.c_str());

    //--------------------------------  Make a directory
    if (Debug()) cerr << "FrWrite: Opening output file: " << line << endl;
    index = string(line).rfind("/");
    if (index < format.size()) {
        string dir(line, index);
	if (access(dir.c_str(), W_OK)) {
	    if (access(dir.c_str(), F_OK)) {
	        if (mkdir(dir.c_str(), 0777)) {
		    cerr << "Can't make directory: " << dir << endl;
		    return true;
 		}
	    } else {
	        cerr << "Directory: " << dir << " is not writeable." << endl;
		return true;
	    }
	}
    }

    //--------------------------------  Open the output file
    mCurFile = line;
#if LDAS_VERSION_NUMBER < 109360
    mStream = new ofstream(line, ios::out);
    if (mStream->good()) {
        cout << "FrWrite: Opened " << mCurFile << " for frame output, TOC=" 
	     << mTOC << endl;
    } else {
        cerr << "FrWrite: Can't open " << mCurFile << " for output." << endl;
	return true;
    }
#else
    auto_ptr< FrameBuffer<filebuf> > fb(new FrameBuffer<filebuf>(ios::out));
    if (fb->open(line, ios::out)) {
        cout << "FrWrite: Opened " << mCurFile << " for frame output, TOC=" 
	     << mTOC << endl;
    } else {
        cerr << "FrWrite: Can't open " << mCurFile << " for output." << endl;
	return true;
    }
    mFrameBuf = fb.release();
#endif

    //--------------------------------  Create a frame writer.
    try {
#if LDAS_VERSION_NUMBER < 109360
        mWriter = new FrameCPP::OFrameStream(*mStream);
#else
        mWriter = new FrameCPP::OFrameStream(mFrameBuf);
#endif
    } catch (std::exception& e) {
        mWriter = 0;
	cerr << "FrWrite: Unable to open writer: " << e.what() << endl;
    } catch (...) {
        mWriter = 0;
	cerr << "FrWrite: Unable to open writer: Unknown exception." << endl;
    }
    if (!mWriter) return true;

    //--------------------------------  Set counter(s), return
    mFrDone = 0;
    return false;
}

//======================================  Process one frame (FrameCPP).
void
FrWrite::ProcessFrame(frame_ptr_type frame)
{
    if (Debug()) cout << "FrWrite: in ProcessFrame." << endl;

    //----------------------------------  Loop over subframes
    Interval lenF = getDacc().getDt();
    Interval dT = mResize;
    if (!dT) dT = lenF;
    for (Interval Off=0.0; Off<lenF; Off += dT) {

        //------------------------------  See if this frame passes the test
        if (mOSC) {
	    getDacc().zeroChans(dT);
	    getDacc().fillChans(Interval(0.0), dT);
	    if (mOSC->satisfied(trigName)) mBackWrite = mBackValid+mTrigFrame;
	} else {
	    mBackWrite = 1;
	}

	//cout << "mBackWrite = " << mBackWrite << endl;
	//cout << "mBackValid = " << mBackValid << endl;
	//cout << "mTrigFrame = " << mTrigFrame << endl;
	//cout << "mBackFrame = " << mBackFrame << endl;
    
	//------------------------------  Open a writer if necessary
	if (mBackWrite) {
	    if (!mWriter && mBackWrite) {
	        GPSTime T = (mBackValid) ? mBacklog[0]->GetGTime() 
		            : frame->GetGTime();
		Time t0(T.getSec(), T.getNSec());
		if (!mBackValid) t0 += Off;
		if (openWriter(Time(t0))) {
		    finish();
		    return;
		}
	    }

	    //--------------------------  Write the frame
	    if (mBackValid) {
	        WriteFrame(mBacklog[0]);
#ifndef FCPP_SHARED_PTRS
		delete mBacklog[0];
#endif
		for (int i=1; i<mBackValid; i++) mBacklog[i-1] = mBacklog[i];
		mBacklog[mBackValid-1] = MakeFrame(frame, Off, dT);
	    } else if (mMode == m_CopyAll) {
	        WriteFrame(frame);
	    } else {
	        frame_ptr_type newframe = MakeFrame(frame, Off, dT);
		WriteFrame(newframe);
#ifndef FCPP_SHARED_PTRS
		delete newframe;
#endif
	    }
	    mBackWrite--;
	    mFrDone++;
	    mFrTotal++;

	    //--------------------------  See if this is the end
	    if (mFrDone >= mFrOut) {
	        closeWriter();
		if (mRepeat != 0 && mFileOut >= mRepeat) finish();
	    }

	    //--------------------------  Record the frame if necessary
	} else if (mBackFrame) {
	    if (mBackValid == mBackFrame) {
#ifndef FCPP_SHARED_PTRS
	        delete mBacklog[0];
#endif
		for (int i=1 ; i<mBackValid ; i++) mBacklog[i-1] = mBacklog[i];
		mBacklog[mBackValid-1] = MakeFrame(frame, Off, dT);
	    } else {
	        mBackValid++;
		
		while ((int)mBacklog.size() < mBackValid) {
		    mBacklog.push_back(MakeFrame(frame, Off, dT));
		}
	    }
	}
    }
}

//======================================  Write out the frame
//                                        Note that this kludge is needed to 
//                                        write a const FrameH with FrameCPP
//                                        version 6
void
FrWrite::WriteFrame(frame_ptr_type frame) {
    if (Debug()) cerr << "Frwrite: Start writing frame." << endl;
    try {
#ifndef FCPP_SHARED_PTRS
        mWriter->WriteFrame(*frame);
#else
        mWriter->WriteFrame(frame);
#endif
    } catch (std::exception& c) {
        cerr << "FrWrite: Caught exception in WriteFrame: " 
	     << c.what() << endl;
    } catch (...) {
        cerr << "FrWrite: Caught unidentified exception in WriteFrame." 
	     << endl;
    }
    if (Debug()) cerr << "Frwrite: Finished writing frame." << endl;    
}

//======================================  Copy selected parts of a frame;
FrWrite::frame_ptr_type
FrWrite::MakeFrame(frame_ptr_type frame, Interval Off, Interval dT)
{
#ifdef FCPP_SHARED_PTRS
    frame_ptr_type newFrame;
#else
    frame_ptr_type newFrame(0);
#endif

    if (Debug()) cout << "FrWrite: starting MakeFrame." << endl;
    Time tStart = GetGPSTime(frame->GetGTime()) + Off;
    GPSTime gStart(tStart.getS(), tStart.getN());

    //--------------------------------  Check for raw data in frame
    frrawdata_pointer raw(frame->GetRawData());
    if (!raw) {
        cerr << "FrWrite:ProcessFrame: No raw data in frame." << endl;
	return newFrame;
    }

    //--------------------------------  Process according to mode.
    switch (mMode) {

    //--------------------------------  Write frame unmodified.
    case m_CopyAll:
        newFrame = frame_ptr_type(new FrameH(*frame));
	break;

    //--------------------------------  Select specific channels
    case m_Select:

        //----------------------------  Build a frame with identical params
        newFrame = frame_ptr_type(new FrameH(frame->GetName(), frame->GetRun(), 
					     frame->GetFrame(), gStart,
					     frame->GetULeapS(), dT));
	newFrame->RefDetectSim()  = frame->RefDetectSim();
	newFrame->RefDetectProc() = frame->RefDetectProc();

        //----------------------------  Add a like-named raw data block
	newFrame->RefHistory()  = frame->RefHistory();
	newFrame->RefAuxData()  = frame->RefAuxData();
	newFrame->RefProcData() = frame->RefProcData();
	newFrame->RefSimData()  = frame->RefSimData();
	newFrame->RefSimEvent() = frame->RefSimEvent();
#ifndef FCPP_SHARED_PTRS
	newFrame->SetRawData(FrRawData(raw->GetName()));
#else
	newFrame->SetRawData(frrawdata_pointer(new FrRawData(raw->GetName())));
#endif

        //----------------------------  Loop over channel list, copy channels.
        FrRawData::const_firstAdc_iterator 
	    AdcIter = raw->RefFirstAdc().begin();
	for (chan_iter itr=mChannel.begin(); itr != mChannel.end(); itr++) {
	    const char* ChName = itr->getName();
	    if (Debug()> 3) cout << "Get channel: " << ChName << endl;
	    chan_iter tent = itr;

	    //------------------------  Get the adc pointer.
	    AdcIter = raw->RefFirstAdc().find(ChName, AdcIter);
	    if (AdcIter == raw->RefFirstAdc().end()) {
	        AdcIter = raw->RefFirstAdc().find(ChName, raw->RefFirstAdc().begin());
		if (AdcIter == raw->RefFirstAdc().end()) {
		    if (Debug() || !itr->getError()) {
		        cerr << "Channel " << ChName << " not found in frame "
			     << getNFrames() << endl;
		    }
		    itr->setError(true);
		    // AdcIter = raw->refAdc().begin();
		    continue;
		} else if (itr != mChannel.begin()) {
		    chan_iter save = tent;
		    --itr;
		    tent = mChannel.insert(itr, *save);
		    mChannel.erase(save);
		    if (Debug() > 2) {
		        cout << "Reorder channels " << itr->getName() 
			     << " <-> " << tent->getName() << endl;
		    }
		}
		itr->setError(false);
	    }
	    tent->Copy(*AdcIter, newFrame, Off, dT);
	}
    }
    if (Debug()) cout << "FrWrite: exit MakeFrame." << endl;
    return newFrame;
}

//======================================  Print final Results
void
FrWrite::closeWriter(void)
{
    if (Debug()) cout << "FrWrite: in closeWriter." << endl;
    if (mWriter) {
        if (Debug()) {
	    cout << "Closing file: " << mCurFile;
	    if (mTOC) cout << " and writing TOC.";
	    cout << endl;
	}
#if LDAS_VERSION_NUMBER < 109360
        mWriter->close();
        delete mWriter;
#else
        mWriter->Close();
        delete mWriter;
	mFrameBuf = 0;
#endif
	mFileOut++;
	mWriter = 0;
    }

#if LDAS_VERSION_NUMBER < 109360
    if (mStream) {
        delete mStream;
	mStream = 0;
    }
#else
    if (mFrameBuf) {
        delete mFrameBuf;
	mFrameBuf = 0;
    }
#endif
}

//======================================  Print final Results
FrWrite::~FrWrite()
{
    closeWriter();
    if (mFileOut) {
        cout << "FrWriter terminating after writing " << mFileOut 
	     << " files containing " << mFrTotal << " total frames." 
	     << endl;
    }
}

//======================================  Channel descriptor constructors.
ChanDesc::ChanDesc(void) 
  : mRate(0), mError(false)
{
}

ChanDesc::ChanDesc(const ChanDesc& x) {
    mName  = x.mName;
    mRate  = x.mRate;
    mError = x.mError;
}

ChanDesc::ChanDesc(const char* configstr) 
  : mRate(0.0), mError(false)
{
    int lName = string(configstr).find(' ');
    if (lName <= 0) lName = strlen(configstr);
    mName = string(configstr, lName);
}

//======================================  Overload ChanDesc operators.
ChanDesc&
ChanDesc::operator=(const ChanDesc& x) {
    mName  = x.mName;
    mRate  = x.mRate;
    mError = x.mError;
    return *this;
}

//======================================  Fradc substring

//======================================  Copy Adc data into a frame
void
ChanDesc::Copy(fradcdata_pointer in, frameh_pointer out, 
	       Interval Off, Interval dT) {

    //----------------------------------  Get the raw data block
    frrawdata_pointer raw(out->GetRawData());
    if (!raw) {
#ifndef FCPP_SHARED_PTRS
        out->SetRawData(FrRawData("Raw Data"));
#else
        out->SetRawData(frrawdata_pointer(new FrRawData("Raw Data")));
#endif
	raw = out->GetRawData();
    }

    //----------------------------------  Copy the adc data for this channel
    FrRawData::firstAdc_type* cadc = &raw->RefFirstAdc();
    if (!dT) {
        FrRawData::firstAdc_iterator  iadc = cadc->append(in);
    } else {
#if LDAS_VERSION_NUMBER < 109000
        frvect_pointer vect(in->RefData().front());
        Dimension dim = vect->GetDim(0);
	size_t inx = size_t((double(Off) - dim.GetStartX())/dim.GetDx() + 0.5);
	size_t len = size_t(double(dT)/dim.GetDx() + 0.5);
	if (len+inx > dim.GetNx()) len = dim.GetNx() - inx;
	dim.SetNx(len);
	short  type  = vect->GetType();
	size_t tSize = FrVect::GetTypeSize(type);
	frvect_pointer newV(new FrVect(vect->GetName(),
				       vect->GetType(),
				       1,
				       &dim,
				       FrameCPP::BYTE_ORDER_HOST,
				       vect->GetData(true)+inx*tSize,
				       vect->GetUnitY()));
	auto_ptr<FrAdcData> newA(new FrAdcData( in->GetName(), 
						in->GetChannelGroup(),
						in->GetChannelNumber(), 
						in->GetNBits(),
						in->GetSampleRate(),
						in->GetBias(),
						in->GetSlope(),
						in->GetUnits(),
						in->GetFShift(),
						in->GetTimeOffset(),
						in->GetDataValid(),
						in->GetPhase())
				 );
	newA->RefData().append(newV, false, true);
        cadc->append(newA.release(), false, true);
#elif !defined(FCPP_SHARED_PTRS)
	auto_ptr<FrAdcData> newA = in->SubFrAdcData(Off, dT);
        cadc->append(newA.release(), false, true);
#else
	fradcdata_pointer newA(in->Subset(Off, dT));
        cadc->append(newA);
#endif
    }
}

//======================================  Copy adc from one frame to another
void
ChanDesc::Copy(frameh_pointer in, frameh_pointer out, 
	       Interval Off, Interval dT) {
    frrawdata_pointer raw(in->GetRawData());
    if (!raw) {
        cout << "No raw data in input frame" << endl;
    } else {
        FrRawData::const_firstAdc_iterator AdcIter;
	AdcIter = (raw->RefFirstAdc()).find(getName(), (raw->RefFirstAdc()).begin());
	if (AdcIter != (raw->RefFirstAdc()).end()) {
	    Copy(*AdcIter, out, Off, dT);
	    setError(false);
	} else if (!getError()) {
	    cout << "Channel: " << getName() << " is not in the input frame."
		 << endl;
	    setError(true);
	}
    }
}

//======================================  Set Chandesc error bit.
void
ChanDesc::setError(bool t) {
    mError = t;
}
