#ifndef TRIGRSLT_HH
#define TRIGRSLT_HH

#include "Interval.hh"
#include "TrigBase.hh"

namespace trig {
    class TrigBase;
    class TrigRaw;

/**  The trigger result object contains a trigger dependent list of data
  *  that describes the trigger. In the present implementation, the data 
  *  list is a vector of double precision floats. Other trigger-dependent 
  *  result classes may be defined based on the TrigBase class.
  *  \brief Trigger result class.
  *  \author J. Zweizig
  *  \version $Id: TrigRslt.hh 6802 2013-04-22 06:21:02Z john.zweizig@LIGO.ORG $
  */ 
class TrigRslt : public TrigBase {
  public:
    /** Result data type.
      */
    typedef double param_type;

    /**  A result object is created and the parameter data are copied into 
      *  the trigger data list.
      *  \brief Create a result object.
      *  \param ID    Trigger type identifier string.
      *  \param SubID Trigger sub-type string.
      *  \param nData Number of data words.
      *  \param DList Array of data words.
      */
    TrigRslt(const char* ID, const char* SubID=0, 
	     index_type nData=0, const param_type* DList=0);

    /**  A result object is created and the parameter data are copied into 
      *  the trigger data list.
      *  \brief Copy constructor.
      *  \param x Reference to trigger to be copied.
      */
    TrigRslt(const TrigRslt& x);

    /**  Release the data after recording the trigger.
      *  @memo Result destructor.
      */
    virtual ~TrigRslt(void);

    /**  The number of defined data words is returned.
      *  \brief Number of data words.
      *  \return Number of data words.
      */
    index_type getNData(void) const;

    /**  Get a specified result data word. 
      *  @memo Get a data word.
      *  \param i Number of parameter to be read.
      *  \return Constant reference to selected parameter.
      */
    const param_type& operator[](unsigned int i) const;

    /**  Get or set a specified data word.
      *  \brief Read/Modify a data word.
      *  \param i Number of parameter to be read/modified.
      *  \return Reference to selected parameter.
      */
    param_type& operator[](unsigned int i);

    /**  Copy contents of a specified trigger to this instance.
      *  \brief Assignment operator.
      *  \param x Reference to trigger to be copied.
      *  \result Reference to this instance.
      */
    TrigRslt& operator=(const TrigRslt& x);

  private:
};   // trig::TrigRslt

}    //  namespace trig

//--------------------------------------  Get the number of data words.
inline trig::TrigBase::index_type 
trig::TrigRslt::getNData(void) const {
    return getResultLength()/sizeof(param_type);
}

//--------------------------------------  Get a data word.
inline const trig::TrigRslt::param_type& 
trig::TrigRslt::operator[](unsigned int i) const {
    return ((const param_type*)getResult())[i];
}

//--------------------------------------  Read/Modify a data word
inline trig::TrigRslt::param_type& 
trig::TrigRslt::operator[](unsigned int i) {
    return ((param_type*)getResult())[i];
}

//--------------------------------------  Assignment operator
inline trig::TrigRslt& 
trig::TrigRslt::operator=(const TrigRslt& x) {
    return dynamic_cast<TrigRslt&>(TrigBase::operator=(x));
}

#endif   //  TRIGRSLT_HH
