/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//    TrigProc implementation functions.
//
#include "PConfig.h"
#if defined(sun) && !defined(__EXTENSIONS__)
#define __EXTENSIONS__
#endif
#include <cstdlib>
#include <cstdio>
#include <time.h>
#include <unistd.h>
#include <cstring>
#include "TrigProc.hh"
#include "Time.hh"
#include <iostream>
#ifdef P__DARWIN
// Pick up the definition of _NSGetExecutablePath() for OS X
#include <mach-o/dyld.h>
#endif

using namespace std;
using namespace trig;


//======================================  Create a trigger process identifier.
TrigProc::TrigProc(const char* cvsid, const char* title) 
  : mCreatorDB(0)
{
    //----------------------------------  Parse out the cvs info
    char Source[128], Version[8], Author[64];
    Source[0]  = 0;
    Version[0] = 0;
    Author[0]  = 0;
    if (cvsid) mCvsID = cvsid;

    // cout << "cvs ID string: " << mCvsID << endl;

    if (mCvsID.substr(0,9) == "$Header: ") {
	int yy, mo, dd, hh, mm, ss;
	struct tm strtime;
	time_t stime;

        sscanf(cvsid, "$Header: %s %s %4d-%2d-%2d %2d:%2d:%2dZ %s ", 
	       Source, Version, &yy, &mo, &dd, &hh, &mm, &ss, Author);
	strtime.tm_sec   = ss;
	strtime.tm_min   = mm;
	strtime.tm_hour  = hh;
	strtime.tm_mday  = dd;
	strtime.tm_mon   = mo-1;
	strtime.tm_year  = yy-1900;
	strtime.tm_wday  = 0;
	strtime.tm_yday  = 0;
	strtime.tm_isdst = 0;
#ifndef P__SOLARIS
	stime = timegm(&strtime); // get the gm time where available
#else
	stime = mktime(&strtime); // otherwise incorrect local time.
#endif
	mTime = fromUTC(stime).getS();
    }
    else if (mCvsID.substr(0,5) == "$Id: ") {
	int yy, mo, dd, hh, mm, ss;
	struct tm strtime;
	time_t stime;

        sscanf(cvsid, "$Id: %s %s %4d-%2d-%2d %2d:%2d:%2dZ %s ", 
	       Source, Version, &yy, &mo, &dd, &hh, &mm, &ss, Author);
	strtime.tm_sec   = ss;
	strtime.tm_min   = mm;
	strtime.tm_hour  = hh;
	strtime.tm_mday  = dd;
	strtime.tm_mon   = mo-1;
	strtime.tm_year  = yy-1900;
	strtime.tm_wday  = 0;
	strtime.tm_yday  = 0;
	strtime.tm_isdst = 1;
#ifndef P__SOLARIS
	stime = timegm(&strtime); // get the gm time where available
#else
	stime = mktime(&strtime); // otherwise incorrect local time.
#endif
	mTime = fromUTC(stime).getS(); //convert to GPS
    }
    else {
	mTime = 0;
    }
    mSource  = Source;
    mVersion = Version;
    mAuthor  = Author;

    //cout << "Monitor source: " << mSource << " version: " << mVersion
    //	   << " Author: " << mAuthor << " GPS: " << mTime << endl;

    //----------------------------------  Get the descriptive title
    if (title) mComment = title;

    //----------------------------------  Program name is the executable
#define MAXXNAME 512
#if defined (P__WIN32) || defined (P__LINUX)
    char Xname[MAXXNAME], Path[MAXXNAME];
#if defined (P__WIN32) 
    sprintf(Path, "/proc/%i/exename", getpid());
#else
    sprintf(Path, "/proc/%i/exe", getpid());
#endif
    int rc = readlink(Path, Xname, MAXXNAME);
    if (rc > 0) Xname[rc] = 0;
    else        std::cout << "Process name not available!" << std::endl;
    mProgram = Xname;
#elif defined (P__DARWIN)
    char Xname[MAXXNAME], Path[MAXXNAME];
    uint32_t pathsize = sizeof(Path);
    int rc = _NSGetExecutablePath(Path, &pathsize);
    if (rc == 0 )
    {
       int rc = readlink(Path, Xname, MAXXNAME);
       if (rc > 0) 
	 Xname[rc] = 0;
       else        
	 std::cout << "Process name not available!" << std::endl;
    }
    else        
    {
       std::cout << "Process name not available!" << std::endl;
    }
    mProgram = Xname;
#else
    const char* Xname = getexecname();
    if (Xname) mProgram = Xname;
    else       mProgram.clear();
#endif
    std::string::size_type inx = mProgram.find_last_of("/");
    if (inx != std::string::npos) mProgram.erase(0, inx+1);

    //----------------------------------  Set the rest of the parameters.
    mStartTime = Now().getS();
    mEndTime   = 0;
    mIsOnline  = true;
    mProcID    = getpid();
    char Hname[128];
    gethostname(Hname, sizeof(Hname));
    mNode = Hname;
    const char* IFOs = getenv("DMTIFOS");
    if (IFOs)  mIfos = IFOs;
}

//======================================  Construct from everything
TrigProc::TrigProc(const char* Source, const char* Version, 
			 gps_t Start, const char* Host, pid_t PID, 
			 bool Online, const char* IFOs, const char* Author, 
			 gps_t Mod, const char* Title, const char* Exec, 
			 gps_t End) 
  : mCreatorDB(0), mVersion(Version), mSource(Source), mTime(Mod),
    mIsOnline(Online), mProcID(PID), mStartTime(Start), mEndTime(End)
{

    //----------------------------------  Get the descriptive title
    if (Title)  mComment = Title;
    if (Author) mAuthor  = Author;
    if (Exec)   mProgram = Exec;
    if (Host)   mNode    = Host;
    if (IFOs)   mIfos    = IFOs;
}

//======================================  Destroy a trigger process.
TrigProc::~TrigProc() {
}

//======================================  Display the trigger process info
std::ostream&
TrigProc::display(std::ostream& out) const {
    out << "Program name:     " << mProgram   << endl;
    out << "Source version:   " << mVersion   << endl;
    out << "Source file name: " << mSource    << endl;
    out << "Last update time: " << mTime      << endl;
    out << "Author name:      " << mAuthor    << endl;
    out << "Comment string:   " << mComment   << endl;
    out << "Online flag:      " << mIsOnline  << endl;
    out << "Node name:        " << mNode      << endl;
    out << "Unix Process ID:  " << mProcID    << endl;
    out << "GPS start time:   " << mStartTime << endl;
    out << "GPS end time:     " << mEndTime   << endl;
    out << "cvs/svn string:   " << mCvsID     << endl;
    out << "Process DB key:   " << mProcessID << endl;
    out << "Param set key:    " << mParamSet  << endl;
    out << "IFO list:         " << mIfos      << endl;
    return out;
}

//======================================  Get the author's name.
const char*
TrigProc::getAuthor() const {
    return mAuthor.c_str();
}

//======================================  Get the end time.
TrigProc::gps_t 
TrigProc::getEndTime(void) const {
    return mEndTime;
}

//======================================  Get the interferometer list.
const char* 
TrigProc::getIFOs(void) const {
    return mIfos.c_str();
}

//======================================  Get the source version time.
TrigProc::gps_t 
TrigProc::getModTime(void) const {
    return mTime;
}

//======================================  Get the process name.
const char*
TrigProc::getName() const {
    return mProgram.c_str();
}

//======================================  Get the node name
const char* 
TrigProc::getNode(void) const {
    return mNode.c_str();
}

//======================================  Get the parameter list
const char* 
TrigProc::getParamSet(void) const {
    return mParamSet.c_str();
}

//======================================  Get the database id
const char* 
TrigProc::getProcessID(void) const {
    return mProcessID.c_str();
}

//======================================  Get the process ID.
pid_t
TrigProc::getProcID() const {
    return mProcID;
}

//======================================  Get the source code location
const char* 
TrigProc::getSource(void) const {
    return mSource.c_str();
}

//======================================  Get the Start time
TrigProc::gps_t 
TrigProc::getStartTime(void) const {
    return mStartTime;
}

//======================================  Get the process title.
const char*
TrigProc::getTitle() const {
    return mComment.c_str();
}

//======================================  Get the source version
const char* 
TrigProc::getVersion(void) const {
    return mVersion.c_str();
}

//======================================  Test if process is online.
bool 
TrigProc::isOnline(void) const {
    return mIsOnline;
}

//======================================  Overloaded operators.
bool 
TrigProc::operator==(const TrigProc& x) const {
    return  (mProcID == x.mProcID && mStartTime == x.mStartTime &&
	     mNode   == x.mNode   && mProgram   == x.mProgram);
}

//=====================================  Set the process comment string
void 
TrigProc::setComment(const std::string& comment) {
    mComment = comment;
}

//=====================================  Set the cvs ID
void
TrigProc::setCvsID(const std::string& cvs) {
    mCvsID = cvs;
}

//=====================================  Set the process end time
void 
TrigProc::setEndTime(gps_t tEnd) {
    mEndTime = tEnd;
}

//=====================================  Set the process IFO list
void
TrigProc::setIFOs(const std::string& ifos) {
    mIfos = ifos;
}

//=====================================  Set the process database ID
void
TrigProc::setProcessID(const std::string& ID) {
    mProcessID = ID;
}
