/* -*-  mode: c++; c-basic-offset: 4; -*- */
#include "LockSegment.hh"
#include <iostream>
#include <stdexcept>

using namespace std;

//======================================  Segment constructor
LockSegment::LockSegment(id_type id, const Time& t0, const Interval dT, 
			 flag_type flg) 
  : mID(id), mStart(t0), mDuration(dT), mFlags(flg)
{
}

LockSegment::LockSegment(id_type id, const Time& t0, const Time& tEnd, 
			 flag_type flg) 
  : mID(id), mStart(t0), mDuration(tEnd-t0), mFlags(flg)
{
}

//======================================  Set segment range
void
LockSegment::setRange(const Time& t0, const Interval dT) {
    mStart    = t0;
    mDuration = dT;
}

//======================================  And the segments
LockSegment& 
LockSegment::operator&= (const LockSegment& s) {
    Time tEnd  = getEndTime();
    Time tEnd1 = s.getEndTime();
    if (mStart < s.mStart) mStart = s.mStart;
    if (tEnd1 < tEnd) tEnd = tEnd1;
    if (tEnd <= mStart) throw runtime_error("Segments don't overlap");
    mDuration = tEnd - mStart;
    mFlags &= s.mFlags;
    return *this;
}

//======================================  Or the segments
LockSegment& 
LockSegment::operator|= (const LockSegment& s) {
    Time tEnd  = getEndTime();
    Time tEnd1 = s.getEndTime();
    if (tEnd1<mStart || tEnd<s.mStart) 
        throw runtime_error("Segments can't be joined");
    if (mStart > s.mStart) mStart = s.mStart;
    if (tEnd1 > tEnd) tEnd = tEnd1;
    mDuration = tEnd - mStart;
    mFlags |= s.mFlags;
    return *this;
}

//======================================  Pad the front and back 
void 
LockSegment::pad(Interval pFront, Interval pBack) {
    Interval left = (Time(SEGMENT_TINF) - mStart) - mDuration;
    if (pBack > left) pBack = left;
    if (pFront > Interval(0) && mStart < Time(0) + pFront) {
	pFront  = mStart - Time(0);
	mStart  = Time(0);
    } else {
	mStart -= pFront;
    }
    mDuration  += pFront + pBack;
    if (mDuration < Interval(0)) mDuration = 0;
}

//======================================  write formatted data to ostream.
ostream&
LockSegment::write(ostream& o, const std::string& f) const {
    string::size_type N=f.size();
    string::size_type inx=0;
    for (string::size_type i=0; i<N; ++i) {
	if (f[i] == '%') {
	    if (inx < i) o << f.substr(inx, i-inx);
	    switch (f[++i]) {
	    case 'd':
		{
		    double dt = getDuration();
		    long   di = long(dt);
		    if (dt == double(di)) o << di;
		    else                  o << dt;
		}
		break;
	    case 'e':
		o << getEndTime().getS();
		break;
	    case 'i':
		o << getID();
		break;
	    case 's':
		o << getStartTime().getS();
		break;
	    case '%':
		o << "%";
		break;
	    default:
		throw runtime_error("Invalid output format character");
	    }
	    inx = i+1;
	}
    }
    if (inx < N) o << f.substr(inx, N-inx);
    return o;
}

//======================================  Print to ostream.
ostream&
operator<<(ostream& o, const LockSegment& s) {
    return s.write(o, "%s:%d");
}
