#ifndef _LIGO_EVENTVETO_H
#define _LIGO_EVENTVETO_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: Veto							*/
/*                                                         		*/
/* Module Description: Event veto					*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: Veto.html						*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#include "events/TimeWindow.hh"
#include "events/Condition.hh"
#include "events/ConditionPtr.hh"


namespace events {


/** Describes an event veto. An event veto consists of a condition and
    a time window around the event time. The veto conditions yields
    true if an event which fullfills the condition falls within 
    the veto window around the analysed event. This means a condition
    which wants to rejects events which are close to a veto event has
    to be specified something like
    \begin{verbatim}
    Filter ("burst::*") && !Veto (Filter ("glitch::psl"), 0.1, -0.05)
    \end{verbatim}

    The time window is defined relative to the time of the first
    current event. Current events are ignored in the veto condition
    and none of them is checked against the veto condition. The 
    veto condition is called with same event list as the veto class,
    but with a single current event which is set to the event test for.

    A veto condition can also be used to specify a coincidence 
    condition! The following example demonstrates this:
    \begin{verbatim}
    Filter ("burst::*") && Veto (Filter ("burst::*"), 0.1)
    \end{verbatim}
    looks for two burst events which fall within 100ms. To avoid 
    the confusion this example may create, the alias Coincide is
    defined which makes the example more readable, but otherwise
    behaves like the veto condition:
    \begin{verbatim}
    Filter ("burst::*") && Coincide (Filter ("burst::*"), 0.1)
    \end{verbatim}
    yields the same result as the previous example.

    Basically, the terms veto and conicide do not describe different
    outcomes of a condition, but rather how the occurance of a 
    close-by second event is treated in the analysis.
   
    @memo Defines an veto condition
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class Veto : public Condition {
   public:
      /** Creates an default event veto (+/-0.5sec)
          @memo Default constructor
       ******************************************************************/
      Veto() : mWindow (1.) {
      }
      /** Creates an event veto. Any event is considered.
          @memo Constructor
   	  @param cond Condition
   	  @param duration Duration of veto window
   	  @param offset Time offset for veto start (relative to event)
       ******************************************************************/
      Veto (const Interval& duration, const Interval& offset)
      : mWindow (duration, offset) {
      }
      /** Creates an event veto. Any event is considered.
          The veto window is symmetric around the analyzed event.
          @memo Constructor
   	  @param cond Condition
   	  @param duration Duration of veto window
       ******************************************************************/
      explicit Veto (const TimeWindow& window) 
      : mWindow (window) {
      }
      /** Creates an event veto. Takes a condition as an argument.
          Typically an event filter is used.
          @memo Constructor
   	  @param cond Condition
   	  @param offset Time offset for veto start (relative to event)
   	  @param duration Duration of veto window
       ******************************************************************/
      Veto (const Condition& cond, const Interval& duration, 
           const Interval& offset)
      : mCond (cond), mWindow (duration, offset) {
      }
      Veto (const ConditionPtr& cond, const Interval& duration, 
           const Interval& offset)
      : mCond (cond), mWindow (duration, offset) {
      }
      /** Creates an event veto. Takes a condition as an argument.
          The veto window is symmetric around the analyzed event.
          @memo Constructor
   	  @param cond Condition
   	  @param duration Duration of veto window
       ******************************************************************/
      Veto (const Condition& cond, const TimeWindow& window) 
      : mCond (cond), mWindow (window) {
      }
      Veto (const ConditionPtr& cond, const TimeWindow& window) 
      : mCond (cond), mWindow (window) {
      }
      /** Returns a copy of the event condition. This method must be 
          overriden by all descendents.
          @memo Copy the event
          @return event copy
       ******************************************************************/
      virtual Veto* Copy() const {
         return new Veto (*this); }
   
      /** Evaluate the veto. This function must be overriden
          by descendents.
          @memo Evaluate
          @param arg Event argument list
          @param val Condition value (return)
          @return True if condition could be evaluated
       ******************************************************************/
      virtual bool Evaluate (const Argument& arg, bool& val) const;
   
      /** Get the event condition.
          @memo Get event condition
       ******************************************************************/
      const ConditionPtr& GetCondition () const {
         return mCond; }
      /** Set the time window.
          @memo Set time window
       ******************************************************************/
      void SetWindow (const TimeWindow& window) {
         mWindow = window; }
      /** Set the time window.
          @memo Set time window
          @param width Window duration
          @param offset Window offset
       ******************************************************************/
      void SetWindow (const Interval& width, const Interval& offset) {
         mWindow = TimeWindow (width, offset); }
      /** Set the time offset.
          @memo Set time offset
       ******************************************************************/
      void SetOffset (const Interval& offset) {
         mWindow.SetOffset (offset); }
      /** Get the time offset.
          @memo Get time offset
       ******************************************************************/
      Interval GetOffset () const {
         return mWindow.GetOffset(); }
      /** Set the time duration.
          @memo Set time duration
       ******************************************************************/
      void SetWidth (const Interval& duration) {
         mWindow.SetOffset (duration); }
      /** Get the time duration.
          @memo Get time duration
       ******************************************************************/
      Interval GetWidth () const {
         return mWindow.GetWidth(); }
   
   private:
      /// Event condition 
      ConditionPtr	mCond;
      /// Time window
      TimeWindow	mWindow;
   };

/** Describes an event coincidence. This is an alias to the veto
    condition! See the explaination there. 

    There is a big difference between the Coincide condition and the 
    Coincidence algorithm. Coincide merly makes sure that the 
    current event coincides with another one and it keeps no
    record about the other event. On the other hand the Coincidence
    algorithm evaluates conditions with two or more current
    events and, if a coincidence was found, builds a new result event 
    which contains the participants as column values.
   
    @memo Defines an coincidence condition
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   typedef Veto Coincide;

}

#endif // _LIGO_EVENTVETO_H
