/* -*- mode: c++; c-basic-offset: 3; -*- */
//////////////////////////////////////////////////////////////////////////
//  									//
//  Iterator								//
//  									//
//////////////////////////////////////////////////////////////////////////

#include "PConfig.h"
#include <cstdio>
#include <cmath>
#include <ctype.h>
#include <stdlib.h>
#include <cstring>
#include "Time.hh"
#include "Interval.hh"
#include "events/Value.hh"
#include "events/Event.hh"
#include "events/IfoSet.hh"

#ifdef __GNU_STDC_OLD
namespace std {
   template <class T>
   inline complex<T> tan (const complex<T>& a) { return sin(a)/cos(a); }
   template <class T>
   inline complex<T> tanh (const complex<T>& a) { return sinh(a)/cosh(a); }

   template <class T>
   complex<T> log10 (const complex<T>& a) {
      const T log10e = ::log10(::exp(T(1)));
      return log10e * log(a);
   }
}
#endif

namespace events {

//______________________________________________________________________________
   const int Value::kMaxValueSize = sizeof (Value::Union); 

//______________________________________________________________________________
   Value::~Value() 
   {
      DestructValue (mType, mData);
   }

//______________________________________________________________________________
   Value& Value::operator= (const Value& val)
   {
      if (this != &val) {
         DestructValue (mType, mData);
         mType = val.mType;
         ConstructValue (mType, mData, val.mData);
      }
      return *this;
   }

//______________________________________________________________________________
   Value::Value (Real x) : mType (kReal) 
   {
      ConstructValue (mType, mData, &x);
   }

//______________________________________________________________________________
   Value::Value (Int i) : mType (kInt)
   {
      ConstructValue (mType, mData, &i);
   }

//______________________________________________________________________________
   Value::Value (const Time& t) : mType (kTime)
   {
      ConstructValue (mType, mData, &t);
   }

//______________________________________________________________________________
   Value::Value (const Complex& c) : mType (kComplex)
   {
      ConstructValue (mType, mData, &c);
   }

//______________________________________________________________________________
   Value::Value (const std::string& s) : mType (kString)
   {
      ConstructValue (mType, mData, &s);
   }

//______________________________________________________________________________
   Value::Value (const char* p, int len) : mType (kString)
   {
      String s;
      if (p) {
         if (len < 0) {
            s = p;
         }
         else {
            s.assign (p, len);
         }
      }
      ConstructValue (mType, mData, &s);
   }

//______________________________________________________________________________
   Value::Value (const Event& e) : mType (kEvent)
   {
      ConstructValue (mType, mData, &e);
   }

//______________________________________________________________________________
   Value::Value (Enum type, const void* p) : mType (kInvalid)
   {
      if ((type > kInvalid) && (type <= kEvent)) {
         mType = type;
         ConstructValue (mType, mData, p);
      }
   }

//______________________________________________________________________________
   bool Value::Write (Real& x) const
   {
      switch (mType) {
         case kReal:
            {
               CopyValue (kReal, &x, mData);
               return true;
            }
         case kInt:
            {
               Int i; Write (i);
               x = i;
               return true;
            }
         case kTime:
            {
               Time t; Write (t);
               x = t.totalS();
               return true;
            }
         case kString:
            {
               String s; Write (s);
               char* p = 0;
               double y = strtod (s.c_str(), &p);
               if (p == s.c_str()) {
                  return false;
               }
               while (isspace (*p)) ++p;
               if (*p) {
                  return false;
               }
               x = y;
               return true;
            }
         default:
            return false;
      }
      return false;
   }

//______________________________________________________________________________
   bool Value::Write (Int& i) const
   {
      switch (mType) {
         case kReal:
            {
               Real x; Write (x);
               i = (Int)x;
               return true;
            }
         case kInt:
            {
               CopyValue (kInt, &i, mData);
               return true;
            }
         case kTime:
            {
               Time t; Write (t);
               i = t.getS();
               return true;
            }
         case kString:
            {
               String s; Write (s);
               char* p = 0;
               int j = strtol (s.c_str(), &p, 10);
               if (p == s.c_str()) {
                  return false;
               }
               while (isspace (*p)) ++p;
               if (*p) {
                  return false;
               }
               i = j;
               return true;
            }
         default:
            return false;
      }
      return false;
   }

//______________________________________________________________________________
   bool Value::Write (Complex& c) const
   {
      switch (mType) {
         case kComplex:
            {
               CopyValue (kComplex, &c, mData);
               return true;
            }
         case kReal:
            {
               Real x; Write (x);
               c = x;
               return true;
            }
         case kInt:
            {
               Int i; Write (i);
               c = i;
               return true;
            }
         case kTime:
            {
               Time t; Write (t);
               c = t.totalS();
               return true;
            }
         case kString:
            {
               String s; Write (s);
               char* p = 0;
               Real x = strtod (s.c_str(), &p);
               if (p == s.c_str()) {
                  return false;
               }
               while (isspace (*p)) ++p;
               Real y = strtod (p, &p);
               if (p == s.c_str()) {
                  return false;
               }
               while (isspace (*p)) ++p;
               if (*p) {
                  return false;
               }
               c = Complex (x, y);
               return true;
            }
         default:
            return false;
      }
      return false;
   }

//______________________________________________________________________________
   bool Value::Write (Time& t) const
   {
      switch (mType) {
         case kReal:
            {
               Real x; Write (x);
               Interval i (x);
               t = Time (i.GetS(), i.GetN());
               return true;
            }
         case kInt:
            {
               Int i; Write (i);
               t = Time (i, 0);
               return true;
            }
         case kTime:
            {
               CopyValue (kTime, &t, mData);
               return true;
            }
         case kString:
            {
               String s; Write (s);
               char* p = 0;
               Real x = strtod (s.c_str(), &p);
               if (p == s.c_str()) {
                  return false;
               }
               while (isspace (*p)) ++p;
               if (*p) {
                  return false;
               }
               Interval i (x);
               t = Time (i.GetS(), i.GetN());
               return true;
            }
         default: 
            return false;
      }
      return false;
   }

//______________________________________________________________________________
   bool Value::Write (std::string& s) const
   {
      char buf[256];
      switch (mType) {
         case kComplex:
            {
               Complex c; Write (c);
               sprintf (buf, "%g %g", c.real(), c.imag());
               s = buf;
               return true;
            }
         case kReal:
            {
               Real x; Write (x);
               sprintf (buf, "%g", x);
               s = buf;
               return true;
            }
         case kInt:
            {
               Int i; Write (i);
               sprintf (buf, "%d", i);
               s = buf;
               return true;
            }
         case kTime:
            {
               Time t; Write (t);
               sprintf (buf, "%g", t.totalS());
               s = buf;
               return true;
            }
         case kString:
            {
               CopyValue (kString, &s, mData);
               return true;
            }
         case kEvent:
            {
               String tmp;
               Layout layout = ((Event*)mData)->GetLayout();
               if (!layout.IsRegistered()) {
                  return false;
               }
               for (ColumnInfoList::const_iterator i = 
                   layout.GetColumnList().begin(); 
                   i != layout.GetColumnList().end(); ++i) {
                  Value col;
                  String val;
                  if (!((Event*)mData)->GetValue (i->GetName(), col) ||
                     !col.Write (val)) {
                     continue;
                  }
                  if (col.Type() == kEvent) {
                     tmp += i->GetName();
                     if (val.empty()) {
                        tmp += " -\n";
                     }
                     else {
                        for (int pos = val.size() - 1; pos > 0; ) {
                           --pos;
                           if (val[pos] == '\n') {
                              val.insert (pos + 1, "   ");
                           }
                        }
                        tmp += "\n" + val;
                     }
                  }
                  else {
                     tmp += i->GetName(); tmp += " " + val + "\n";
                  }
               }
               return true;
            }
         default: 
            return false;
      }
      return false;   
   }

//______________________________________________________________________________
   bool Value::Write (char* p, int& len) const
   {
      std::string s;
      if (!p || (len < 0) || !Write (s)) {
         len = 0;
         return false;
      }
      else if (len == 0) {
         return true;
      }
      int max = len - 1;
      if (max > (int)s.size()) max = s.size();
      memcpy (p, s.data(), max);
      p[max] = 0;
      len = max;
      return true;
   }

//______________________________________________________________________________
   bool Value::Write (Event& e) const
   {
      switch (mType) {
         case kEvent:
            {
               CopyValue (kEvent, &e, mData);
               return true;
            }
         default: 
            return false;
      }
      return false;   
   }

//______________________________________________________________________________
   bool Value::Write (Enum type, data_ptr p) const
   {
      if (!p) {
         return false;
      }
      switch (type) {
         case kComplex:
            {
               return Write (*(Complex*)p);
            }
         case kReal:
            {
               return Write (*(Real*)p);
            }
         case kInt:
            {
               return Write (*(Int*)p);
            }
         case kTime:
            {
               return Write (*(Time*)p);
            }
         case kString:
            {
               return Write (*(String*)p);
            }
         case kEvent:
            {
               return Write (*(Event*)p);
            }
         default: 
            return false;
      }
      return false;   
   }

//______________________________________________________________________________
   bool Value::Read (Real x)
   {
      *this = Value (x);
      return true;
   }

//______________________________________________________________________________
   bool Value::Read (Int i)
   {
      *this = Value (i);
      return true;
   }

//______________________________________________________________________________
   bool Value::Read (const Complex& c)
   {
      *this = Value (c);
      return true;
   }

//______________________________________________________________________________
   bool Value::Read (const Time& t)
   {
      *this = Value (t);
      return true;
   }

//______________________________________________________________________________
   bool Value::Read (const String& s)
   {
      *this = Value (s);
      return true;
   }

//______________________________________________________________________________
   bool Value::Read (const char* p, int len)
   {
      *this = Value (p, len);
      return true;
   }

//______________________________________________________________________________
   bool Value::Read (const Event& e)
   {
      *this = Value (e);
      return true;
   }

//______________________________________________________________________________
   bool Value::Read (Enum type, const_data_ptr p)
   {
      *this = Value (type, p);
      return true;
   }


//______________________________________________________________________________
// Conversion table
// String only supported for +

   const int kInvld = ColumnType::kInvalid;
   const int kCmplx = ColumnType::kComplex;
   const int kTime  = ColumnType::kTime;
   const int kReal  = ColumnType::kReal;
   const int kInt   = ColumnType::kInt;
   const int kStrng = ColumnType::kString;
   const int kEvent = ColumnType::kEvent;
   const int kNum   = ColumnTypekNum;

   const int plus_table[kNum + 1][kNum + 1] = {
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld},
   {kInvld, kCmplx, kInvld, kCmplx, kCmplx, kInvld, kInvld},
   {kInvld, kInvld, kTime,  kTime , kTime , kInvld, kInvld},
   {kInvld, kCmplx, kTime , kReal , kReal , kInvld, kInvld},
   {kInvld, kCmplx, kTime , kReal , kInt  , kInvld, kInvld},
   {kInvld, kInvld, kInvld, kInvld, kInvld, kStrng, kInvld},
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld}};
   const int minus_table[kNum + 1][kNum + 1] = {
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld},
   {kInvld, kCmplx, kInvld, kCmplx, kCmplx, kInvld, kInvld},
   {kInvld, kInvld, kReal,  kTime , kTime , kInvld, kInvld},
   {kInvld, kCmplx, kInvld, kReal , kReal , kInvld, kInvld},
   {kInvld, kCmplx, kInvld, kReal , kInt  , kInvld, kInvld},
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld},
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld}};
   const int times_table[kNum + 1][kNum + 1] = {
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld},
   {kInvld, kCmplx, kInvld, kCmplx, kCmplx, kInvld, kInvld},
   {kInvld, kInvld, kInvld, kTime , kTime , kInvld, kInvld},
   {kInvld, kCmplx, kTime , kReal , kReal , kInvld, kInvld},
   {kInvld, kCmplx, kTime , kReal , kInt  , kInvld, kInvld},
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld},
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld}};
   const int over_table[kNum + 1][kNum + 1] = {
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld},
   {kInvld, kCmplx, kInvld, kCmplx, kCmplx, kInvld, kInvld},
   {kInvld, kInvld, kInvld, kTime , kTime , kInvld, kInvld},
   {kInvld, kCmplx, kInvld, kReal , kReal , kInvld, kInvld},
   {kInvld, kCmplx, kInvld, kReal , kInt  , kInvld, kInvld},
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld},
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld}};
   const int intonly_table[kNum + 1][kNum + 1] = {
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld},
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld},
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld},
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld},
   {kInvld, kInvld, kInvld, kInvld, kInt  , kInvld, kInvld},
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld},
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld}};
   const int pow_table[kNum + 1][kNum + 1] = {
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld},
   {kInvld, kCmplx, kInvld, kCmplx, kCmplx, kInvld, kInvld},
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld},
   {kInvld, kCmplx, kInvld, kReal , kReal , kInvld, kInvld},
   {kInvld, kCmplx, kInvld, kReal , kReal , kInvld, kInvld},
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld},
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld}};
   const int atan2_table[kNum + 1][kNum + 1] = {
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld},
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld},
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld},
   {kInvld, kInvld, kInvld, kReal , kReal , kInvld, kInvld},
   {kInvld, kInvld, kInvld, kReal , kReal , kInvld, kInvld},
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld},
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld}};
   const int polar_table[kNum + 1][kNum + 1] = {
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld},
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld},
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld},
   {kInvld, kInvld, kInvld, kReal , kReal , kInvld, kInvld},
   {kInvld, kInvld, kInvld, kReal , kReal , kInvld, kInvld},
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld},
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld}};
   const int comp1_table[kNum + 1][kNum + 1] = {
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld},
   {kInvld, kCmplx, kCmplx, kCmplx, kCmplx, kInvld, kInvld},
   {kInvld, kCmplx, kTime , kTime,  kTime , kInvld, kInvld},
   {kInvld, kCmplx, kTime,  kReal , kReal , kInvld, kInvld},
   {kInvld, kCmplx, kTime,  kReal , kInt  , kInvld, kInvld},
   {kInvld, kInvld, kInvld, kInvld, kInvld, kStrng, kInvld},
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld}};
   const int comp2_table[kNum + 1][kNum + 1] = {
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld},
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld},
   {kInvld, kInvld, kTime , kTime,  kTime , kInvld, kInvld},
   {kInvld, kInvld, kTime,  kReal , kReal , kInvld, kInvld},
   {kInvld, kInvld, kTime,  kReal , kInt  , kInvld, kInvld},
   {kInvld, kInvld, kInvld, kInvld, kInvld, kStrng, kInvld},
   {kInvld, kInvld, kInvld, kInvld, kInvld, kInvld, kInvld}};

   const int minus_functable[kNum + 1] = 
   {kInvld, kCmplx, kInvld, kReal , kInt  , kInvld, kInvld};
   const int intonly_functable[kNum + 1] = 
   {kInvld, kInvld, kInvld, kInvld, kInt  , kInvld, kInvld};
   const int abs_functable[kNum + 1] = 
   {kInvld, kReal, kInvld, kReal , kInt  , kInvld, kInvld};
   const int realcmplx_functable[kNum + 1] = 
   {kInvld, kCmplx, kInvld, kReal , kReal , kInvld, kInvld};


//______________________________________________________________________________
   struct plusequal {
   template <typename type>
      void doit (type& v1, const type& v2) {
         v1 += v2; }
   };

template <>
   void plusequal::doit<Time> (Time& v1, const Time& v2) {
      v1 += Interval ((double)v2.getS() + (double)v2.getN()/1E9); }

//______________________________________________________________________________
   struct minusequal {
   template <typename type>
      void doit (type& v1, const type& v2) {
         v1 -= v2; }
   };

//______________________________________________________________________________
   struct timesequal {
   template <typename type>
      void doit (type& v1, const type& v2) {
         v1 *= v2; }
   };

//______________________________________________________________________________
   struct overequal {
   template <typename type>
      void doit (type& v1, const type& v2) {
         v1 /= v2; }
   };

//______________________________________________________________________________
   struct modequal {
   template <typename type>
      void doit (type& v1, const type& v2) {
         v1 %= v2; }
   };

//______________________________________________________________________________
   struct orequal {
   template <typename type>
      void doit (type& v1, const type& v2) {
         v1 |= v2; }
   };

//______________________________________________________________________________
   struct andequal {
   template <typename type>
      void doit (type& v1, const type& v2) {
         v1 &= v2; }
   };

//______________________________________________________________________________
   struct xorequal {
   template <typename type>
      void doit (type& v1, const type& v2) {
         v1 ^= v2; }
   };

//______________________________________________________________________________
   struct leftequal {
   template <typename type>
      void doit (type& v1, const type& v2) {
         v1 <<= v2; }
   };

//______________________________________________________________________________
   struct rightequal {
   template <typename type>
      void doit (type& v1, const type& v2) {
         v1 >>= v2; }
   };

//______________________________________________________________________________
   struct minus_func {
   template <typename type>
      void doit (type& v) {
         v = -v; }
   };

//______________________________________________________________________________
   struct inc_func {
   template <typename type>
      void doit (type& v) {
         ++v; }
   };

//______________________________________________________________________________
   struct dec_func {
   template <typename type>
      void doit (type& v) {
         --v; }
   };

//______________________________________________________________________________
   struct neg_func {
   template <typename type>
      void doit (type& v) {
         v = ~v; }
   };

//______________________________________________________________________________
   struct cmp_equal {
   template <typename type>
      int doit (const type& v1, const type& v2) {
         return (int)(v1 == v2); }
   };
//______________________________________________________________________________
   struct cmp_unequal {
   template <typename type>
      int doit (const type& v1, const type& v2) {
         return (int)(v1 != v2); }
   };

//______________________________________________________________________________
   struct cmp_less {
   template <typename type>
      int doit (const type& v1, const type& v2) {
         return (int)(v1 < v2); }
   };
//______________________________________________________________________________
   struct cmp_lessequal {
   template <typename type>
      int doit (const type& v1, const type& v2) {
         return (int)(v1 <= v2); }
   };

//______________________________________________________________________________
   struct cmp_greater {
   template <typename type>
      int doit (const type& v1, const type& v2) {
         return (int)(v1 > v2); }
   };

//______________________________________________________________________________
   struct cmp_greaterequal {
   template <typename type>
      int doit (const type& v1, const type& v2) {
         return (int)(v1 >= v2); }
   };



//______________________________________________________________________________
template <typename type, class ops>
   inline void binary_op (Value& res, 
                     const Value& v1, const Value& v2) {
      type x, y;
      v1.Write (x);
      v2.Write (y);
      ops().doit (x, y);
      res.Read (x);
   }

//______________________________________________________________________________
template <typename type, class ops>
   inline void cmp_op (Value& res, 
                     const Value& v1, const Value& v2) {
      type x, y;
      v1.Write (x);
      v2.Write (y);
      Value::Int cmp = ops().doit (x, y);
      res.Read (cmp);
   }

//______________________________________________________________________________
template <typename type, class ops>
   inline void unary_op (Value& res, const Value& v) {
      type x;
      v.Write (x);
      ops().doit (x);
      res.Read (x);
   }


//______________________________________________________________________________
   Value& Value::operator += (const Value& val)
   {
      switch (plus_table[mType][val.mType]) {
         case kComplex:
            {
               binary_op<Complex, plusequal> (*this, *this, val);
               break;
            }
         case kReal:
            {
               binary_op<Real, plusequal> (*this, *this, val);
               break;
            }
         case kInt:
            {
               binary_op<Int, plusequal> (*this, *this, val);
               break;
            }
         case kTime:
            {
               binary_op<Time, plusequal> (*this, *this, val);
               break;
            }
         case kString:
            {
               binary_op<String, plusequal> (*this, *this, val);
               break;
            }
         default:
            {
               mType = kInvalid;
               break;
            }
      }
      return *this;
   }

//______________________________________________________________________________
   Value& Value::operator -= (const Value& val)
   {
      switch (minus_table[mType][val.mType]) {
         case kComplex:
            {
               binary_op<Complex, minusequal> (*this, *this, val);
               break;
            }
         case kReal:
            {
               binary_op<Real, minusequal> (*this, *this, val);
               break;
            }
         case kInt:
            {
               binary_op<Int, minusequal> (*this, *this, val);
               break;
            }
         case kTime:
            {
               // special case Time - Time
               if ((mType == kTime) && (val.mType == kTime)) {
                  Time x, y;
                  Interval z;
                  Write (x);
                  val.Write (y);
                  z = x - y;
                  Read ((Real)z);
               }
               // special case Time - Real/Int
               else if ((mType == kTime) && 
                       ((val.mType == kReal) || (val.mType == kInt))) {
                  Time x;
                  Real y;
                  Write (x);
                  val.Write (y);
                  x -= Interval (y);
                  Read (x);
               }
               else {
                  mType = kInvalid;
               }
               break;
            }
         default:
            {
               mType = kInvalid;
               break;
            }
      }
      return *this;
   }

//______________________________________________________________________________
   Value& Value::operator *= (const Value& val)
   {
      switch (times_table[mType][val.mType]) {
         case kComplex:
            {
               binary_op<Complex, timesequal> (*this, *this, val);
               break;
            }
         case kReal:
            {
               binary_op<Real, timesequal> (*this, *this, val);
               break;
            }
         case kInt:
            {
               binary_op<Int, timesequal> (*this, *this, val);
               break;
            }
         case kTime:
            {
               // special case Time * Real/Int
               if ((mType == kTime) && 
                  ((val.mType == kReal) || (val.mType == kInt))) {
                  Time x;
                  Real y;
                  Write (x);
                  val.Write (y);
                  Interval z = x.totalS() * y;
                  x = Time (z.GetS(), z.GetN());
                  Read (x);
               }
               // special case Real/Int * Time
               else if ((val.mType == kTime) && 
                       ((mType == kReal) || (mType == kInt))) {
                  Real x;
                  Time y;
                  Write (x);
                  val.Write (y);
                  Interval z = x * y.totalS();
                  y = Time (z.GetS(), z.GetN());
                  Read (y);
               }
               else {
                  mType = kInvalid;
               }
               break;
            }
         default:
            {
               mType = kInvalid;
               break;
            }
      }
      return *this;
   }

//______________________________________________________________________________
   Value& Value::operator /= (const Value& val)
   {
      switch (over_table[mType][val.mType]) {
         case kComplex:
            {
               binary_op<Complex, overequal> (*this, *this, val);
               break;
            }
         case kReal:
            {
               binary_op<Real, overequal> (*this, *this, val);
               break;
            }
         case kInt:
            {
               binary_op<Int, overequal> (*this, *this, val);
               break;
            }
         case kTime:
            {
               // special case Time / Real/Int
               if ((mType == kTime) && 
                  ((val.mType == kReal) || (val.mType == kInt))) {
                  Time x;
                  Real y;
                  Write (x);
                  val.Write (y);
                  Interval z = x.totalS() / y;
                  x = Time (z.GetS(), z.GetN());
                  Read (x);
               }
               else {
                  mType = kInvalid;
               }
               break;
            }
         default:
            {
               mType = kInvalid;
               break;
            }
      }
      return *this;
   }

//______________________________________________________________________________
   Value& Value::operator %= (const Value& val)
   {
      switch (intonly_table[mType][val.mType]) {
         case kInt:
            {
               binary_op<Int, modequal> (*this, *this, val);
               break;
            }
         default:
            {
               mType = kInvalid;
               break;
            }
      }
      return *this;
   }

//______________________________________________________________________________
   Value& Value::operator |= (const Value& val)
   {
      switch (intonly_table[mType][val.mType]) {
         case kInt:
            {
               binary_op<Int, orequal> (*this, *this, val);
               break;
            }
         default:
            {
               mType = kInvalid;
               break;
            }
      }
      return *this;
   }

//______________________________________________________________________________
   Value& Value::operator &= (const Value& val)
   {
      switch (intonly_table[mType][val.mType]) {
         case kInt:
            {
               binary_op<Int, andequal> (*this, *this, val);
               break;
            }
         default:
            {
               mType = kInvalid;
               break;
            }
      }
      return *this;
   }

//______________________________________________________________________________
   Value& Value::operator ^= (const Value& val)
   {
      switch (intonly_table[mType][val.mType]) {
         case kInt:
            {
               binary_op<Int, xorequal> (*this, *this, val);
               break;
            }
         default:
            {
               mType = kInvalid;
               break;
            }
      }
      return *this;
   }

//______________________________________________________________________________
   Value& Value::operator >>= (const Value& val)
   {
      switch (intonly_table[mType][val.mType]) {
         case kInt:
            {
               binary_op<Int, rightequal> (*this, *this, val);
               break;
            }
         default:
            {
               mType = kInvalid;
               break;
            }
      }
      return *this;
   }

//______________________________________________________________________________
   Value& Value::operator <<= (const Value& val)
   {
      switch (intonly_table[mType][val.mType]) {
         case kInt:
            {
               binary_op<Int, leftequal> (*this, *this, val);
               break;
            }
         default:
            {
               mType = kInvalid;
               break;
            }
      }
      return *this;
   }

//______________________________________________________________________________
   Value operator- (const Value& val)
   {
      Value res;
      switch (minus_functable[val.Type()]) {
         case kCmplx:
            {
               unary_op<Value::Complex, minus_func> (res, val);
               break;
            }
         case kReal:
            {
               unary_op<Value::Real, minus_func> (res, val);
               break;
            }
         case kInt:
            {
               unary_op<Value::Int, minus_func> (res, val);
               break;
            }
      }
      return res;
   }

//______________________________________________________________________________
   Value& operator++ (Value& val)
   {
      switch (intonly_functable[val.Type()]) {
         case kInt:
            {
               unary_op<Value::Int, inc_func> (val, val);
               break;
            }
         default:
            {
               val = Value();
               break;
            }
      }
      return val;
   }

//______________________________________________________________________________
   Value& operator-- (Value& val)
   {
      switch (intonly_functable[val.Type()]) {
         case kInt:
            {
               unary_op<Value::Int, dec_func> (val, val);
               break;
            }
         default:
            {
               val = Value();
               break;
            }
      }
      return val;
   }

//______________________________________________________________________________
   Value operator~ (const Value& val)
   {
      Value res;
      switch (intonly_functable[val.Type()]) {
         case kInt:
            {
               unary_op<Value::Int, neg_func> (res, val);
               break;
            }
      }
      return res;
   }

//______________________________________________________________________________
   Value abs (const Value& val)    
   {
      Value res;
      switch (abs_functable[val.Type()]) {
         case kCmplx:
            {
               Value::Complex x;
               val.Write (x);
               Value::Real y = std::abs (x);
               res.Read (y);
               break;
            }
         case kReal:
            {
               Value::Real x;
               val.Write (x);
               x = ::fabs (x);
               res.Read (x);
               break;
            }
         case kInt:
            {
               Value::Int x;
               val.Write (x);
               x = ::abs (x);
               res.Read (x);
               break;
            }
      }
      return res;
   }

//______________________________________________________________________________
   Value sqrt (const Value& val)        
   {
      Value res;
      switch (realcmplx_functable[val.Type()]) {
         case kCmplx:
            {
               Value::Complex x;  val.Write (x);
               x = std::sqrt (x); res.Read (x);
               break;
            }
         case kReal:
            {
               Value::Real x;  val.Write (x);
               x = ::sqrt (x); res.Read (x);
               break;
            }
      }
      return res;
   }

//______________________________________________________________________________
   Value pow (const Value& v1, const Value& v2)
   {
      Value res;
      switch (pow_table[v1.Type()][v2.Type()]) {
         case kCmplx:
            {
               Value::Complex x;    v1.Write (x);
               Value::Complex y;    v2.Write (y);
               x = std::pow (x, y); res.Read (x);
               break;
            }
         case kReal:
            {
               Value::Real x;       v1.Write (x);
               Value::Real y;       v2.Write (y);
               x = ::exp (y * ::log (x)); 
               res.Read (x);
               break;
            }
      }
      return res;
   }

//______________________________________________________________________________
   Value exp (const Value& val)        
   {
      Value res;
      switch (realcmplx_functable[val.Type()]) {
         case kCmplx:
            {
               Value::Complex x; val.Write (x);
               x = std::exp (x); res.Read (x);
               break;
            }
         case kReal:
            {
               Value::Real x; val.Write (x);
               x = ::exp (x); res.Read (x);
               break;
            }
      }
      return res;
   }

//______________________________________________________________________________
   Value log (const Value& val)        
   {
      Value res;
      switch (realcmplx_functable[val.Type()]) {
         case kCmplx:
            {
               Value::Complex x; val.Write (x);
               x = std::log (x); res.Read (x);
               break;
            }
         case kReal:
            {
               Value::Real x; val.Write (x);
               x = ::log (x); res.Read (x);
               break;
            }
      }
      return res;
   }

//______________________________________________________________________________
   Value log10 (const Value& val)        
   {
      Value res;
      switch (realcmplx_functable[val.Type()]) {
         case kCmplx:
            {
               Value::Complex x;   val.Write (x);
               x = std::log10 (x); res.Read (x);
               break;
            }
         case kReal:
            {
               Value::Real x;   val.Write (x);
               x = ::log10 (x); res.Read (x);
               break;
            }
      }
      return res;
   }

//______________________________________________________________________________
   Value sin (const Value& val)        
   {
      Value res;
      switch (realcmplx_functable[val.Type()]) {
         case kCmplx:
            {
               Value::Complex x; val.Write (x);
               x = std::sin (x); res.Read (x);
               break;
            }
         case kReal:
            {
               Value::Real x; val.Write (x);
               x = ::sin (x); res.Read (x);
               break;
            }
      }
      return res;
   }

//______________________________________________________________________________
   Value cos (const Value& val)        
   {
      Value res;
      switch (realcmplx_functable[val.Type()]) {
         case kCmplx:
            {
               Value::Complex x; val.Write (x);
               x = std::cos (x); res.Read (x);
               break;
            }
         case kReal:
            {
               Value::Real x; val.Write (x);
               x = ::cos (x); res.Read (x);
               break;
            }
      }
      return res;
   }

//______________________________________________________________________________
   Value tan (const Value& val)        
   {
      Value res;
      switch (realcmplx_functable[val.Type()]) {
         case kCmplx:
            {
               Value::Complex x; val.Write (x);
               x = std::tan (x); res.Read (x);
               break;
            }
         case kReal:
            {
               Value::Real x; val.Write (x);
               x = ::tan (x); res.Read (x);
               break;
            }
      }
      return res;
   }

//______________________________________________________________________________
   Value asin (const Value& val)        
   {
      Value res;
      switch (realcmplx_functable[val.Type()]) {
         case kReal:
            {
               Value::Real x;  val.Write (x);
               x = ::asin (x); res.Read (x);
               break;
            }
      }
      return res;
   }

//______________________________________________________________________________
   Value acos (const Value& val)        
   {
      Value res;
      switch (realcmplx_functable[val.Type()]) {
         case kReal:
            {
               Value::Real x;  val.Write (x);
               x = ::acos (x); res.Read (x);
               break;
            }
      }
      return res;
   }

//______________________________________________________________________________
   Value atan (const Value& val)        
   {
      Value res;
      switch (realcmplx_functable[val.Type()]) {
         case kReal:
            {
               Value::Real x;  val.Write (x);
               x = ::atan (x); res.Read (x);
               break;
            }
      }
      return res;
   }

//______________________________________________________________________________
   Value atan2 (const Value& v1, const Value& v2)
   {
      Value res;
      switch (atan2_table[v1.Type()][v2.Type()]) {
         case kReal:
            {
               Value::Real x;      v1.Write (x);
               Value::Real y;      v2.Write (y);
               x = ::atan2 (x, y); res.Read (x);
               break;
            }
      }
      return res;
   }

//______________________________________________________________________________
   Value sinh (const Value& val)        
   {
      Value res;
      switch (realcmplx_functable[val.Type()]) {
         case kCmplx:
            {
               Value::Complex x;  val.Write (x);
               x = std::sinh (x); res.Read (x);
               break;
            }
         case kReal:
            {
               Value::Real x;  val.Write (x);
               x = ::sinh (x); res.Read (x);
               break;
            }
      }
      return res;
   }

//______________________________________________________________________________
   Value cosh (const Value& val)        
   {
      Value res;
      switch (realcmplx_functable[val.Type()]) {
         case kCmplx:
            {
               Value::Complex x;  val.Write (x);
               x = std::cosh (x); res.Read (x);
               break;
            }
         case kReal:
            {
               Value::Real x;  val.Write (x);
               x = ::cosh (x); res.Read (x);
               break;
            }
      }
      return res;
   }

//______________________________________________________________________________
   Value tanh (const Value& val)        
   {
      Value res;
      switch (realcmplx_functable[val.Type()]) {
         case kCmplx:
            {
               Value::Complex x;  val.Write (x);
               x = std::tanh (x); res.Read (x);
               break;
            }
         case kReal:
            {
               Value::Real x;  val.Write (x);
               x = ::tanh (x); res.Read (x);
               break;
            }
      }
      return res;
   }

//______________________________________________________________________________
   Value ceil (const Value& val)        
   {
      Value res;
      switch (realcmplx_functable[val.Type()]) {
         case kReal:
            {
               Value::Real x;  val.Write (x);
               x = ::ceil (x); res.Read (x);
               break;
            }
      }
      return res;
   }

//______________________________________________________________________________
   Value floor (const Value& val)        
   {
      Value res;
      switch (realcmplx_functable[val.Type()]) {
         case kReal:
            {
               Value::Real x;   val.Write (x);
               x = ::floor (x); res.Read (x);
               break;
            }
      }
      return res;
   }

//______________________________________________________________________________
   Value conj (const Value& val)        
   {
      Value res;
      switch (realcmplx_functable[val.Type()]) {
         case kCmplx:
            {
               Value::Complex x;  val.Write (x);
               x = std::conj (x); res.Read (x);
               break;
            }
         case kReal:
            {
               res = val;
               break;
            }
      }
      return res;
   }

//______________________________________________________________________________
   Value polar (const Value& v1, const Value& v2)
   {
      Value res;
      switch (polar_table[v1.Type()][v2.Type()]) {
         case kReal:
            {
               Value::Real x;      v1.Write (x);
               Value::Real y;      v2.Write (y);
               Value::Complex z = std::polar (x, y); 
               res.Read (z);
               break;
            }
      }
      return res;
   }

//______________________________________________________________________________
   Value real (const Value& val)        
   {
      Value res;
      switch (realcmplx_functable[val.Type()]) {
         case kCmplx:
            {
               Value::Complex x;  val.Write (x);
               res.Read ((Value::Real)x.real());
               break;
            }
         case kReal:
            {
               res = val;
               break;
            }
      }
      return res;
   }

//______________________________________________________________________________
   Value imag (const Value& val)        
   {
      Value res;
      switch (realcmplx_functable[val.Type()]) {
         case kCmplx:
            {
               Value::Complex x;  val.Write (x);
               res.Read ((Value::Real)x.imag());
               break;
            }
         case kReal:
            {
               res = Value ((Value::Real)0);
               break;
            }
      }
      return res;
   }

//______________________________________________________________________________
   Value arg (const Value& val)        
   {
      Value res;
      switch (realcmplx_functable[val.Type()]) {
         case kCmplx:
            {
               Value::Complex x;  val.Write (x);
               res.Read ((Value::Real)std::arg (x));
               break;
            }
         case kReal:
            {
               res = Value ((Value::Real)0);
               break;
            }
      }
      return res;
   }

//______________________________________________________________________________
   Value norm (const Value& val)        
   {
      Value res;
      switch (realcmplx_functable[val.Type()]) {
         case kCmplx:
            {
               Value::Complex x;  val.Write (x);
               res.Read ((Value::Real)std::norm (x));
               break;
            }
         case kReal:
            {
               Value::Real x; val.Write (x);
               x *= x;        res.Read (x);
               break;
            }
      }
      return res;
   }

//______________________________________________________________________________
   Value equal (const Value& v1, const Value& v2)
   {
      Value res;
      switch (comp1_table[v1.Type()][v2.Type()]) {
         case kCmplx:
            {
               cmp_op<Value::Complex, cmp_equal> (res, v1, v2);
               break;
            }
         case kReal:
            {
               cmp_op<Value::Real, cmp_equal> (res, v1, v2);
               break;
            }
         case kInt:
            {
               cmp_op<Value::Int, cmp_equal> (res, v1, v2);
               break;
            }
         case kTime:
            {
               cmp_op<Value::time_type, cmp_equal> (res, v1, v2);
               break;
            }
         case kStrng:
            {
               cmp_op<Value::String, cmp_equal> (res, v1, v2);
               break;
            }
      }
      return res;
   }

//______________________________________________________________________________
   Value unequal (const Value& v1, const Value& v2)
   {
      Value res;
      switch (comp1_table[v1.Type()][v2.Type()]) {
         case kCmplx:
            {
               cmp_op<Value::Complex, cmp_unequal> (res, v1, v2);
               break;
            }
         case kReal:
            {
               cmp_op<Value::Real, cmp_unequal> (res, v1, v2);
               break;
            }
         case kInt:
            {
               cmp_op<Value::Int, cmp_unequal> (res, v1, v2);
               break;
            }
         case kTime:
            {
               cmp_op<Value::time_type, cmp_unequal> (res, v1, v2);
               break;
            }
         case kStrng:
            {
               cmp_op<Value::String, cmp_unequal> (res, v1, v2);
               break;
            }
      }
      return res;
   }

//______________________________________________________________________________
   Value less (const Value& v1, const Value& v2)
   {
      Value res;
      switch (comp2_table[v1.Type()][v2.Type()]) {
         case kReal:
            {
               cmp_op<Value::Real, cmp_less> (res, v1, v2);
               break;
            }
         case kInt:
            {
               cmp_op<Value::Int, cmp_less> (res, v1, v2);
               break;
            }
         case kTime:
            {
               cmp_op<Value::time_type, cmp_less> (res, v1, v2);
               break;
            }
         case kStrng:
            {
               cmp_op<Value::String, cmp_less> (res, v1, v2);
               break;
            }
      }
      return res;
   }

//______________________________________________________________________________
   Value lessequal (const Value& v1, const Value& v2)
   {
      Value res;
      switch (comp2_table[v1.Type()][v2.Type()]) {
         case kReal:
            {
               cmp_op<Value::Real, cmp_lessequal> (res, v1, v2);
               break;
            }
         case kInt:
            {
               cmp_op<Value::Int, cmp_lessequal> (res, v1, v2);
               break;
            }
         case kTime:
            {
               cmp_op<Value::time_type, cmp_lessequal> (res, v1, v2);
               break;
            }
         case kStrng:
            {
               cmp_op<Value::String, cmp_lessequal> (res, v1, v2);
               break;
            }
      }
      return res;
   }

//______________________________________________________________________________
   Value greater (const Value& v1, const Value& v2)
   {
      Value res;
      switch (comp2_table[v1.Type()][v2.Type()]) {
         case kReal:
            {
               cmp_op<Value::Real, cmp_greater> (res, v1, v2);
               break;
            }
         case kInt:
            {
               cmp_op<Value::Int, cmp_greater> (res, v1, v2);
               break;
            }
         case kTime:
            {
               cmp_op<Value::time_type, cmp_greater> (res, v1, v2);
               break;
            }
         case kStrng:
            {
               cmp_op<Value::String, cmp_greater> (res, v1, v2);
               break;
            }
      }
      return res;
   }

//______________________________________________________________________________
   Value greaterequal (const Value& v1, const Value& v2)
   {
      Value res;
      switch (comp2_table[v1.Type()][v2.Type()]) {
         case kReal:
            {
               cmp_op<Value::Real, cmp_greaterequal> (res, v1, v2);
               break;
            }
         case kInt:
            {
               cmp_op<Value::Int, cmp_greaterequal> (res, v1, v2);
               break;
            }
         case kTime:
            {
               cmp_op<Value::time_type, cmp_greaterequal> (res, v1, v2);
               break;
            }
         case kStrng:
            {
               cmp_op<Value::String, cmp_greaterequal> (res, v1, v2);
               break;
            }
      }
      return res;
   }



//______________________________________________________________________________
   static char niple2hex (char nip) 
   {
      if ((nip & 0x0F) < 10) {
         return nip + '0';
      } 
      else {
         return nip - 10 + 'A';   
      }
   }

//______________________________________________________________________________
   static char hex2niple (char hex) 
   {
      hex = toupper (hex);
      if ((hex >= 'A') && (hex <= 'F')) {
         return hex + 10 - 'A';
      } 
      else if ((hex >= '0') && (hex <= '9')) {
         return hex - '0';
      }
      else {
         return 0;
      }
   }

//______________________________________________________________________________
   Value bin2hex (const Value& val)
   {
      Value res;
      if (val.Type() != kStrng) {
         return res;
      }
   
      ColumnType::String s;
      val.Write (s);
      ColumnType::String r;
      for (ColumnType::String::iterator p = s.begin(); p != s.end(); ++p) {
         r += niple2hex ((*p >> 4) & 0x0F);
         r += niple2hex (*p & 0x0F);
      }
      res.Read (r);
      return res;
   }

//______________________________________________________________________________
   Value hex2bin (const Value& val)
   {
      Value res;
      if (val.Type() != kStrng) {
         return res;
      }
      ColumnType::String s;
      val.Write (s);
      ColumnType::String r;
      for (ColumnType::String::iterator p = s.begin(); p != s.end(); ) {
         if (!isxdigit (*p)) {
            return res;
         }
         if (p + 1 == s.end()) {
            r += hex2niple (*p) << 4;
            ++p;
         }
         else {
            r += (hex2niple (*p) << 4) + hex2niple (*(p+1));
            p += 2;
         }
      }
      res.Read (r);
      return res;
   }



//______________________________________________________________________________
   bool IVal::Evaluate (const Argument& arg, Value& val) const
   {
      Value tmp;
      if (!mPtr.Get() || !mPtr->Evaluate (arg, tmp)) {
         return false;
      }
      Value::Int i;
      return tmp.Write (i) && val.Read (i);
   }

//______________________________________________________________________________
   bool RVal::Evaluate (const Argument& arg, Value& val) const
   {
      Value tmp;
      if (!mPtr.Get() || !mPtr->Evaluate (arg, tmp)) {
         return false;
      }
      Value::Real x;
      return tmp.Write (x) && val.Read (x);
   }

//______________________________________________________________________________
   bool CVal::Evaluate (const Argument& arg, Value& val) const
   {
      Value tmp;
      if (!mPtr.Get() || !mPtr->Evaluate (arg, tmp)) {
         return false;
      }
      Value::Complex c;
      return tmp.Write (c) && val.Read (c);
   }

//______________________________________________________________________________
   bool TVal::Evaluate (const Argument& arg, Value& val) const
   {
      Value tmp;
      if (!mPtr.Get() || !mPtr->Evaluate (arg, tmp)) {
         return false;
      }
      Value::time_type t;
      return tmp.Write (t) && val.Read (t);
   }

//______________________________________________________________________________
   bool SVal::Evaluate (const Argument& arg, Value& val) const
   {
      Value tmp;
      if (!mPtr.Get() || !mPtr->Evaluate (arg, tmp)) {
         return false;
      }
      Value::String s;
      return tmp.Write (s) && val.Read (s);
   }


//______________________________________________________________________________
   IfoVal::IfoVal (const char* ifoset)
   : Value (IfoSet (ifoset).GetIfoSet())
   {
   }

//______________________________________________________________________________
   IfoVal::IfoVal (const std::string& ifoset)
   : Value (IfoSet (ifoset).GetIfoSet())
   {
   }

//______________________________________________________________________________
   IfoVal::IfoVal (const IfoSet& ifoset)
   : Value (ifoset.GetIfoSet())
   {
   }

}
