/* -*- mode: c++; c-basic-offset: 3; -*- */
//////////////////////////////////////////////////////////////////////////
//  									//
//  Info								//
//  									//
//////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <cstdlib>
#include "events/Info.hh"
#include "events/Argument.hh"
#include "events/Value.hh"
#include "events/Event.hh"


namespace events {

   bool Info::operator== (const Info& info) const
   {
      return strcasecmp (mName.c_str(), info.mName.c_str()) == 0;
   }

//______________________________________________________________________________
   bool Info::InfoValue (const Event& event, Value& val) const
   {
      Window win ((Event*)&event);
      return Evaluate (win, val);
   }

//______________________________________________________________________________
   bool Info::Evaluate (const Argument& arg, Value& val) const
   {
      // check index
      if ((mIndex < 0) || (mIndex >= arg.GetOrder())) {
         return false;
      }
      // Get event
      const Event* event = mColumn.GetEvent (arg, mIndex);
      if (!event) {
         return false;
      }
      // Get column information list
      const ColumnInfoList& clist = event->GetLayout().GetColumnList();
      // Get column index
      ColumnType::Int col1 = mColIndex1;
      ColumnType::Int col2 = mColIndex2;
      ColumnType::String cols = mColName;
      if ((mToken >= kColName) && (mToken <= kColOffset)) {
         // Read index from function?
         if ((col1 < 0) && cols.empty()) {
            // Calculate column index
            Value v;
            if (!mArg1.Get() || !mArg1->Evaluate (arg, v)) {
               return false;
            }
            // Column index is string?
            if (v.Type() == ColumnType::kString) {
               v.Write (cols);
               col1 = -1;
            }
            // Column index is int?
            else if (!v.Write (col1)) {
               return false;
            }
            col2 = col1;
         }
         // Any column index?
         if ((col1 < 0) && cols.empty()) {
            return false;
         }
         // Second index iterator to last?
         if (cols.empty() && (col2 < 0)) {
            col2 = clist.size();
         }
         // Check if indices are within range
         if (cols.empty() && (col1 >= (int)clist.size())) {
            return false;
         }
         // Second index beyond end?
         if (col2 > (int)clist.size()) {
            col2 = clist.size();
         }
      }
      // Get event type
      Type type;
      if (!event->GetLayout().GetType (type)) {
         return false;
      }
      // get information
      bool ret = false;
      switch (mToken) {
         case kInvalid:
         default:
            {
               ret = false;
            }
         case kType:
            {
               ColumnType::String u = type.GetName();
               ret = val.Read (u);
               break;
            }
         case kName:
            {
               ColumnType::String u = event->GetName().GetName();
               ret = val.Read (u);
               break;
            }
         case kTypeId:
            {
               ColumnType::Int u = type.GetId();
               ret = val.Read (u);
               break;
            }
         case kNameId:
            {
               ColumnType::Int u = event->GetName().GetId();
               ret = val.Read (u);
               break;
            }
         case kSize:
            {
               ColumnType::Int u = event->GetLayout().DataSize();
               ret = val.Read (u);
               break;
            }
         case kRefCount:
            {
               ColumnType::Int u = event->GetLayout().GetRefCount();
               ret = val.Read (u);
               break;
            }
         case kColNum:
            {
               ColumnType::Int u = clist.size();
               ret = val.Read (u);
               break;
            }
         case kColName:
            {
               // index is name
               if (!mColName.empty()) {
                  ColumnType::String u;
                  const ColumnInfo* c = 
                     event->GetLayout().GetColumn (mColName.c_str());
                  if (c) u = c->GetName();
                  ret = c && val.Read (u);
               }
               // index is single number
               else if (col1 == col2) {
                  ColumnType::String u;
                  u = clist[col1].GetName();
                  ret = val.Read (u);
               }
               // index is range
               else {
                  ColumnType::String u;
                  for (int i = col1; i < col2; ++i) {
                     if (!u.empty()) u += ",";
                     u += clist[i].GetName();
                  }
                  ret = val.Read (u);
               }
               break;
            }
         case kColType:
            {
               // index is name
               if (!mColName.empty()) {
                  ColumnType::String u;
                  const ColumnInfo* c = 
                     event->GetLayout().GetColumn (mColName.c_str());
                  if (c) u = c->GetTypeName();
                  ret = c && val.Read (u);
               }
               // index is single number
               else if (col1 == col2) {
                  ColumnType::String u;
                  u = clist[col1].GetTypeName();
                  ret = val.Read (u);
               }
               // index is range
               else {
                  ColumnType::String u;
                  for (int i = col1; i < col2; ++i) {
                     if (!u.empty()) u += ",";
                     u += clist[i].GetTypeName();
                  }
                  ret = val.Read (u);
               }
               break;
            }
         case kColTypeId:
            {
               // index is name
               if (!mColName.empty()) {
                  ColumnType::Int u = 0;
                  const ColumnInfo* c = 
                     event->GetLayout().GetColumn (mColName.c_str());
                  if (c) u = c->GetType();
                  ret = c && val.Read (u);
               }
               // index is single number
               else if (col1 == col2) {
                  ColumnType::Int u = 0;
                  u = clist[col1].GetType();
                  ret = val.Read (u);
               }
               // index is range
               else {
                  ColumnType::String u;
                  for (int i = col1; i < col2; ++i) {
                     if (!u.empty()) u += ",";
                     char buf[32];
                     sprintf (buf, "%i", clist[i].GetType());
                     u += buf;
                  }
                  ret = val.Read (u);
               }
               break;
            }
         case kColTypeSize:
         case kColIndex:
         case kColOffset:
            {
               typedef int (ColumnInfo::*prototype) () const;
               prototype proto = 
                  (mToken == kColTypeSize) ? &ColumnInfo::GetTypeSize :
                  (mToken == kColIndex) ? &ColumnInfo::GetColumn :
                  (mToken == kColOffset) ? &ColumnInfo::GetOffset : 0;
               // index is name
               if (!mColName.empty()) {
                  ColumnType::Int u = 0;
                  const ColumnInfo* c = 
                     event->GetLayout().GetColumn (mColName.c_str());
                  if (c) u = (c->*proto)();
                  ret = c && val.Read (u);
               }
               // index is single number
               else if (col1 == col2) {
                  ColumnType::Int u = 0;
                  u = (clist[col1].*proto)();
                  ret = val.Read (u);
               }
               // index is range
               else {
                  ColumnType::String u;
                  for (int i = col1; i < col2; ++i) {
                     if (!u.empty()) u += ",";
                     char buf[32];
                     sprintf (buf, "%i", (clist[i].*proto)());
                     u += buf;
                  }
                  ret = val.Read (u);
               }
               break;
            }
         case kColFixed:
            {
               // index is name
               if (!mColName.empty()) {
                  ColumnType::Int u = 0;
                  const ColumnInfo* c = 
                     event->GetLayout().GetColumn (mColName.c_str());
                  if (c) u = (c->IsFixed() != 0);
                  ret = c && val.Read (u);
               }
               // index is single number
               else if (col1 == col2) {
                  ColumnType::Int u = 0;
                  u = (clist[col1].IsFixed() != 0);
                  ret = val.Read (u);
               }
               // index is range
               else {
                  ColumnType::String u;
                  for (int i = col1; i < col2; ++i) {
                     if (!u.empty()) u += ",";
                     char buf[32];
                     sprintf (buf, "%i", (clist[i].IsFixed() != 0));
                     u += buf;
                  }
                  ret = val.Read (u);
               }
               break;
            }
      }
      return ret;
   }

//______________________________________________________________________________
   void Info::SetName (const char* name)
   {
      // If mColName is not empty, it is used
      // else If mColIndex1 is negative, the index is read from mArg1
      // else If mColIndex2 is negative, it is a range to the end-of-list
      // else If mColIndex2 > mColIndex1, it is a range
      // else It is a single index
      mIndex = 0;
      mName = "";
      mToken = kInvalid;
      mColIndex1 = -1;
      mColIndex2 = -1;
      mColName = "";
      mColumn = Column();
      // Check if valid name
      if (!name || (strlen (name) == 0)) {
         return;
      }
      // Eliminate space
      mName = name;
      std::string::size_type pos;
      while ((pos = mName.find_first_of (" \t\f\n\r\v")) != 
            std::string::npos) {
         mName.erase (pos, 1);
      }
      // Check for index
      if ((pos = mName.find ('[')) != std::string::npos) {
         mIndex = atoi (mName.c_str() + pos + 1);
         mName.erase (pos);
      }
      // Separate out "Column()::"
      bool colspec = false;
      std::string colspecstr;
      if ((pos = mName.find ("::")) != std::string::npos) {
         std::string ctok = mName.substr (0, pos);
         // remove all event columns
         while ((pos = ctok.find_first_of ('.')) != std::string::npos) {
            ctok.erase (pos + 1);
         }
         // remove "Column()::" from name
         mName.erase (mName.find ("::") - ctok.size(), ctok.size() + 2);
         // must be of above format
         if ((strncasecmp (ctok.c_str(), "Column(", 7) != 0) ||
            (ctok.size() <= 7)) {
            return;
         }
         colspecstr = ctok;
         ctok.erase (0, 7);
         // rest must be column name/index
         if (ctok[0 ] == '$') {
            // run time specification
         }
         // index specified
         else if (isdigit (ctok[0])) {
            // get first index
            mColIndex1 = 0;
            while (!ctok.empty() && isdigit (ctok[0])) {
               mColIndex1 = 10* mColIndex1 + ctok[0] - '0';
               ctok.erase (0, 1);
            }
            if (ctok.empty()) {
               mColIndex1 = -1;
               return;
            }
            // range?
            mColIndex2 = mColIndex1;
            if (ctok[0] == ':') {
               ctok.erase (0, 1);
               if (!ctok.empty() && (ctok[0] == ')')) {
                  mColIndex2 = - 1; // range till end
               }
               else if (!ctok.empty() && isdigit (ctok[0])) {
                  // get second index
                  mColIndex2 = 0;
                  while (!ctok.empty() && isdigit (ctok[0])) {
                     mColIndex2 = 10* mColIndex1 + ctok[0] - '0';
                     ctok.erase (0, 1);
                  }
               }
            }
            // check closing )
            if ((ctok.size() != 1) || (ctok[0] != ')')) {
               mColIndex1 = -1;
               mColIndex2 = -1;
               return;
            }
         }
         // assume it is a name
         else {
            // get name
            while (!ctok.empty() && (ctok[0] != ')')) {
               mColName += ctok[0];
               ctok.erase (0, 1);
            }
            // check closing )
            if ((ctok.size() != 1) || (ctok[0] != ')')) {
               mColName = "";
               return;
            }
         }
         colspec = true;
      }
      // seprate out event column array indices
      std::string colname;
      if (mName.find_first_of (".(") != std::string::npos) {
         if ((pos = mName.find_last_of ('.')) != std::string::npos) {
            colname = mName.substr (0, pos + 1);
            mName.erase (pos + 1);
         }   
         if ((pos = mName.find ('(')) != std::string::npos) {
            colname += "Event";
            colname += mName.substr (pos);
            mName.erase (pos);
         }
      }
      // set column 
      mColumn.SetName (colname);
      // now test for info token names
      if (strcasecmp (mName.c_str(), "Type") == 0) {
         mToken = colspec ? kColType : kType;
      }
      else if ((strcasecmp (mName.c_str(), "Name") == 0) &&
              colname.empty()) {
         mToken = kName;
      }
      else if (strcasecmp (mName.c_str(), "TypeId") == 0) {
         mToken = colspec ? kColTypeId : kTypeId;
      }
      else if (strcasecmp (mName.c_str(), "NameId") == 0) {
         mToken = kNameId;
      }
      else if (strcasecmp (mName.c_str(), "Size") == 0) {
         mToken = kSize;
      }
      else if (strcasecmp (mName.c_str(), "RefCount") == 0) {
         mToken = kRefCount;
      }
      else if (strcasecmp (mName.c_str(), "ColumnNumber") == 0) {
         mToken = kColNum;
      }
      else if (strcasecmp (mName.c_str(), "Name") == 0) {
         mToken = kColName;
      }
      else if (strcasecmp (mName.c_str(), "TypeSize") == 0) {
         mToken = kColTypeSize;
      }
      else if (strcasecmp (mName.c_str(), "Fixed") == 0) {
         mToken = kColFixed;
      }
      else if (strcasecmp (mName.c_str(), "Index") == 0) {
         mToken = kColIndex;
      }
      else if (strcasecmp (mName.c_str(), "Offset") == 0) {
         mToken = kColOffset;
      }
      // check validity
      if ((mToken == kInvalid) ||
         ((mToken >= kColName) && (mToken <= kColOffset) && !colspec) ||
         ((mToken < kColName) && colspec) ||
         ((strlen (mColumn.GetName()) > 0) && !mColumn.IsValid())) {
         mToken = kInvalid;
         mColIndex1 = -1;
         mColIndex2 = -1;
         mColName = "";
         mName = name;
         return;
      }
      // set canonical name
      std::string infotok;
      if (colspec) {
         infotok = colspec ? (colspecstr + "::" + mName) : mName;
      }
      mName = mColumn.GetName();
      if (!mName.empty()) mName += '.';
      mName += infotok;
   }

}
