/* -*- mode: c++; c-basic-offset: 3; -*- */
#ifndef _LIGO_EVENTIFOSET_H
#define _LIGO_EVENTIFOSET_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: IfoSet							*/
/*                                                         		*/
/* Module Description: Defines an event	ifo set				*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: IfoSet.html						*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#include <string>
#include "events/Condition.hh"
#include "events/Column.hh"
#include "events/ColumnType.hh"


namespace events {


/** The event ifo set descibes a condition on the origin of the event.
    An Interferometer set is specified with a string consisting of
    one or multiple interferometer tags. An interferometer tag consists
    of a letter and a number. For example "H1" stands for the 4k 
    interferometer at Hanford, whereas "L1H1" stands for both LIGO
    4k interferometers. An ifo set specification can contain 
    wildcards, e.g., "H*" stands for all Hanford interferometers, 
    or "*" stands for all intereferometers.

    The interferometer set is bit encoded and a match can be defined 
    as exact, as a subset, as a superset or as any-one. If exact 
    match is specified, the event must define exaclty the same set 
    of interferometers as the ifo set. If subset is specified 
    the interferometer(s) of the event must be a subset of the ones
    in the IfoSet. Conversely, if superset is specified, the
    event must specifiy at least the interfereometers of the
    IfoSet. If any-one is specified, the event must defined exaclty 
    one out of the interferometers of the IfoSet.
   
    If an ifo set is specified without an index it gets applied to 
    all event indices. An event index can be explicitly specified 
    as part of the string or as a separate argument.
    Examples:
    \begin{verbatim}
    IfoSet ("H1[0]") && IfoSet ("L1", 1)
    \end{verbatim}
    selects LHO 4K events from the first event stream and LLO 4K
    events from the second one.

    As with the column class an interferometer set can also 
    access the information of events stored as event columns. 
    The same notation applies. For example:
    \begin{verbatim}
    IfoSet ("Event(1).H1")
    IfoSet ("H1(1)")
    \end{verbatim}
    selects events which contain an LHO 4K event at column "Event(1)".

    @memo Defines an event ifo set
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class IfoSet : public Condition {
   public:
      /// Ifo set type
      typedef ColumnType::ColumnIfo_t ifoset_type;
      /// Matching criterium
      enum Match {
      /// Exact
      kExact = 0,
      /// Subset
      kSubset,
      /// Superset
      kSuperset,
      /// Any one
      kAnyOne
      };
   
      /** Creates a default event ifo set (matches events which 
          have no interferometer set defined).
          @memo Default constructor
       ******************************************************************/
      IfoSet ()
      : mValid (true), mMatch (kExact), mIndex (-1), mIfo (0) { 
         SetIfo (0); }
      /** Creates an event ifo set from the specified pattern
          and the specified matching criterium.
          @memo Default constructor
          @param ifoset String describing the interferometer set
          @param match Matching criterium
       ******************************************************************/
      explicit IfoSet (const char* ifoset, Match match = kExact)
      : mValid (true), mMatch (match), mIndex (-1), mIfo (0) { 
         SetIfo (ifoset); }
      /** Creates a default event ifo set (matches events which 
          have no interferometer set defined).
          @memo Default constructor
          @param ifoset String describing the interferometer set
          @param match Matching criterium
          @param index Event index
       ******************************************************************/
      explicit IfoSet (const char* ifoset, Match match,
                      int index)
      : mValid (true), mMatch (match), mIndex (-1), mIfo (0) { 
         SetIfo (ifoset); SetIndex (index); }
      /** Creates an ifo set from the specified string.
          @memo Constructor
          @param ifoset String describing the interferometer set
          @param match Matching criterium
       ******************************************************************/
      explicit IfoSet (const std::string& ifoset, Match match = kExact)
      : mValid (true), mMatch (match), mIndex (-1), mIfo (0){
         SetIfo (ifoset.c_str()); }
      /** Creates an ifo set with the specified ID.
          @memo Constructor
          @param ifoset Bit set describing the interferometer(s)
          @param match Matching criterium
          @param index Event index
       ******************************************************************/
      explicit IfoSet (ifoset_type ifoset, Match match = kExact,
                      int index = -1)
      : mValid (true), mMatch (match), mIndex (index), mIfo (ifoset) {
      }
   
      /** Returns a copy of the event condition. This method must be 
          overriden by all descendents.
          @memo Copy the event
          @return event copy
       ******************************************************************/
      virtual IfoSet* Copy() const {
         return new IfoSet (*this); }
      /** Checks if this is a valid ifo set.
          @memo Is valid?
          @return true if valid
       ******************************************************************/
      bool IsValid() const {
         return mValid; }
      /** Equality operator (case insensitive)
          @memo Equality operator
       ******************************************************************/
      bool operator== (const IfoSet& ifoset) const;
      /** Inequality operator.
          @memo Inequality operator
       ******************************************************************/
      bool operator!= (const IfoSet& ifoset) const {
         return !(*this == ifoset); }
   
      /** Evaluate the condition. This function must be overriden
          by descendents.
          @memo Evaluate
          @param arg Event argument list
          @param val Condition value (return)
          @return True if condition could be evaluated
       ******************************************************************/
      virtual bool Evaluate (const Argument& arg, bool& val) const;
   
      /** Set the event index.
          @memo Set event index
          @param index Event index
       ******************************************************************/
      void SetIndex (int index = -1) {
         mIndex = index; }
      /** Returns the event index.
          @memo Get event index
          @return Event index
       ******************************************************************/
      int GetIndex() const {
         return mIndex; }
      /** Set the ifo set from the specified string. Expects a
          full name including (optional) event array, event index
          and interferometer tags.
          @memo Set the ifo set
          @return true if valid string
       ******************************************************************/
      bool SetIfo (const char* ifoset);
      /** Add the specified ifo set to the existing one.
          Only ifo tags are accepted as an argument.
          @memo Add to the ifo set
          @return true if valid string
       ******************************************************************/
      bool AddIfo (const char* ifoset);
      /** Remove the specified ifo set from the existing one.
          Only ifo tags are accepted as an argument.
          @memo Remove from the ifo set
          @return true if valid string
       ******************************************************************/
      bool RemoveIfo (const char* ifoset);
      /** Get the string corresponding to the ifo set. Does include
          the event columns in canonocal form if used.
          @memo Get the ifo set
          @return ifo set string
       ******************************************************************/
      std::string GetIfo () const;
      /** Get the string corresponding to the ifo set tags. Does not
          include event column names.
          @memo Get the ifo set
          @return ifo set string
       ******************************************************************/
      std::string GetIfoTags () const;
      /** Set the ifo set.
          @memo Set the ifo set
          @param ifoset Interferometer set
       ******************************************************************/
      void SetIfoSet (ifoset_type ifoset) {
         mIfo = ifoset; }
      /** Get the ifo set.
          @memo Get the ifo set
          @return Interferometer set
       ******************************************************************/
      ifoset_type GetIfoSet () const {
         return mIfo; }
      /** Set the match criterium.
          @memo Set match criterium
          @param match Matching criterium
       ******************************************************************/
      void SetMatch (Match match) {
         mMatch = match; }
      /** Get the match criterium.
          @memo Get match criterium
          @return Matching criterium
       ******************************************************************/
      Match GetMatch () const {
         return mMatch; }
   
      /** Returns the number of bits.
          @memo Number of bits
          @return Number of bits
       ******************************************************************/
      static int Size() {
         return kIfoNumber; }
      /** Counts the number of bits.
          @memo Number of bits set
          @return Number of bits set
       ******************************************************************/
      int Count() const {
         int c = 0; 
         for (ifoset_type i = mIfo; i; i >>= 1) 
            if (i & 1) ++c;
         return c; }
      /** True if any bit is set.
          @memo Any bit?
          @return True if any bit is set
       ******************************************************************/
      bool Any() const {
         return mIfo != 0; }
      /** True if no bit is set.
          @memo No bit?
          @return True if no bit is set
       ******************************************************************/
      bool None() const {
         return mIfo == 0; }
      /** Test if bit is set.
          @memo Test bit
          @return True if specified bit is set
       ******************************************************************/
      bool TestBit (int bit) const {
	 return (mIfo & (1 << bit)) != 0; }
      /** Set all bits.
          @memo Set all bits
       ******************************************************************/
      IfoSet& Set() {
         mIfo = ~0; 
         return *this; }
      /** Set bit at specified index.
          @memo Set bit
       ******************************************************************/
      IfoSet& Set (int bit, bool set = true) {
         if (set) mIfo ^= (1 << bit); 
         else mIfo &= ~(1 << bit); 
         return *this; }
      /** Reset all bits.
          @memo Reset all bits
       ******************************************************************/
      IfoSet& Reset() {
         mIfo = 0; 
         return *this; }
      /** Reset bit at specified index.
          @memo Reset bit
       ******************************************************************/
      IfoSet& Reset (int bit) {
         return Set (bit, false); }
      /** Flip all bits.
          @memo Flip all bits
       ******************************************************************/
      IfoSet& Flip() {
         mIfo = ~mIfo; 
         return *this; }
      /** Flip bit at specified index.
          @memo Flip bit
       ******************************************************************/
      IfoSet& Flip (int bit) {
         mIfo ^= (1 << bit); 
         return *this; }
   
      /** Tests if the specified ifo set matches according to the
          match criterium. Ignores the event index.
          @memo Test
          @return true if test answer is positive
       ******************************************************************/
      bool Test (const IfoSet& ifoset) const {
         return Test (ifoset.mIfo); }
      /** Tests if the specified ifo set matches according to the
          given criterium. Ignores the event index.
          @memo Test
          @return true if test answer is positive
       ******************************************************************/
      bool Test (ifoset_type ifoset) const;
   
      /** Register a new interferometer.
          @memo Register the ifo
          @param ifostring Interferometer string
          @return Bit index of ifo, or <0 on error
       ******************************************************************/
      static int Register (const char* ifostring);
      /** Get the bit index of an interferometer tag. An ifo tag has to 
          contain a single letter and a number. Returns -1 if
          the interferometer does not exist.
          @memo Get the ifo bit
          @param ifostring Interferometer string
          @return Bit index of tag
       ******************************************************************/
      static int GetBit (const char* ifostring);
      /** Get the encoded value of a set of interferometer tags.
          @memo Get the ifo bits
          @param ifostring Interferometer string
          @param ifoset Bit encoded value  (return)
          @return True if successful
       ******************************************************************/
      static bool GetBits (const char* ifostring, ifoset_type& ifoset);
      /** Get the string of an ifo tag. An empty string is returned if 
          the tag does not exist.
          @memo Get the ifo tag
          @param bit Index of tag
          @return Interferometer string
       ******************************************************************/
      static std::string GetTag (int bit);
      /** Get the interferometer set string from the bit encoded value.
          @memo Get the ifo bits
          @param ifoset Bit encoded value
          @param ifostring Interferometer string (return)
          @return True if successful
       ******************************************************************/
      static bool GetTags (const ifoset_type& ifoset, 
                        std::string& ifostring);
   
   private:
      /// Maximum number of interferometers allowed in a set (32).
      const static int kIfoNumber = 8 * sizeof (ColumnType::ColumnIfo_t);
   
      /// Valid ifo set?
      bool		mValid;
      /// Matching
      Match		mMatch;
      /// Event index
      int		mIndex;
      /// Ifo set
      ifoset_type	mIfo;
      /// Event column access
      Column		mColumn;
   };


}

#endif // _LIGO_EVENTIFOSET_H
