#ifndef _LIGO_EVENTFACTORY_H
#define _LIGO_EVENTFACTORY_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: Factory							*/
/*                                                         		*/
/* Module Description: Event factory					*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: Factory.html						*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/
#ifndef __CINT__
#include <string>
#include <utility>
#include <vector>
#include <iosfwd>
#include "events/Type.hh"
#include "events/TypeInfo.hh"
#include "events/Layout.hh"
#include "events/LayoutInfo.hh"
#include "events/ColumnType.hh"
#include "events/ColumnInfo.hh"
#include "events/IfoSet.hh"
#include "events/IndexList.hh"


namespace events {


/** Defines the event factory. Every program creates a global event 
    factory which is used to manage event types and event layouts. 
   
    @memo Defines an event factory
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class Factory {
   public:      
      /// Fixed column list
      typedef LayoutInfo::ColumnList ColumnList;
      /// Interferometer tag
      typedef std::pair <char, int> IfoTag;
      /// Interferometer tag list
      typedef std::vector <IfoTag> IfoTagList;
      /// Name record
      class NameRecord {
      public:
         NameRecord (const char* name) : mName (name) {
         }
         const char* GetName() const {
            return mName.c_str(); }
         void SetName (const char* name) {
            mName = name; }
      private:
         std::string	mName;
      };
      /// Name list
      typedef IndexList<NameRecord> NameList;
   
      /** Reference to the global event factory.   
          @memo Event factory
       ******************************************************************/
      static Factory& Get();
   
      /** Register an event type.
          @memo Register type
   	  @param name Event type name
   	  @param type Event type information (return)
   	  @param subtype Event subtype id (return)
   	  @return true if successful
       ******************************************************************/
      bool RegisterType (const char* name, TypeInfo*& type);
      /** Get an event type.
          @memo Get type
   	  @param name Event name
   	  @return Pointer to type information if exist
       ******************************************************************/
      TypeInfo* GetType (const char* name);
      /** Tests if the pattern matches the event type. The pattern
          can contain a wildcard.
          @memo Match operator
   	  @param type Event type to match against
   	  @param pat Pattern for event name
   	  @return true if event type matches pattern
       ******************************************************************/
      static bool MatchType (const Type& type, const char* pat);
      /** Check if a valid type or subtype name.
          May not contain parentheses, brackets, dots, stars,
          question marks or spaces.
          @memo Check type/subtype name
   	  @param name Type or Subtype name
   	  @return True if a valid type/subtype name
       ******************************************************************/
      static bool CheckTypeName (const char* name);
   
      /** Register an event name.
          @memo Register name
   	  @param name Event name
   	  @param id Event name id information (return)
   	  @return true if successful
       ******************************************************************/
      bool RegisterName (const char* name, int& id);
      /** Get an event name id.
          @memo Get name id
   	  @param name Event name
   	  @return name Id or zero
       ******************************************************************/
      int GetNameId (const char* name);
      /** Get an event name.
          @memo Get name
   	  @param id Event id
   	  @return name string or NULL
       ******************************************************************/
      const char* GetNameStr (int id);
      /** Tests if the pattern matches the event name. The pattern
          can contain a wildcard.
          @memo Match operator
   	  @param name Event name to match against
   	  @param pat Pattern for event name
   	  @return true if event name matches pattern
       ******************************************************************/
      static bool MatchName (const Name& name, const char* pat);
      /** Check if a valid event name.
          May not contain parentheses, brackets, dots, stars,
          question marks or spaces.
          @memo Check name
   	  @param name Name
   	  @return True if a valid event name
       ******************************************************************/
      static bool CheckNameName (const char* name);
   
      /** Initializes the basic set of event layout.
          @memo Initializes the layout list
       ******************************************************************/
      void InitBasicLayouts();
      /** Register an event layout.
          @memo Register layout
   	  @param layout Event layout
   	  @return Layout info
       ******************************************************************/
      const LayoutInfo* RegisterLayout (const LayoutInfo& layout);
      /** Lookup a layout.
          @memo Lookup layout
   	  @param type Event type
   	  @return Layout info
       ******************************************************************/
      const LayoutInfo* LookupLayout (const Type& type);
      /** Lookup a predefined layout.
          @memo Standard layout
   	  @param name Event type
   	  @return Layout info
       ******************************************************************/
      const LayoutInfo* StandardLayout (const char* name);
      /** Get the Layout add column modification version.
          @memo Get layout add column modification version
   	  @return version
       ******************************************************************/
      int GetLayoutAddColVers() {
         return mLayoutAddColVers; }
      /** Increase the layout add column modification version.
          @memo Increase layout add column modification version
       ******************************************************************/
      void IncreaseLayoutAddColVers() {
         ++mLayoutAddColVers; }
      /** Add a fixed event column (must be done before anything else).
          @memo Add fixed column
   	  @param desc Column descriptor
   	  @return True if successful
       ******************************************************************/
      bool AddFixedColumn (const ColumnInfo& desc);
      /** Check if the name corresponds to a fixed column.
          @memo Check fixed column
   	  @param name Event column name
   	  @return Column descriptor
       ******************************************************************/
      const ColumnInfo* GetFixedColumn (const char* name);
      /** Get the fixed column descriptor list.
          @memo Get list of fixed columns
   	  @return Column descriptor list
       ******************************************************************/
      const ColumnInfoList& FixedColumns();
      /** Get the fixed column index list.
          @memo Get index list of fixed columns
   	  @return Column descriptor list
       ******************************************************************/
      const ColumnList& FixedColumnIndex();
      /** Get next available column offset (in bytes).
          @memo Get first available offset
   	  @return Offset
       ******************************************************************/
      int GetNextAvailableColumnOffset() {
         return mNextColumnOfs; }
      /** Get next available column number.
          @memo Get first available column number
   	  @return Offset
       ******************************************************************/
      int GetNextAvailableColumnNumber() {
         return mFixedColNum; }
   
   
      /** Returns the tag corresponding to the interferometer string.
          @memo String to tag
          @param ifostring Interferometer string
          @param tag Interferometer tag (return)
          @return True if successful
       ******************************************************************/
      static bool IfoString2Tag (const char* ifostring, IfoTag& tag);
      /** Returns the tag list corresponding to the interferometer string.
          @memo String to tag list
          @param ifostring Interferometer string
          @param tag Interferometer tag list (return)
          @return True if successful
       ******************************************************************/
      static bool IfoString2Tags (const char* ifostring, 
                        IfoTagList& list);
      /** Returns the string corresponding to the interferometer tag.
          @memo Tag to string
          @param tag Interferometer tag
          @param ifostring Interferometer string (return)
          @return True if successful
       ******************************************************************/
      static bool IfoTag2String (IfoTag& tag, std::string& ifostring);
      /** Register a new interferometer.
          @memo Register the ifo
          @param ifostring Interferometer string
          @return Bit index of ifo, or <0 on error
       ******************************************************************/
      int IfoRegister (const char* ifostring);
      /** Get the bit index of an interferometer tag. An ifo tag has to 
          contain a single letter and a number. Returns -1 if
          the interferometer does not exist.
          @memo Get the ifo bit
          @param ifostring Interferometer string
          @return Bit index of tag
       ******************************************************************/
      int IfoGetBit (const char* ifostring);
      /** Get the encoded value of a set of interferometer tags.
          @memo Get the ifo bits
          @param ifostring Interferometer string
          @param ifoset Bit encoded value  (return)
          @return True if successful
       ******************************************************************/
      bool IfoGetBits (const char* ifostring, 
                      IfoSet::ifoset_type& ifoset);
      /** Get the string of an ifo tag. An empty string is returned if 
          the tag does not exist.
          @memo Get the ifo tag
          @param bit Index of tag
          @return Interferometer string
       ******************************************************************/
      std::string IfoGetTag (int bit);
      /** Get the interferometer set string from the bit encoded value.
          @memo Get the ifo bits
          @param ifoset Bit encoded value
          @param ifostring Interferometer string (return)
          @return True if successful
       ******************************************************************/
      bool IfoGetTags (const IfoSet::ifoset_type& ifoset, 
                      std::string& ifostring);
   
      /** Dump all registered types.
          @memo Dump all registered types.
          @param os output stream
       ******************************************************************/
      void DumpAllTypes (std::ostream& os) const;
      /** Dump all registered names.
          @memo Dump all registered names.
          @param os output stream
       ******************************************************************/
      void DumpAllNames (std::ostream& os) const;
   
      /** Dump all registered layouts.
          @memo Dump all registered layouts.
          @param os output stream
       ******************************************************************/
      void DumpAllLayouts (std::ostream& os) const;
   
   private:
      /** Creates an event factory.
          @memo Default constructor
       ******************************************************************/
      Factory();
      /** Initializes an event factory.
          @memo Initializes
       ******************************************************************/
      void Init();
      /// Disable copy constructor
      Factory (const Factory&);
      /// Disable assignment
      Factory& operator= (const Factory&);
   
      /** Event type list (must be a pointer list since objects are 
          referenced globaly) */
      TypeList		mTypes;
      /** Event name list */
      NameList		mNames;
      /** Event layout list (must be a pointer list since objects are 
          referenced globaly) */
      LayoutList	mLayouts;
      /// Layout initialized?
      bool		mLayoutInit;
      /// Layout add column modification version
      int		mLayoutAddColVers;
   
      /// Fixed column descriptor list
      ColumnList	mFixedColumns;
      /// Number of fixed columns
      int		mFixedColNum;
      /// Next available offset
      int		mNextColumnOfs;
      /// Can we still add fixed columns?
      bool		mInitFixedColumns;
   
      /// Interferometer tag list; ordered by bits (LSB first)
      IfoTagList	mIfoList;
   
      /// Global event factory
      static Factory*	gFactory;
   };


}
#endif // __CINT__
#endif // _LIGO_EVENTFACTORY_H
