#ifndef _LIGO_EVENTS_H
#define _LIGO_EVENTS_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: Events							*/
/*                                                         		*/
/* Module Description: Header to include all event analysis headers	*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: Events.html						*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#include "events/Type.hh"
#include "events/Name.hh"
#include "events/ColumnType.hh"
#include "events/ColumnInfo.hh"
#include "events/FixedColumns.hh"
#include "events/Layout.hh"
#include "events/Event.hh"
#include "events/EventPtr.hh"

#include "events/Iterator.hh"
#include "events/TimeWindow.hh"
#include "events/Window.hh"
#include "events/WindowIterator.hh"

#include "events/List.hh"
#include "events/ListPtr.hh"
#include "events/Chain.hh"
#include "events/ChainPtr.hh"
#include "events/Set.hh"

#include "events/Argument.hh"
#include "events/Function.hh"
#include "events/FunctionPtr.hh"
#include "events/Value.hh"
#include "events/IfoSet.hh"
#include "events/Column.hh"
#include "events/Info.hh"
#include "events/Math.hh"

#include "events/Condition.hh"
#include "events/ConditionPtr.hh"
#include "events/ConditionLogic.hh"
#include "events/Comparison.hh"
#include "events/Filter.hh"
#include "events/Veto.hh"
#include "events/Cluster.hh"

#include "events/Algorithm.hh"

#include "events/MetaDataUtil.hh"

/** @name Event Analysis Package.
     
    The event analysis package provides means to analysis events 
    generated by the LIGO detectors. Its major components are:
    \begin{verbatim}
    Event - A set of classes defining a general purpose event
    Containers - A set of containers to store events
    Iterators - Iterators for traversing event containers
    Functions - Function objects acting on events
    Conditions - Conditions acting on events
    Algorithms - A set of algorithms to select and histogram events 
    \end{verbatim}

    An event is an object consisting of a time, a type, an 
    interferometer set and an arbitrary number of columns.

    The layout of the class library is as follows
    \begin{verbatim}
  Basic event handling:
    Type - Describes an event type
    Name - Describes an event name
    TypeInfo - Implementation details of the event type class
    ColumnType - Describes the supported types of event columns
    ColumnInfo - Describes an event column
    FixedColumns - Interface to fixed event columns
    Layout - Describes the column layout of an event
    LayoutInfo - Implementation details of the event layout class
    Event - The basic event object (inherits from Type and IfoSet)
    EventPtr - A smart pointer for storing events
    Factory - Defines the event factory which manages types and layouts
  Containers:
    IndexList - A template for a vector with a alphabetic index
    List - A list container for storing a series of events
    ListPtr - A smart pointer for the list container
    Chain - A chain of list containers
    ChainPtr - A smart pointer for the chain container
    Set - A set of chain containers
    TimeWindow - Handles a time window (duration and offset)
    Window - A container to manage the events within a time window
  Iterators:
    Iterator - An iterator for traversing any event container
    ConstIterator - A const iterator for traversing event containers
    IteratorImp - A base class for implmenting the event iterators
    WindowIterator - An iterator for moving a window thru an event container
    ConstWindowIterator - A const window iterator
  Functions:
    Argument - A utility class to manage event function argument lists
    Value - Describes a column value or a constant event function
    Function - The base class of all event functions
    FunctionPtr - A smart pointer for event functions
    ColumnCache - A utility class to implement fast column access
    Column - An event function which returns a column value
    Info - An event function which returns layout information
    Math - Common math routines for the event functions
    MathOps - Defines the event function operators
  Condition:
    Condition - The basic class for event conditions
    ConditionPtr - A smart pointer for event conditions
    ConditionLogic - Defines the boolean operators between conditions
    Comparison - Defines a class to compare event functions
    ComparisonOps - Defines the relational operators
    Filter - A condition which acts as an event filter
    IfoSet - Describes a condition on the interferometer/detector mask
    Veto - A condition which acts as a veto
    Cluster - A condition which recognizes event clusters
  Algorithms:
    Algorithm - Conicidence, cluster analysis, etc.
    \end{verbatim}
    
    A global event "factory" is used to manage the type and layout
    information of events. The following utility classes are
    used internally and generally not accessed directly:
    \begin{verbatim}
    IndexList - A list with an index
    TypeInfo - Information record for a type
    LayoutInfo - Information record for the column layout
    Factory - Event factory
    \end{verbatim}

    Header files are in \URL[here]{headers}.
    Source files are in \URL[here]{sources}.

    Most of these classes are utility classes or are transparent to 
    the user. Typically a user will use the Set container to load 
    events from file; the Column class to pick out column values,
    the Filter and Veto conditions to preselect events and the
    select, coincidence and histgram algorithms of the Set class
    to analyze the events.
     
    A typical session could look like this:
    \begin{verbatim}
    #include "events/Events.hh"
    using namespace std;
    using namespace events;

      // Read some events from file
      Set events ("eventfile.xml");
      // Define an empty event set
      Set selected;
      // Define a second empty event set
      Set selected2;
   
      // Select a all burst events with amplitude > 5
      selected.Select (events, Filter ("burst::*") && Column("Amplitude") > 5);
      // Define a histogram (20 bins from 0 to 20)
      Histogram1 h1 (20, 0, 20);
      // Fill it with the amplitude of the selected events
      selected.Histogram (h1, Column("Amplitude"));
   
      // Select another set of events: all Burst which are not vetoed by
      // a PSL glitch
      selected.Clear();
      // Important: use ConditionPtr rather than condition
      ConditionPtr cond_bursts = 
         Filter ("burst::*") && Column("Amplitude") > 5;
      ConditionPtr cond_pslglitch = 
         Filter ("glitch::psl") && (Column("Amplitude") > 10);
      // Set the veto window to -0.5 to +0.5 sec
      selected.Select (events, cond_bursts && !Veto (cond_pslglitch, -0.5, 1.0));
      // Again histogram it
      Histogram1 h2 (20, 0, 20);
      selected.Histogram (h2, Column("Amplitude"));
   
      // Or do it all in one
      ConditionPtr cond_vetoedbursts = 
         cond_bursts && !Veto (cond_pslglitch, -0.5, 1.0);
      Histogram1 h3 (20, 0, 20);
      events.Histogram (h3, Column("Amplitude"), cond_vetoedbursts);
   
      // Make sure we only have single H1 and L1 events
      events.Select (IfoSet ("H1L1", IfoSet::kAnyOne));
      // select all non-vetoed H1 burst
      selected.Select (events, IfoSet ("H1") && cond_vetoedbursts);
      // add all non-vetoed L1 burst
      selected.SelectAdd (events, IfoSet ("L1") && cond_vetoedbursts);
      // Now look for coincidences; set the coincidence window is 100 ms
      selected.Coincidence (0.1, IfoSet ("H1[0]") && IfoSet ("L1[1]"));
      // Make a 1D histogram of the time difference
      Histogram1 h4 (100, -0.1, 0.1);
      selected.Histogram (h4, Column("Time[1]") - Column("Time[0]"));
   
      // Or do it all in one
      ConditionPtr cond_H1 = IfoSet ("H1[0]") && cond_vetoedbursts;
      // Shift the condition to the second event
      ConditionPtr cond_L1 = IfoSet ("L1[1]") && Shift (cond_vetoedbursts);
      events.Histogram (h4, Column("Time[1]") - Column("Time[0]"), 
                       cond_H1 && cond_L1);
   
      // Check up on clusters
      Set clusters;
      clusters.SetWindow (10.0, -5.0);
      clusters.Clusters (2, 10);
      // Plot cluster number against time
      TSeries t1;
      clusters.TimeSeries (t1, Column("N"));
    \end{verbatim}
   

    @memo Event Analysis Package
    @author Written 2001, 2002 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
//@{

/** @name Definition
   
    @memo Event definition
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
//@{
//@Include: Type.hh
//@Include: Name.hh
//@Include: ColumnType.hh
//@Include: ColumnInfo.hh
//@Include: FixedColumns.hh
//@Include: Layout.hh
//@Include: Event.hh
//@Include: EventPtr.hh
//@Include: TimeWindow.hh
//@}

/** @name Functions 
   
    @memo Event functions
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
//@{
//@Include: Argument.hh
//@Include: Function.hh
//@Include: FunctionPtr.hh
//@Include: Value.hh
//@Include: Column.hh
//@Include: Info.hh
//@Include: Math.hh
//@}

/** @name Conditions 
   
    @memo Event functions
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
//@{
//@Include: Condition.hh
//@Include: ConditionPtr.hh
//@Include: ConditionLogic.hh
//@Include: Comparison.hh
//@Include: Filter.hh
//@Include: IfoSet.hh
//@Include: Veto.hh
//@Include: Cluster.hh
//@}

/** @name Containers
   
    @memo Event containers
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
//@{
//@Include: List.hh
//@Include: ListPtr.hh
//@Include: Chain.hh
//@Include: ChainPtr.hh
//@Include: Set.hh
//@Include: Window.hh
//@}

/** @name Iterators
   
    @memo Event iterators
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
//@{
//@Include: Iterator.hh
//@Include: IteratorImp.hh
//@Include: WindowIterator.hh
//@}

/** @name Algorithms
   
    @memo Event algorithms
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
//@{
//@Include: Algorithm.hh
//@}

/** @name Factory and helper classes
   
    @memo Event factory and helper classes
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
//@{
//@Include: IndexList.hh
//@Include: TypeInfo.hh
//@Include: LayoutInfo.hh
//@Include: Factory.hh
//@}

//@}

#endif // _LIGO_EVENTS_H
