#ifndef TREND_HH
#define TREND_HH

#include "TrendChan.hh"
#include <map>
class FrWriter;

/**  The %Trend class collects LIGO standard trend information and records 
  *  it in trend frames. Multiple channels may be trended and recorded by 
  *  a single %Trend object. The trend is specified with a sample
  *  interval and a number of points per frame. The %Trend class averages 
  *  over all data within the sample interval and produces a frame after 
  *  the desired number of sample intervals has been accumulated. 
  *
  *  The trender can write data frames automatically when data are added to a 
  *  later frames or only when the Update() method is issued. By default, the 
  *  updates are made automatically. This option can be reset using 
  *  setAutoUpdate(false). This option must be set (i.e. no automatic update) 
  *  if data added to a multi-channel trend may overlap the end of a frame.
  *
  *  The trend file name may be specified explicitly with the setFile()
  *  method or may be left to default. The default value for the trend
  *  file name is \c \$DMTRENDOUT/\<ifo\>-\<name\>_[MT]-\<gps\>_\<sec\>.gwf, 
  *  where \c \<ifo\> is the specified IFO name, \c \<name\> is the channel 
  *  group (or monitor) name, \c [MT] is "M" for minute trends or "T" for 
  *  second trends, \c \<gps\> is the GPS time of the start of the trend and 
  *  \c \<sec\> is the number of seconds of data in the trend frame file.
  *
  *  A channel index file may be written in the format used by the CDS
  *  Network Data Server (NDS) by calling writeIndex() after all trend 
  *  channels have been configured.
  *
  *  The basic steps to using the %Trend class are shown below:
  *  \verbatim
  
     ----  Initialization
   
     Trend trend;
     trend.setName("MyTrend");
     trend.setType(Trend::kSecond);
     trend.setIFO("H2");
     trend.setAutoUpdate(false);
     trend.setFrameCount(1);
     trend.addChannel("MyChannelA");
     trend.addChannel("MyChannelB");
     trend.writeIndex();
   
     ----  Adding data
   
     Time t; TSeries ts; Trend::math_t Data;
     ...
     trend.trendData("MyChannelA", t, Data);
     trend.trendData("MyChannelB", ts);
     ...
     trend.Update(ts.getEndTime());
   
     ----  Clean up and go away
   
     trend.write();
     trend.close(); \endverbatim
  *
  *  @memo Accumulate and write trend data frames.
  *  @author John Zweizig.
  *  @version 1.3; Modified April 23, 2002.
  */
class Trend {
public:
  /**  Data type for data counts.
    */
  typedef TrendAcc::count_t count_t;

  /**  Arithmetic data type.
    */
  typedef TrendAcc::math_t math_t;

  /**  Type of data contained in TSeries.
    */
  typedef float trend_t;

  /**  Enumerate trend types
   */
  enum TrendType {
    kSecond,      ///< Second trend, average data over 1 second intervals
    kMinute,      ///< Minute trend, average over aligned 60 second intervals
    kNonStandard  ///< User defined averaging interval.
  };

private:
  /**  Channel database.
   */
  typedef std::map<std::string, TrendChan> ChanMap;

  /**  Iterator over the trend channel database.
   */
  typedef ChanMap::iterator chan_iter;

  /** Constant iterator over the trend channel database
   */
  typedef ChanMap::const_iterator const_chan_iter;

public:
  /**  Construct an empty and unnamed trender.
    *  @memo Default constructor.
    */
  Trend(void);

  /**  Construct an identical copy of an existing trender. The new trender 
    *  is not opened and its frame count is zeroed.
    *  @memo Copy constructor
    *  @param x Trender to be copied.
    */
  Trend(const Trend& x);

  /**  Construct and initialize a %Trend instance.
    *  \brief Data constructor.
    *  \param Name   %Trend monitor name
    *  \param type   %Trend type (minute, second, etc)
    *  \param FrameL Frame length in seconds.
    */
  explicit Trend(const char* Name, TrendType type=kSecond, count_t FrameL=0);

  /**  Destroy the trender. Free all allocated channel data (TrendChan 
    *  instances) and close and release the frame writer.
    *  \brief Destructor.
   */
  ~Trend(void);

  /**  Copy a trender. If the current instance is attached to an FrWriter, it
    *  will be closed and deleted. If the argument trender is attached to an 
    *  FrWriter, it will not be copied. The frame count is set to zero.
    *  \brief Assignment operator.
    *  \param x %Trend to be copied.
    *  \return Reference to the current instance.
    */
  Trend& operator=(const Trend& x);

  /**  Add channels defined by the argument trend to the current instance.
    *  \brief Add argument trend channels.
    *  \param x %Trend containing channels to be added.
    *  \return Reference to the current instance.
    */
  Trend& operator+=(const Trend& x);

  /**  Test whether the specified channel has been defined.
    *  @memo Test for channel
    *  @param ChName %Channel name to be tested.
    *  @return true if channel is defined.
    */
  bool exists(const char* ChName) const;

  /**  Find the %TrendChan entry for the specified channel.
    *  @memo Find a channel entry.
    *  @param ChName Name of channel to be located.
    *  @return Reference to the %TrendChan for the specified channel.
    */
  TrendChan& find(const char* ChName);

  /**  Get a constant reference to the %TrendChan entry for the specified 
    *  channel.
    *  @memo Find a channel entry.
    *  @param ChName Name of channel to be located.
    *  @return Reference to the %TrendChan for the specified channel.
    */
  const TrendChan& find(const char* ChName) const;

  /**  Return the trend channel group name.
    *  @memo %Trend channel group name.
    *  @return String containing the trend name.
    */
  std::string getName(void) const;

  /**  Return the trend file name. The file name is either specified 
    *  explicitly, or written to the data.
    *  @memo   %Trend file name.
    *  @return String containing the trend file name.
    */
  std::string getFileName(void) const;

  /**  Test whether data have been accumulated in any of trend
    *  channels.
    *  @memo Test for non-zero data.
    *  @return True if non-zero data.
    */
  bool isEmpty(void) const;

  /**  Test whether data have been accumulated in any of the trend
    *  channels between the specified start and stop times.
    *  @memo Test for non-zero data.
    *  \param start Start of time interval to be tested
    *  \param stop  End of time interval to be tested
    *  @return True if non-zero data.
    */
  bool isEmpty(const Time& start, const Time& stop) const;

  /**  A channel is added to the trend channel list. Each trend channel 
    *  list entry consists of a trend accumulator, and five time series
    *  containing accumulated entry count, mean, rms, minimum and maximum.
    *  @memo Add a trend channel.
    *  @param Name Name of the new trend channel.
    */
  void addChannel(const char* Name);

  /**  Clear all data from the trend.
    *  @memo Clear the trend.
    */
  void clear(void);

  /**  Write out the currently accumulated trend data and close the file.
    *  @memo Close the trend.
    */
  void close(void);

  /**  Dump the trend status to the specified output stream.
    *  \brief Dump the trend status.
    *  \param os STL output stream.
    *  \return Reference to the output stream.
    */
  std::ostream& dump(std::ostream& os) const;

  /**  Open the trend Frame writer.
    *  @memo open the trend FrWriter.
    */
  void open(void);

  /**  Read a trend from the specified frame file.
    *  \brief Read a trend frame.
    *  \param file %Trend frame path.
    */
  void read(const std::string& file);

  /**  Add all data points in a time series to the trend. Data for the
    *  latest point are left in the running average.
    *  @memo Add points from a time series to the trend.
    *  @param Name %Trend channel name to which the data are added.
    *  @param ts %Time series containing data to be trended.
    */
  void trendData(const char* Name, const TSeries& ts);

  /**  Add a data point to the specified trend running average. If the 
    *  time stamp is in a different time bin than the previous, the 
    *  accumulated average, rms, minimum, maximum and point count are 
    *  stored  in the trend time series and the accumulated values are 
    *  reset. The new data point is then added to the accumulated average.
    *  @memo %Trend a single point.
    *  @param Name %Trend channel name to which the data point is added.
    *  @param t %Time of trended point.
    *  @param point The value of the next element of a series.
    */
  void trendData(const char* Name, const Time& t, math_t point);

  /**  Enable or disable auto-update mode. If the trend writer is in auto
    *  update mode, a trend frame will be written as soon as a data point 
    *  later than the current frame is trended. This may cause the trender
    *  to choke if further data are received for the current frame.
    *  @memo Set auto-update mode.
    *  @param enable The trender is placed in auto-update mode if true.
    */
  void setAutoUpdate(bool enable);

  /**  Set the trend file name.
    *  @memo Set the trend name.
    *  @param Name Name for the trend data.
    */
  void setFile(const char* Name);

  /**  Set the maximum number of trend frames to be written in a single 
    *  file.
    *  @memo Set the maximum number of frames in a file.
    *  @param nFrame Maximum number of frames to be written to a file.
    */
  void setFrameCount(int nFrame);

  /**  Set the maximum number of trend samples to be written in a single 
    *  frame.
    *  @memo Set the maximum number of samples in a frame.
    *  @param nSample Maximum number of samples to be written to a frame.
    */
  void setFrameLen(int nSample);

  /**  Set the interferometer ID. The interferometer ID is used as the
    *  first field of the trend frame file name.
    *  @memo Set IFO name.
    *  @param ifo IFO name.
    */
  void setIFO(const char* ifo);

  /**  Set the trend channel Locale name. The same monitor ID should appear
    *  as the third field of all channel names.
    *  @memo Set the channel monitor ID.
    *  @param Name Monitor name to be used in trend channel names.
    */
  void setMonID(const std::string& Name);

  /**  Set the trend channel group name. When the trend is written the name 
    *  is used as a file extension and as a frame name.
    *  @memo Set the trend channel group name.
    *  @param Name Name for the trend channel group.
    */
  void setName(const char* Name);

  /**  Set the sample interval for non-standard trends.
    *  \brief Set the sample interval
    *  \param time %Trend data sample interval.
    */
  void setSample(Interval time);

  /**  Set the trend type and assigns an appropriate sample interval.
    *  \brief Set the trend type.
    *  \param type %Trend type, either \c kSecond, \c kMinute or 
    *              \c kNonStandard.
    */
  void setType(TrendType type);

  /**  Determine whether data have passed the current frame. If so, write
    *  a frame to the frame file.
    *  @memo Write trend if time is past.
    *  \param t %Time to which the trend is updated/
    */
  void Update(const Time& t=Time(0));

  /**  Write an index file for the Network Data Server (NDS). The NDS 
    *  index file has a list of the channels in the trend frame, organized
    *  as text file with one channel name per line. The channel name is
    *  followed by a data precision argument. At present the trender 
    *  writes all data as "32bit_float". If the file name is not specified 
    *  is specified as a null string, \c \$DMTRENDOUT/channel.cfg is used.
    *  @memo Write an NDS index file
    *  @param file Index file name.
    */
  void writeIndex(const char* file=0) const;

private:
  /**  Set up to build the frame containing the specified time. The start
    *  time is rounded off to an even multiple of 60 for minute frames.
    *  @memo Set up for a new frame.
    *  @param t0 %Time contained in frame.
    */
  void setFrame(const Time& t0);

  /**  Tell the frame writer to write a table of contents. By default, no 
    *  table of contents is written.
    *  @memo Write table of contents.
    *  @param enabletoc Write a Table of contents if enabled.
    */
  void setWriteTOC(bool enabletoc);

  /**  Set up to accumulate the frame containing the specified time. Note 
    *  that a second frame will be set with a GPS minute boundary (an even
    *  multiple of 60 seconds) and a minute frame will be set on a GPS hour
    *  boundary (an even multiple of 3600 seconds).
    *  @memo Set the frame start time.
    *  @param t The start time of the series to be written.
    */
  void startFrame(const Time& t);

  /**  Synchronize all channels to the specified time. If the current 
    *  running average is not empty, it is copied to the time series.
    *  @memo Copy accumulated data to the time series.
    *  @param t %Time to synchronize the data to.
    */
  void synch(const Time& t=Time(0));

  /**  Reset the running average.
    *  @memo add a single point.
    *  @param point The value of the next element of a series.
    */
  void writeFrame(void);

private:
  /**  %Trend channel name base.
    */
  std::string mName;

  /**  File Name.
   */
  std::string mFileName;

  /**  File Name.
   */
  std::string mMonID;

  /**  Interferometer Name
   */
  std::string mIFO;

  /** %Trend type.
   */
  TrendType mType;

  /** Sample time for all trend channels.
   */
  Interval mSample;

  /**  Number of samples per frame.
   */
  count_t mMaxPoints;

  /**  Number of times the Frame has been bumped.
   */
  count_t mNAccess;

  /**  Start time for the current frame.
    */
  Time   mStartFrame;

  /**  End time for the current frame.
    */
  Time   mEndFrame;

  /**  %Time of latest data since start of frame.
    */
  Time   mLastData;

  /**  Channel list
   */
  ChanMap mDict;

  /**  Frame Writer
   */
  FrWriter*  mFWriter;

  /**  Frame Writer
   */
  bool  mAutoUpdate;

  /** Maximum frames in a file
   */
  count_t mFrameMax;

  /** Number of frames in current file
   */
  count_t mFrameCount;

  /** Write a table of contents
   */
  bool mWriteTOC;
};

//======================================  Inline functions
inline std::string 
Trend::getName(void) const {
    return mName;
}

inline std::string 
Trend::getFileName(void) const {
    return mFileName;
}

#endif  // TREND_HH
