/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "ReadTrend.hh"
#include "TSeries.hh"
#include <iostream>
#include <stdexcept>
#include <unistd.h>

using namespace std;

//======================================  Static name extension function
string
ReadTrend::namex(const string& chan, typeID ext) {
    switch (ext) {
    case kCount:
	return chan + ".n";
    case kDelta:
	return chan + ".delta";
    case kError:
	return chan + ".error";
    case kMax:
	return chan + ".max";
    case kMean:
	return chan + ".mean";
    case kMin:
	return chan + ".min";
    case kRMS:
	return chan + ".rms";
    case kSigma:
	return chan + ".sigma";
    }
    throw std::invalid_argument("ReadTrend::namex: Invalid extension code");
}

//======================================  Get extension code
ReadTrend::typeID
ReadTrend::extcode(const std::string& ext) {
    if (ext == "mean")  return kMean;
    if (ext == "sigma") return kSigma;
    if (ext == "min")   return kMin;
    if (ext == "max")   return kMax;
    if (ext == "rms")   return kRMS;
    if (ext == "error") return kError;
    if (ext == "count") return kCount;
    if (ext == "delta") return kDelta;
    throw std::range_error("Invalid trend sub-channel extension.");
}

//======================================  Default constructor
ReadTrend::ReadTrend(void)
  : mPostfix(".gwf"), mType(kMinute), mFrameStat(kEmpty), mDebug(false)
{}

//======================================  Data constructor
ReadTrend::ReadTrend(const char* dir, const char* prefix, 
		     const char* postfix, TrendType Type)
  : mDirec(dir), mType(Type), mFrameStat(kEmpty), mDebug(false)
{
    if (prefix)  mPrefix  = prefix;
    if (postfix) mPostfix = postfix;
    else         mPostfix = ".gwf";
}

//======================================  Get a max data time series
int 
ReadTrend::getMaxSeries(const char* Chan, const Time& t0, Interval dT, 
			TSeries* Data) {

    string_vect  chanvec;
    type_vect    typevec;
    tseries_vect tsvec;

    //----------------------------------  Set up 
    chanvec.push_back(string(Chan));
    typevec.push_back(kMax);
    tsvec.push_back(TSeries());

    int rc = getSeries(chanvec, typevec, t0, dT, tsvec);

    *Data = tsvec[0];
    return rc;
}

//======================================  Get a mean data time series
int 
ReadTrend::getSeries(const char* Chan, const Time& t0, Interval dT, 
		     TSeries* Data, TSeries* Err) {

    string_vect  chanvec;
    type_vect    typevec;
    tseries_vect tsvec;

    //----------------------------------  Set up 
    chanvec.push_back(string(Chan));
    typevec.push_back(kMean);
    tsvec.push_back(TSeries());

    if (Err) {
        chanvec.push_back(string(Chan));
	typevec.push_back(kError);
	tsvec.push_back(TSeries());
    }

    int rc = getSeries(chanvec, typevec, t0, dT, tsvec);

    *Data = tsvec[0];
    if (Err && tsvec.size() > 1) *Err = tsvec[1];
    return rc;
}

//======================================  Add to a list if unique
static void
addUnique(ReadTrend::string_vect& v, const string& s) {
    for (unsigned int i=0 ; i<v.size() ; i++) if (v[i] == s) return;
    v.push_back(s);
}

//======================================  Get a list of data
int 
ReadTrend::getSeries(const string_vect& Chan, const string_vect& Type, 
		     const Time& t0, Interval dT, tseries_vect& Data) {
    type_vect type_j;

    //----------------------------------  Convert type array.
    unsigned int N = Type.size();
    for (unsigned int i=0 ; i<N ; i++) {
	type_j.push_back(extcode(Type[i]));
    }

    //-----------------------------------  Call getSeries
    return getSeries(Chan, type_j, t0, dT, Data);
}

//======================================  Get a list of data
int 
ReadTrend::getSeries(const string_vect& Chan, const type_vect& Type, 
		     const Time& t0, Interval dT, tseries_vect& Data) {

    //----------------------------------  Check arguments
    unsigned int N = Chan.size();
    if (Type.size() != N || Data.size() != N) return 10;
    for (unsigned int i=0 ; i<N ; i++) {
	Data[i].Clear();
    }

    //----------------------------------  Open the files.
    open(t0, dT);
    if (mIn.refList().empty()) return 0;

    //----------------------------------  Add channels.
    string_vect chanList;
    for (unsigned int i=0 ; i<N ; i++) {
        switch (Type[i]) {
	case kCount:
	case kMax:
	case kMean:
	case kMin:
	case kRMS:
	    addUnique(chanList, namex(Chan[i], Type[i]));
	    break;
	case kSigma:
	    addUnique(chanList, namex(Chan[i], kMean));
	    addUnique(chanList, namex(Chan[i], kRMS));
	    break;
	case kError:
	    addUnique(chanList, namex(Chan[i], kMean));
	    addUnique(chanList, namex(Chan[i], kRMS));
	    addUnique(chanList, namex(Chan[i], kCount));
	    break;
	case kDelta:
	    addUnique(chanList, namex(Chan[i], kMin));
	    addUnique(chanList, namex(Chan[i], kMax));
	    break;
	}
    }
    for (unsigned int i=0 ; i<chanList.size() ; i++) {
        mIn.addChannel(chanList[i].c_str());
    }
    if (mDebug) mIn.list(cout);

    //----------------------------------  skip to the right place
    mIn.seek(t0);
    if (mDebug) cout << "ReadTrend: File positioned to GPS " 
		     << mIn.getCurrentTime() << endl;

    //----------------------------------  loop over data stretches
    int      rc    = 0;
    Time     tEnd  = t0 + dT;
    Interval tLeft = tEnd - mIn.getCurrentTime();
    while (!rc && tLeft > Interval(0.0)) {
        rc = mIn.fillData(tLeft);
	Time tStart = mIn.getFillTime();
	for (unsigned int i=0 ; i<N ; i++) {

	    //--------------------------  Pad series to next data
	    if (! Data[i].isEmpty() && Data[i].getEndTime() < tStart ) {
	        if (Type[i] == kError) {
		    Interval    dt = Data[i].getTStep();
		    int      nMiss = int((tStart - Data[i].getEndTime())/dt);
		    double* minus1 = new double[nMiss];
		    for (int j=0 ; j<nMiss ; j++) minus1[j] = -1.0;
		    Data[i].Append(Data[i].getEndTime(), dt, minus1, nMiss);
		    delete[] minus1;
		} else {
		    Data[i].extend(tStart);
		}
	    }

	    //--------------------------  Type dependent processing
	    string ChName;
	    switch (Type[i]) {
	    case kCount:
	    case kMax:
	    case kMean:
	    case kMin:
	    case kRMS:
	        ChName = namex(Chan[i], Type[i]);
		break;
	    case kSigma:
	    case kError:
	        ChName = namex(Chan[i], kRMS);
		break;
	    case kDelta:
	        ChName = namex(Chan[i], kMax);
		break;
	    }

	    //--------------------------  Append the primary time series
	    TSeries* ptr = mIn.refData(ChName.c_str());
	    if (!ptr)  continue;
	    int nBin = ptr->getNSample();
	    if (!nBin) continue;
	    int rc = Data[i].Append(*ptr);
	    if (rc) {
		cerr << "Append failed with rc=" << rc << " for channel: " 
		     << ChName << " at t = " << ptr->getStartTime().getS()
		     << endl;
		continue;
	    }

	    //--------------------------  Calculate sigma
	    if (Type[i] == kSigma) {
	        ChName = namex(Chan[i], kMean);
		ptr = mIn.refData(ChName.c_str());
		if (!ptr) continue; // not a good solution
		const double* pS=reinterpret_cast<const double*>(ptr->refData());
		double*       pD=reinterpret_cast<double*>(Data[i].refData());
		pD += Data[i].getNSample() - nBin;
		for (int j=0 ; j<nBin ; j++) {
		    pD[j] = pD[j]*pD[j] - pS[j]*pS[j];
		    if (pD[j] > 0.0) pD[j] = sqrt(pD[j]);
		    else             pD[j] = 0.0;
		}
	    }

	    //--------------------------  Calculate delta
	    else if (Type[i] == kDelta) {
	        ChName = namex(Chan[i], kMin);
		ptr = mIn.refData(ChName.c_str());
		if (!ptr) continue; // not a good solution
		const float* pS=reinterpret_cast<const float*>(ptr->refData());
		float*       pD=reinterpret_cast<float*>(Data[i].refData());
		pD += Data[i].getNSample() - nBin;
		for (int j=0 ; j<nBin ; j++) {
		    pD[j] -= pS[j];
		}
	    }

	    //--------------------------  Calculate error-bar
	    else if (Type[i] == kError) {
	        ChName = namex(Chan[i], kMean);
		ptr = mIn.refData(ChName.c_str());
		if (!ptr) continue; // not a good solution
	        ChName = namex(Chan[i], kCount);
		TSeries* ptN = mIn.refData(ChName.c_str());
		if (!ptN) continue; // not a good solution
		const double* pS=reinterpret_cast<const double*>(ptr->refData());
		const int*    pN=reinterpret_cast<const int*>(ptN->refData());
		double*       pD=reinterpret_cast<double*>(Data[i].refData());
		pD += Data[i].getNSample() - nBin;
		for (int j=0 ; j<nBin ; j++) {
		    if (pN[j] <= 0) {
		        pD[j] = -1.0;
		    } else if (pN[j] == 1) {
		        pD[j] = 0.0;
		    } else {
		        pD[j] = (pD[j]*pD[j] - pS[j]*pS[j]) / (pN[j]-1);
			if (pD[j] > 0.0) pD[j] = sqrt(pD[j]);
			else             pD[j] = 0.0;
		    }
		}
	    }
	}

	//------------------------------  Recalculate time left.
	tLeft = tEnd - mIn.getCurrentTime();
	if (rc == -1) rc = 0;
	if (rc == -4 && tLeft > Interval(0) && !mIn.refList().empty()) rc = 0;
    }

    //----------------------------------  Remove channels.
    for (unsigned int i=0 ; i<chanList.size() ; i++) {
        mIn.rmChannel(chanList[i].c_str());
    }

    //----------------------------------  close up and go away
    close();
    return rc;
}

//======================================  Set the Directory prefix
void 
ReadTrend::setDirectory(const FrameDir& fd) {
    expunge();
    mFrameDir  = fd;
    mFrameStat = kAssign;
}

//======================================  Set the Directory prefix
void 
ReadTrend::setDirectory(const string& dir, const string& prefix, 
			const string& postfix) {
    //---- Is this right? or is the "&&" supposed to be an "||"
    if (mFrameStat != kPattern || 
	(mDirec != dir && mPrefix != prefix) || 
	mPostfix != postfix) 
    {
	mDirec   = dir;
	mPrefix  = prefix;
	mPostfix = postfix;
	expunge();
    }
}

//======================================  Set the monitor name
void 
ReadTrend::setMonitor(const char* mon) {
    string direc, prefix, postfix;
    const char* tDir = getenv("DMTRENDOUT");
    if (tDir) direc = tDir;

    const char* pIFO = getenv("LIGOSITE");
    if (! pIFO)                     prefix = "X-";
    else if (string(pIFO) == "LHO") prefix = "H-";
    else if (string(pIFO) == "LLO") prefix = "L-";
    else if (string(pIFO) == "CIT") prefix = "C-";
    else if (string(pIFO) == "MIT") prefix = "M-";
    else                            prefix = "X-";
    prefix += mon;
    
    switch (mType) {
    case kMinute:
        prefix += "_M-";
	postfix = "-3600.gwf";
	break;
    case kSecond:
        prefix += "_T-";
	postfix = "-60.gwf";
	break;
    case kNonStandard:
        prefix += "_T-";
	postfix = ".gwf";
	break;
    }
    setDirectory(direc, prefix, postfix);
}

//======================================  Set the trend type
void 
ReadTrend::setType(TrendType Typ) {
    mType = Typ;
}

//======================================  Close the input data accessor
void 
ReadTrend::close(void) {
    mIn.DaccIn::close();
}

//======================================  Expunge the frame file directory
void 
ReadTrend::expunge(void) {
    mFrameDir.clear();
    mFrameStat = kEmpty;
}

//======================================  Open frame data accessor 
void 
ReadTrend::open(const Time& t0, Interval dT) {
    char path[128];
    string namPattern = mDirec + "/";

    mIn.setTOCMode(true);
    mIn.setIgnoreMissingChannel(true);

    Time tEnd   = t0 + dT;
    Time tStart = t0;
    if (mDebug) cout << "Find files for GPS " << tStart.getS() << "-" 
		     << tEnd.getS() << endl;
    if (mType == kNonStandard || mPrefix.empty() || mPostfix.empty()) {
        namPattern += mPrefix + "*" + mPostfix;
	buildDirectory(namPattern.c_str());
	for (FrameDir::file_iterator i=mFrameDir.begin() ; 
	     i != mFrameDir.end() ; i++) {
	    if (i->getStartTime() < tEnd && 
		i->getEndTime()   > tStart) {
		string file = i->getFile();
	        mIn.addFile(file.c_str());
	    } else if (mDebug) {
	        cout << "File: " << i->getFile() << " rejected, start="
		     << i->getStartTime() << ", end=" 
		     << i->getEndTime() << endl;
	    }
	}
    } else if (mType == kMinute) {
        tStart -= Interval(t0.getS()%3600);
	namPattern += mPrefix + "%s" + mPostfix;
	for (Time t=tStart ; t<tEnd ; t+=Interval(3600.0) ) {
	    TimeStr(t, path, namPattern.c_str());
	    if (access(path, R_OK)) mIn.addFile(path);
	}
    } else if (mType == kSecond) {
        tStart -= Interval(t0.getS()%60);
	namPattern += mPrefix + "%s" + mPostfix;
	for (Time t=tStart ; t<tEnd ; t+=Interval(60.0) ) {
	    TimeStr(t, path, namPattern.c_str());
	    if (access(path, R_OK)) mIn.addFile(path);
	}
    } else {
        cerr << "ReadTrend: Internal error. Unable to find trends" << endl;
    }
    if (mDebug) mIn.refList().print(cout);
}

//======================================  Build the frame file directory
void 
ReadTrend::buildDirectory(const char* name) {
    if (mFrameStat == kEmpty) {
	if (mDebug) 
	    cout << "ReadTrend: Building directory for " << name << endl;
	mFrameDir.add(name);
	mFrameStat = kPattern;
    }
}

//======================================  Turn on debugging
void 
ReadTrend::setDebug(void) {
    mDebug = true;
    mIn.setDebug(9);
    mFrameDir.setDebug(9);
}
