/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "FrWriter.hh"
#include "TSeries.hh"
#include "FSeries.hh"
#include "FSpectrum.hh"
#include "DVector.hh"
#include "FrStatDataRef.hh"
#include "framecpp/FrameH.hh"
#include "framecpp/FrRawData.hh"
#include "framecpp/FrSimEvent.hh"
#include "framecpp/FrDetector.hh"
#include "framecpp/FrHistory.hh"
#include "framecpp/FrStatData.hh"

#if LDAS_VERSION_NUMBER < 109360
#include "framecpp/util.hh"
#else
#include "framecpp/FrameCPP.hh"
#if LDAS_VERSION_NUMBER > 119013
#include "framecpp/Common/FrameBuffer.hh"
#endif
#endif
#include "framecpp/OFrameStream.hh"

#if LDAS_VERSION_NUMBER < 200000
#include "general/gpstime.hh"
using General::GPSTime;
#else
#include "framecpp/GPSTime.hh"
using FrameCPP::GPSTime;
#endif

#define DACC_ONLDEV "/online/"
#ifndef DMTOFFLINE
#include "oSMbuf.hh"
#endif

#include <fstream>
#include <sstream>
#include <cstdio>

using namespace std;
using FrameCPP::FrAdcData;
using FrameCPP::FrDetector;
using FrameCPP::FrameH;
using FrameCPP::FrProcData;
using FrameCPP::FrRawData;
using FrameCPP::FrSimData;
using FrameCPP::FrSimEvent;
using FrameCPP::FrStatData;
using FrameCPP::Common::FrameBufferInterface;
using FrameCPP::Common::FrameBuffer;

inline GPSTime 
CPPTime(const Time& t) 
{
    return GPSTime(t.getS(), t.getN());
}

inline GPSTime
CPPTime(const Interval& t) 
{
    return GPSTime(t.GetS(), t.GetN());
}

//======================================  Stream buffer cast
#if LDAS_VERSION_NUMBER >= 109360
template <class SB>
inline SB&
stream_ref(FrameBufferInterface& fbi) {
    return dynamic_cast<FrameBuffer<SB>&>(fbi);
}
#endif

//======================================  Constructors and destructors.
FrWriter::FrWriter(void) 
  : mFile(""), mDebug(0), mFrameID(0), mName(""), mRun(0), 
    mCompress(FrVectRef::kUncompressed), mOnline(false), mChkSumStruct(1),
    mChkSumFile(1)
{
#if LDAS_VERSION_NUMBER < 109360
    mFrame  = 0;
    mStream = 0;
#elif !defined(FCPP_SHARED_PTRS)
    mFrame  = 0;
    mFrameBuf = 0;
#else
    mFrameBuf = 0;
#endif
    mWriter = 0;
}

FrWriter::FrWriter(const char* Name, int Run) 
  : mFile(""), mDebug(0), mFrameID(0), mName(Name), 
    mCompress(FrVectRef::kUncompressed), mOnline(false), mChkSumStruct(1),
    mChkSumFile(1)
{
#if LDAS_VERSION_NUMBER < 109360
    mFrame  = 0;
    mStream = 0;
#elif !defined(FCPP_SHARED_PTRS)
    mFrame  = 0;
    mFrameBuf = 0;
#else
    mFrameBuf = 0;
#endif
    mWriter = 0;
    setRunID(Run);
}

FrWriter::~FrWriter() {
    mChanList.erase(mChanList.begin(), mChanList.end());
    if (mWriter) close();
#if LDAS_VERSION_NUMBER < 109360
    if (mStream) delete mStream;
#else
    if (mFrameBuf) {
	if (mFrameBuf) stream_ref<std::filebuf>(*mFrameBuf).close();
	delete mFrameBuf;
	mFrameBuf = 0;
    }
#endif
    erase();
}

//======================================  Status manipulation
void 
FrWriter::setDebug(int debug) {
    mDebug = debug;
}

//======================================  Channel list manipulation
void 
FrWriter::addChannel(const char* Name, TSeries** TSptr) {
    mChanList.push_back(Channel(Name, TSptr));
}

void 
FrWriter::rmChannel(const char* Name) {
    for (chan_iter i=mChanList.begin() ; i != mChanList.end() ; i++) {
        if (i->EqName(Name)) {
	    mChanList.erase(i);
	    break;
	}
    }
}

//======================================  Translate the checksum method
inline FrameCPP::Common::CheckSum::kind_type
cvt_cksum_type(int i) {
    switch (i) {
    case 0:
	return FrameCPP::Common::CheckSum::NONE;
    case 1:
	return FrameCPP::Common::CheckSum::CRC;
    default:
	throw runtime_error("Undefined checksum type");
    }
}

//======================================  Open the frame output file.
int 
FrWriter::open(const char* File, bool toc) {
    mFile = File;
    if (mDebug) cout << "Opening file: " << File << endl;
    FrameCPP::OFrameStream* w;
#if LDAS_VERSION_NUMBER < 109360
    mStream = new ofstream(File, ios::out);
    if (!mStream->good()) return -1;
    w = new FrameCPP::OFrameStream(*mStream);
#else
    auto_ptr <FrameBufferInterface > fb;
    string::size_type l = strlen(DACC_ONLDEV);
    if (mFile.substr(0, l) == DACC_ONLDEV) {
#ifdef DMTOFFLINE
	cerr << "Shared memory partitions not available in offline version" 
	     << endl;
#else
	fb.reset(new FrameBuffer<oSMbuf>(ios::out));
	if ( !stream_ref<oSMbuf>(*fb).open(mFile.substr(l).c_str(), ios::out) )
	    return -1;
	mOnline = true;
#endif
    } else {
	fb.reset(new FrameBuffer<filebuf>(ios::out));
	if (!stream_ref<filebuf>(*fb).open(mFile.c_str(), ios::out)) return -1;
	mOnline = false;
    }
    mFrameBuf = fb.release();
    w = new FrameCPP::OFrameStream(mFrameBuf);
    w->SetCheckSumFile(cvt_cksum_type(mChkSumFile));
#endif
    return open(w);
}
 
int 
FrWriter::open(FrameCPP::OFrameStream* writer) {
    mWriter = writer;
    return 0;
}

//======================================  Close the frame writer.
void
FrWriter::close(void) {
#if LDAS_VERSION_NUMBER < 109360
    if (mWriter) {
        delete mWriter;
	mWriter = 0;
    }
    if (mStream) {
	((ofstream*)mStream)->close();
        delete mStream;
	mStream = 0;
    }
#else
    if (mWriter) {
	mWriter->Close();
	if (mFrameBuf) {
	    if (!mOnline) stream_ref<filebuf>(*mFrameBuf).close();
	    //	    else          stream_ref<oSMbuf> (*mFrameBuf).close();
	}
        delete mWriter; // deletes FrameBuf too!
	mWriter = 0;
	mFrameBuf = 0;
    }
#endif
}

//======================================  Erase the frame.
void
FrWriter::erase(void) {
#ifndef FCPP_SHARED_PTRS
    if (mFrame) {
        delete mFrame;
	mFrame = 0;
    }
    while (!mStatTBD.empty()) {
        delete mStatTBD.back();
	mStatTBD.pop_back();
    }
#else
    mFrame.reset();
    mStatTBD.clear();
#endif
}
 
//======================================  Construct the frame from nothing
//
//   Return codes:
//     0   Everything worked OK
//     1   One or more TSeries are missing.
//     2   Insufficient data
//
int 
FrWriter::buildFrame(const Time& Start, const Interval& dT) {
  if (mDebug) cerr << "Building frame: " << Start.getS() << "-" 
		   << long(dT) << endl; 
    //----------------------------------  Find the start and interval
    if (!Start || !dT) {
	Time tStart(0), tStop(0);
	for (chan_iter i=mChanList.begin() ; i != mChanList.end() ; i++) {
	    const TSeries* p = i->refSeries();
	    if (!p) return 1;
	    if (i == mChanList.begin()) {
	        tStart = p->getStartTime();
		tStop  = p->getEndTime();
	    } else {
	        if (p->getStartTime() < tStart) tStart = p->getStartTime();
		if (p->getEndTime()   > tStop ) tStop  = p->getEndTime();
	    }
	}

	if (!Start) mTime = tStart;
	else        mTime = Start;

	if (dT)          mFrLength = dT;
	else if (!tStop) mFrLength = 0.0;
	else             mFrLength = tStop - mTime;
    } else {
        mTime     = Start;
        mFrLength = dT;
    }

    //----------------------------------  Create the frame header
    mFrame = frame_ptr_type(new FrameH(mName, mRun, mFrameID++, CPPTime(mTime),
				       Leap0+LeapS(mTime), mFrLength));

    //----------------------------------  Add copy in all TSeries
#ifndef FCPP_SHARED_PTRS
    mFrame->SetRawData(FrRawData("Raw Data"));
#else
    frrawdata_pointer raw(new FrRawData("Raw Data"));
    mFrame->SetRawData(raw);
#endif
    for (chan_iter i=mChanList.begin() ; i != mChanList.end() ; i++) {
        const TSeries* p = i->refSeries();
	if (!p) return 1;
	if (p->getInterval() <= mFrLength) {
	    addRawSeries(i->getName(), *p);
	} else {
	    addRawSeries(i->getName(), p->extract(mTime, mFrLength));
	}
    }

    //----------------------------------  Done, return
    return 0;
}

//======================================  Add a raw data series
void
FrWriter::addFrequencySeries(const char* Name, const FSeries& fs) {
    double fMin = fs.isDoubleSided() ? fs.getLowFreq() : fs.getCenterFreq();
    double dF   = fs.getFStep();
    FrVectRef pVect(*fs.refDVect(), fMin, dF, "Hz");
    pVect.setName(Name);
    addProcData(Name,                         // Chan. Name
		"",                           // Comment
		FrProcData::FREQUENCY_SERIES, // type
		FrProcData::DFT,              // Subtype
		fs.getStartTime(),            // Time offset
		fs.getEndTime(),              // Time duration
		0.0,                          // Freq. shift
		0.0,                          // Phase
		double(fs.getNStep())*dF,     // Freq. Range
		0.0,                          // BandWidth
		pVect);                       // FrVect
}

//======================================  Add a processed data structure
void 
FrWriter::addProcData(const std::string& Name, const std::string& comment, 
		      int type, int subtype, const Time& Start, const Time& End,
		      double fShift, double fPhase, double fRange, double bw,
		      FrVectRef& vect) 
{
    if (vect.empty()) return;
    if (mDebug > 1) cerr << "Adding FrProcData: " << Name << endl;

    //----------------------------------  Make sure start time is valid.
    Interval Offset = Start > mTime ? Start - mTime : Interval(0.0);
    Interval tRange = End - Start;
    if (!mFrLength) mFrLength = Offset + tRange;

    //----------------------------------  Build the processed data structure
    FrProcData pdata(Name, comment, type, subtype, Offset, tRange, 
		     fShift, fPhase, fRange, bw);
    FrameH::procData_iterator iproc = mFrame->RefProcData().append(pdata);

    //----------------------------------  Compress and add the data vector
    vect.compress(mCompress);
#ifndef FCPP_SHARED_PTRS
    (*iproc)->RefData().append(vect.release(), false, true);
#else
    (*iproc)->RefData().append(vect.release());
#endif
}

//======================================  Add a raw data series
void
FrWriter::addRawSeries(const char* Name, const TSeries& TS) {
    if (TS.isEmpty()) return;

    //----------------------------------  Make sure start time is valid.
    Time Start = TS.getStartTime();
    Interval Offset = Start - mTime;
    if (!mFrLength) mFrLength = TS.getEndTime() - mTime;

    //----------------------------------  Make a Vect, get data length
    FrVectRef pVect(*TS.refDVect(), Offset, double(TS.getTStep()), "s");
    if (pVect.empty()) return;
    pVect.setName(Name);
    pVect.setUnits(TS.getUnits());
    pVect.compress(mCompress);

    //----------------------------------  Get a pointer to the Adc Container.
    frrawdata_pointer raw(mFrame->GetRawData());

    //----------------------------------  Add the adc block
    FrRawData::firstAdc_iterator 
        iadc = raw->RefFirstAdc().append(FrAdcData(Name,      // Chan. Name
					    0,                // Crate #
					    0,                // Channel #
					    pVect.getNBits(), // # bits
					    1./TS.getTStep(), // Sample rate
					    0.0,              // Bias
					    1.0,              // Slope
					    TS.getUnits(),    // Units
					    TS.getF0(),       // Freq. shift
					    Offset)           // Time offset
				    );
#ifndef FCPP_SHARED_PTRS
    (*iadc)->RefData().append(pVect.release(), false, true);
#else
    (*iadc)->RefData().append(pVect.release());
#endif
}

//======================================  Add a processed data channel
void
FrWriter::addProcSeries(const char* Name, const TSeries& TS) {

    //----------------------------------  Build a vector
    Interval Offset = TS.getStartTime() - mTime;
    FrVectRef pVect(*TS.refDVect(), Offset, TS.getTStep(), "s");
    pVect.setName(Name);
    pVect.setUnits(TS.getUnits());
    addProcData(Name,                    // Chan. Name
		TS.getName(),            // Comment
		FrProcData::TIME_SERIES, // type
		0,                       // subtype
		TS.getStartTime(),       // Start time.
		TS.getEndTime(),         // End time.
		TS.getF0(),              // Freq. shift
		0.0,                     // Phase
		0.0,                     // f-Range
		0.0,                     // BW
		pVect);                  // FrVect
}

//======================================  Add a simulated data channel
void 
FrWriter::addSimEvent(const char* Name, const char* Comment, 
		      const char* Inputs,  const Time& GPSMax, 
		      Interval before, Interval after, float A, 
		      const std::vector<std::pair<std::string, float> >& v)
{
#if LDAS_VERSION_NUMBER < 109360
    FrameCPP::FrSimEvent sim(Name, Inputs, CPPTime(GPSMax), before, after, A);
    sim.AppendComment(Comment);
#else
    FrameCPP::FrSimEvent sim(Name, Comment, Inputs, CPPTime(GPSMax), before, 
			     after, A, FrameCPP::FrSimEvent::ParamList_type());
#endif
    int N = v.size();
    sim.GetParam().resize(N);
    for (int i=0 ; i<N ; i++)
        sim.GetParam()[i] = FrSimEvent::Param_type(v[i].first, v[i].second);
    mFrame->RefSimEvent().append(sim);
}

//======================================  Add a simulated data channel
void
FrWriter::addSimSeries(const char* Name, const TSeries& TS) {

    //----------------------------------  Make sure start time is valid.
    Time Start = TS.getStartTime();
    Interval Offset = Start - mTime;
    Interval tRange = TS.getEndTime() - Start;
    if (!mFrLength) mFrLength = Offset + tRange;
    double sRate = 1.0/double(TS.getTStep());

    //----------------------------------  Build a vector
    FrVectRef pVect(*TS.refDVect(), double(Offset), TS.getTStep(), "s");
    if (pVect.empty()) return;
    pVect.setName(Name);
    pVect.setUnits(TS.getUnits());
    pVect.compress(mCompress);

    //----------------------------------  Add the processed data structure
    FrameH::simData_iterator 
        iproc = mFrame->RefSimData().append(FrSimData(
					        Name,         // Chan. Name
					        TS.getName(), // Comment
					        sRate,        // Sample rate
					        TS.getF0(),   // Freq. shift
					        0.0)          // Phase
					    );
#ifndef FCPP_SHARED_PTRS
    (*iproc)->RefData().append(pVect.release(), false, true);
#else
    (*iproc)->RefData().append(pVect.release());
#endif
}

//======================================  Add a PSD channel
void
FrWriter::addSpectrum(const char* Name, const FSpectrum& Sp) {
    FrVectRef pVect(*Sp.refDVect(), Sp.getLowFreq(), Sp.getFStep(), "Hz");
    pVect.setName(Name);
    addProcData(Name,                               // Chan. Name
		"",                                 // Comment
		FrProcData::FREQUENCY_SERIES,       // type
		FrProcData::POWER_SPECTRAL_DENSITY, // subtype
		Sp.getStartTime(),                  // Time offset
		Sp.getEndTime(),                    // Time duration
		0.0,                                // Freq. shift
		0.0,                                // Phase
		Sp.getHighFreq() - Sp.getLowFreq(), // f range
		0.0,                                // Bandwidth
                pVect);                             // FrVect
}

//=======================================  Add an arbitrary static structure
void 
FrWriter::addStatic(FrStatDataRef& s) 
{
    if (mDebug > 1) cerr << "Adding FrStatData: " << s.getName() 
			 << " vector length: " << s.getFrVect().size() << endl;
    FrStatDataRef::stat_ptr_type StatData = s.release();
    mStatTBD.push_back(StatData);
#ifndef FCPP_SHARED_PTRS
    mWriter->WriteFrStatData( *StatData );
#else
    mWriter->WriteFrStatData( StatData );
#endif
}

//======================================  Add a static FSeries
void
FrWriter::addStatic(const string& Name, unsigned long vsn, const Time& start, 
		    const Time& end, void* det, const FSeries& fs) {
    double fMin = fs.isDoubleSided() ? fs.getLowFreq() : fs.getCenterFreq();
    double dF   = fs.getFStep();
    FrVectRef pVect(*fs.refDVect(), fMin, dF, "s^-1");
    pVect.setName(Name);
    string cmt = fs.getName();
    string rep = "freq_series";
    FrStatDataRef stat(Name, cmt, rep, vsn, start, end, det, pVect);
    addStatic(stat);
}

//======================================  Add a static TSeries
void
FrWriter::addStatic(const string& Name, unsigned long vsn, const Time& start, 
		    const Time& end, void* det, const TSeries& ts) {
    double tMin = ts.getStartTime() - start;
    double dT   = ts.getTStep();
    FrVectRef pVect(*ts.refDVect(), tMin, dT, "s");
    pVect.setName(Name);
    pVect.setUnits(ts.getUnits());
    string cmt = ts.getName();
    string rep = "time_series";
    FrStatDataRef stat(Name, cmt, rep, vsn, start, end, det, pVect);
    addStatic(stat);
}

//======================================  Add a detector record.
void* 
FrWriter::addStdDetector(const string& prefix) {
    string pfx(prefix.substr(0, 2));
    FrDetector* pDet(0);
    if (pfx == "H1") {
        pDet = new FrDetector("LHO_4k", "H1", -2.08406, 0.810795, 142.554,
			      5.65488, 4.08408, -0.0006195, 1.25e-05, 
			      1997.54, 1997.52, 1);
    } else if (pfx == "H2") {
        pDet = new FrDetector("LHO_2k", "H2", -2.08406, 0.810795, 142.554,
			      5.65488, 4.08408, -0.0006195, 1.25e-05, 
			      1004.5, 1004.5, 1);
    } else if (pfx == "L1") {
        pDet = new FrDetector("LLO_4k", "L1", -1.58431, 0.533423, -6.574,
			      4.40318, 2.83238, -0.0003121, -0.0006107, 
			      1997.57, 1997.57, 1);
    }
#ifndef FCPP_SHARED_PTRS
    if (pDet) mFrame->RefDetectProc().append(pDet, false, true);
#else
    if (pDet) mFrame->RefDetectProc().append(frdetector_pointer(pDet));
#endif
    return pDet;
}

//======================================  Identify this writer in frame
void 
FrWriter::addWriterHistory(void) {
    ostringstream comment;
    comment << "Frwriter version $Name$ using FrameCPP Version " 
	    << FrameCPP::GetVersion() << " added to CVS at "
	    << FrameCPP::GetCVSDate() << " built "
	    << FrameCPP::GetBuildDate() << ends;
    addHistory("FrWriter", Now(), comment.str());
}

//======================================  Add a history record to frame
void 
FrWriter::addHistory(const string& Name, const Time& t, const string& cmt) {
    if (!mFrame) return;
    mFrame->RefHistory().append(FrameCPP::FrHistory(Name, t.getS(), cmt));
}

//======================================  Write frame to output stream.
int 
FrWriter::writeFrame(void) {
    if (!mFrame) return 0;
    if (!mWriter) return -1;

#ifndef FCPP_SHARED_PTRS
    mWriter->WriteFrame(*mFrame, cvt_cksum_type(mChkSumStruct));
#else
    mWriter->WriteFrame(mFrame, cvt_cksum_type(mChkSumStruct));
#endif

#if !defined(DMTOFFLINE) && LDAS_VERSION_NUMBER >= 109360
    if (mOnline) stream_ref<oSMbuf>(*mFrameBuf).set_id(mTime.getS());
#endif
    erase();
    return 0;
}
 
//======================================  Get the debug level.
int 
FrWriter::getDebug(void) const {
    return mDebug;
}

//======================================  Get the output file name.
const char* 
FrWriter::getFile(void) const {
    return mFile.c_str();
}

//======================================  List requested channels
ostream&
FrWriter::list(ostream& out) const {
    char buf[2048];
    sprintf (buf, "Channel                  Latest-Time   Pointer\n");
    out << buf;
    for (const_chan_iter i=mChanList.begin() ; i != mChanList.end() ; i++) {
        sprintf (buf, "%-25s %10li  %08zx \n", i->getName(), 
		 i->getLast().getS(), (ptrdiff_t)i->refSeries());
	out << buf;
    }
    return out;
}

//======================================  Copy other frame structures
void 
FrWriter::copyDetectors(const FrameH& f) 
{
    mFrame->RefDetectSim()  = f.RefDetectSim();
    mFrame->RefDetectProc() = f.RefDetectProc();
}

void 
FrWriter::copyHistory(const FrameH& f) 
{
    mFrame->RefHistory() = f.RefHistory();
}

void 
FrWriter::copyAuxData(const FrameH& f) 
{
    mFrame->RefAuxData() = f.RefAuxData();
}

void 
FrWriter::addRawAdc(frame_ptr_type f, const char* Name) 
{
    frrawdata_pointer raw(f->GetRawData());
    if (!raw) return;
    FrRawData::const_firstAdc_iterator AdcIter;
    AdcIter = raw->RefFirstAdc().find(Name, raw->RefFirstAdc().begin());
    if (AdcIter == raw->RefFirstAdc().end()) return;
    addRawAdc(**AdcIter);
}

void 
FrWriter::addRawAdc(const FrAdcData& adc) 
{
    frrawdata_pointer raw(mFrame->GetRawData());
    if (!raw) return;
    raw->RefFirstAdc().append(adc);
}

//======================================  Set the FrVect compression mode
void 
FrWriter::setCompress(FrVectRef::compression_mode cmode) {
    mCompress = cmode;
}

//======================================  Set the checksum methods
void 
FrWriter::setChecksum(int ck_struct, int ck_file) {
    mChkSumFile   = ck_file;
    mChkSumStruct = ck_struct;
}

//======================================  Set the run number
void 
FrWriter::setRunID(int run) {
    mRun     = run;
    mFrameID = 0;
}

//======================================  Set the frame identifier
void 
FrWriter::setName(const std::string& name) {
    mName = name;
}

#if !defined(DMTOFFLINE) && LDAS_VERSION_NUMBER >= 109360
#include "framecpp/Common/IOStream.icc"
 template class FrameCPP::Common::FrameBuffer< oSMbuf >;
#endif
