/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef MULTILIST_HH
#define MULTILIST_HH

#include <list>
#include <string>
#include <vector>
#include <iosfwd>

/**  Maintain a list of input stream list files for the MultiStream class.
 */
class MultiList {
public:
    ///  Path list container type definition
    typedef std::vector< std::string > multi_list;

public:
    /**  Construct an empty list container.
      *  \brief default (empty) constructor.
      */
    MultiList(void);

    /**  Destroy the container and release allocated storage.
      *  \brief Destructor.
      */
    ~MultiList(void);

    /**  Add the content of a list file containing multiple file list paths 
      *  to the container.addMulti opens the specified file path and reads
      *  its contents. Leading and trailing blanks are removed from each 
      *  input line. If the remaining string starts with a '#', the line
      *  is ignored. Otherwise the string is saved as an entry in the 
      *  file list container. No processing or ordering of the paths is
      *  performed.
      *  \brief Add multiple lists.
      *  \param multinm List file path.
      */
    void addMulti(const char* multinm);

    /** Add a single list file path name to the container.
      *  \brief Add single lists.
      *  \param singlenm List file path.
      */
    void addSingle(const char* singlenm);

    /**  Test whether the container is empty.
      *  \brief Test empty.
      *  \return Trtue if the container is empty.
      */
    bool empty(void) const;

    /**  Number of entries in the container.
      *  \brief Number of entries.
      *  \return  Number of entries.
      */
    unsigned long size(void) const;

    /**  Get the ith path string.
      *  \brief Get a file path.
      *  \param i Path string index.
      *  \return Ith file path string.
      */
    std::string getList(int i);

    /**  Print the file paths in the container.
      *  \brief Print the file paths
      *  \param out STL output stream to receive the path list.
      *  \return Output stream reference.
      */
    std::ostream& print(std::ostream& out) const;
  
private:
    /**  mFile contains the input data list name(s)
      *  @memo Input data list name.
      */
    multi_list mList;
};

//======================================  Inline methods.
inline bool
MultiList::empty(void) const {
    return mList.empty();
}

inline unsigned long
MultiList::size(void) const {
    return mList.size();
}

inline std::string
MultiList::getList(int i) {
    return mList[i];
}

#endif // MULTILIST_HH
