/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//    Multiple streams of data access implementation
//
#include "MultiDacc.hh"
#include <stdio.h>
#include <iostream>

using namespace std;

//  dTcontig is the maximum acceptable discontinuity in nsec.
static const int dTcontig = 1;

//======================================  Default (Null) constructor.
MultiDacc::MultiDacc(void)
{
}

//======================================  Construct and connect to source.
MultiDacc::MultiDacc(const char* Source, const Time& STime)
{
    addMulti(Source);
}

//======================================  Destructor.
MultiDacc::~MultiDacc(void) {
    close();
    mChanList.clear();
    size_type N = mIn.size();
    for(size_type i=0; i < N; ++i) {
	delete mIn[i];
    }
}

//======================================  Add multiple frame sources.
void
MultiDacc::addMulti(const string& name) {
    size_t n0 = mList.size();
    mList.addMulti(name.c_str());
    for(size_t i=n0; i<mList.size(); i++) {
	Dacc* newDacc = new Dacc();
	newDacc->addPathList(mList.getList(i).c_str());
	mIn.push_back(newDacc);
    }
}

//======================================  Add a single frame source.
void
MultiDacc::addSingle(const string& name) {
    mList.addSingle(name.c_str());
    Dacc* newDacc = new Dacc();
    if (name[0] == '=') newDacc->addPath(name.substr(1));
    else                newDacc->addPathList(name);
    mIn.push_back(newDacc);
}

//======================================  Close all frame readers
void
MultiDacc::close(void) {
    size_type N = mIn.size();
    for(size_type i=0; i < N; ++i) {
	mIn[i]->close();
    }
}

//======================================  Open a frame file on all streams
int
MultiDacc::open() {
    size_type N = mIn.size();
    for(size_type i=0; i < N; ++i) {
	int j=mIn[i]->open();
	if(j!=0) return j;
    }
    return 0;
}

//======================================  Set the buffer count on all streams
void
MultiDacc::setBuffer(int nBuf) {
    size_type N = mIn.size();
    for(size_type i=0; i < N; ++i) {
	mIn[i]->setBuffer(nBuf);
    }
}

//======================================  Set the debug level on all streams
void
MultiDacc::setDebug(int nDbg) {
    size_type N = mIn.size();
    for(size_type i=0; i < N; ++i) {
	mIn[i]->setDebug(nDbg);
    }
}

//======================================  Set the ignore flag on all streams
void
MultiDacc::setIgnoreMissingChannel(bool yn) {
    size_type N = mIn.size();
    for(size_type i=0; i < N; ++i) {
	mIn[i]->setIgnoreMissingChannel(yn);
    }
}

//======================================  Set the nowait flag on all streams
void
MultiDacc::setNoWait(bool now) {
    size_type N = mIn.size();
    for(size_type i=0; i < N; ++i) {
	mIn[i]->setNoWait(now);
    }
}

//======================================  Set the stride on all streams
void
MultiDacc::setStride(Interval Dt) {
    size_type N = mIn.size();
    for (size_type i=0; i < N; ++i) {
	mIn[i]->setStride(Dt);
    }
}

//======================================  Get debug level
int
MultiDacc::getDebug(void) {
    if (mIn.empty()) return 0;
    return mIn[0]->getDebug();
}

//======================================  Get number of frames read
long
MultiDacc::getTotalFrames(void) const {
    if (mIn.empty()) return 0;
    return mIn[0]->getTotalFrames();
}

//======================================  Get current time (or Time(0))
Time
MultiDacc::getCurrentTime(void) const {
    if (mIn.empty()) return Time(0);
    return mIn[0]->getCurrentTime();
}

//======================================  Get fill time (or Time(0))
Time
MultiDacc::getFillTime(void) const {
    if (mIn.empty()) return Time(0);
    return mIn[0]->getFillTime();
}

//======================================  Get stride (or 0)
Interval
MultiDacc::getStride(void) const {
    if(mIn.empty()) return 0.0;
    return mIn[0]->getStride();
}

//======================================  Add a channel to the request list.
void 
MultiDacc::addChannel(const char* Name, int id, int Decimate, TSeries **TSptr) {
    chan_iter i=findChannel(Name);
    if (i != mChanList.end()) {
	if (getDebug()) cout << "Existing channel " << Name 
			     << " entry replaced." << endl;
	mChanList.erase(i);
    }
    ChannelIndex newchan(Name, id);
    mChanList.push_back(newchan);
    int index=getDaccIndex(Name);
    if(index!=-1) {
	mIn[index]->addChannel(Name, Decimate, TSptr);
    }
    else {
	if (getDebug()) cout << "Channel " << Name 
			     << " does not exist in data streams" << endl;
	chan_iter j=findChannel(Name);
	if (j != mChanList.end()) {
	    mChanList.erase(j);
	}
    }
}

//======================================  Add a channel to the request list.
void 
MultiDacc::addFSeries(const char* Name, int id, FSeries **FSptr) {
    chan_iter i=findChannel(Name);
    if (i != mChanList.end()) {
	if (getDebug()) cout << "Existing channel " << Name 
			     << " entry replaced." << endl;
	mChanList.erase(i);
    }
    ChannelIndex newchan(Name, id);
    mChanList.push_back(newchan);
    int index=getDaccIndex(Name);
    if(index!=-1) {
	mIn[index]->addFSeries(Name, FSptr);
    }
    else {
	if (getDebug()) cout << "Channel " << Name 
			     << " does not exist in data streams" << endl;
	chan_iter j=findChannel(Name);
	if (j != mChanList.end()) {
	    mChanList.erase(j);
	}
    }
}

//======================================  Add a channel to the request list.
void 
MultiDacc::addRaw(const char* Name, int id, int Decimate, TSeries **TSptr) {
    chan_iter i=findChannel(Name);
    if (i != mChanList.end()) {
	if (getDebug()) cout << "Existing channel " << Name 
			     << " entry replaced." << endl;
	mChanList.erase(i);
    }
    ChannelIndex newchan(Name, id);
    mChanList.push_back(newchan);
    int index=getDaccIndex(Name);
    if(index!=-1) {
	mIn[index]->addRaw(Name, Decimate, TSptr);
    }
    else {
	if (getDebug()) cout << "Channel " << Name 
			     << " does not exist in data streams" << endl;
	chan_iter j=findChannel(Name);
	if (j != mChanList.end()) {
	    mChanList.erase(j);
	}
    }
}

//======================================  Add a channel to the request list.
void 
MultiDacc::addProcessed(const char* Name, int id, int Decimate, TSeries **TSptr) {
    chan_iter i=findChannel(Name);
    if (i != mChanList.end()) {
	if (getDebug()) cout << "Existing channel " << Name 
			     << " entry replaced." << endl;
	mChanList.erase(i);
    }
    ChannelIndex newchan(Name, id);
    mChanList.push_back(newchan);
    int index=getDaccIndex(Name);
    if(index!=-1) {
	mIn[index]->addProcessed(Name, Decimate, TSptr);
    }
    else {
	if (getDebug()) cout << "Channel " << Name 
			     << " does not exist in data streams" << endl;
	chan_iter j=findChannel(Name);
	if (j != mChanList.end()) {
	    mChanList.erase(j);
	}
    }
}

//======================================  Add a channel to the request list.
void 
MultiDacc::addSimulated(const char* Name, int id, int Decimate, TSeries **TSptr) {
    chan_iter i=findChannel(Name);
    if (i != mChanList.end()) {
	if (getDebug()) cout << "Existing channel " << Name 
			     << " entry replaced." << endl;
	mChanList.erase(i);
    }
    ChannelIndex newchan(Name, id);
    mChanList.push_back(newchan);
    int index=getDaccIndex(Name);
    if(index!=-1) {
	mIn[index]->addSimulated(Name, Decimate, TSptr);
    }
    else {
	if (getDebug()) cout << "Channel " << Name 
			     << " does not exist in data streams" << endl;
	chan_iter j=findChannel(Name);
	if (j != mChanList.end()) {
	    mChanList.erase(j);
	}
    }
}

//======================================  See if channel was requested.
bool 
MultiDacc::isChannelRead(const char* Name) const {
    return findChannel(Name) != mChanList.end();
}

//======================================  List requested channels
ostream&
MultiDacc::list(ostream& out) const {
    char buf[2048];
    sprintf (buf, "Channel\n");
    out << buf;
    for (const_chan_iter i=mChanList.begin() ; i != mChanList.end() ; i++) {
	sprintf (buf, "%-25s \n", i->getName());
	out << buf;
    }
    return out;
}

//======================================  Stream with the specified frame type
int
MultiDacc::frame_type(const std::string& frameType) const {

   //--------------------------------  Find an input stream ID.
   int id = -1;
   size_t N = mIn.size();
   for (size_t i=0; i < N; ++i) {
      string framepath = mIn[i]->refList().first();
      string::size_type inx = framepath.rfind("/");
      if (inx != string::npos) framepath.erase(0, inx+1);
      inx = framepath.find("-");
      if (inx != string::npos) {
	 string::size_type inx2 = framepath.find("-", inx);
	 if (inx2 != string::npos) inx = inx2;
      }
      if (inx != string::npos) framepath.erase(inx);
      if (framepath == frameType) id = i;
   }
   return id;
}

//======================================  Find channel in request list.
MultiDacc::const_chan_iter 
MultiDacc::findChannel(const char* Name) const {
    for (const_chan_iter i=mChanList.begin() ; i != mChanList.end() ; i++) {
	if (i->EqName(Name)) return i;
    }
    return mChanList.end();
}

MultiDacc::chan_iter 
MultiDacc::findChannel(const char* Name) {
    for (chan_iter i=mChanList.begin() ; i != mChanList.end() ; i++) {
	if (i->EqName(Name)) return i;
    }
    return mChanList.end();
}

//======================================  Remove channel from request list.
void 
MultiDacc::rmChannel(const char* Name) {
    int index=getDaccIndex(Name);
    if (index != -1) {
	mIn[index]->rmChannel(Name);
    }
    chan_iter i=findChannel(Name);
    if (i != mChanList.end()) {
	mChanList.erase(i);
    }
}

//======================================  Synchronize the data streams
int 
MultiDacc::synch(void) {

    //----------------------------------  See how they stack up
    Time tEarly(0), tLate(0);
    if (getDebug()) cerr << "Start Synch:";
    size_type N = mIn.size();
    for(size_type i=0; i<N; ++i) {
	int rc = mIn[i]->synch();
	if (rc != 0) return rc;
	Time ti = mIn[i]->getCurrentTime();
	if (getDebug()) cerr << " t[" << i << "]=" << ti;
	if (ti > tLate) tLate = ti;
	if (!i || ti < tEarly) tEarly =ti;
    }
    if (getDebug()) cerr << endl;


    //----------------------------------  Seek the latest.
    while (! Almost(tEarly, tLate) ) {
	Time tSeek = tLate;
	if (getDebug()) cerr << "Synch to: " << tSeek << endl;
	for(size_type i=0; i<N; ++i) {
	    int rc = mIn[i]->seek(tSeek);
	    if (rc != 0) return rc;
	    Time ti = mIn[i]->getCurrentTime();
	    if (getDebug()) cerr << " t[" << i << "]=" << ti;
	    if (ti > tLate) tLate = ti;
	    if (!i || ti < tEarly) tEarly =ti;
	}
	if (getDebug()) cerr << endl;
    }
    return 0;
}

//======================================  Fill requested series containers.
//    Error codes:
//      0  Successful completion
//     -1  Frame start not contiguous to previous data
//     -2  Sample rate isn't compatible with previous data.
//     -3  Requested data not found in current frame 
//     -4  Error reading frame
//     -5  Frame data are not self-consistent.
//     -6  TSeries is not allocated.
//     -7  Unsupported data type
//     -8  Interrupter by signal.
int 
MultiDacc::fillData(Interval Stride, bool start) {
    int rc = 0;
    size_type N = mIn.size();
    for (size_type i=0; i<N; i++) {
	rc = mIn[i]->fillData(Stride,start);
    }
    return rc;
}

//======================================  Fill data from a single frame
void
MultiDacc::zeroChans(Interval dT) {
    size_type N = mIn.size();
    for (size_type i=0; i < N; ++i) {
	mIn[i]->zeroChans(dT);
    }
}

//======================================  Fill data from a single frame
int
MultiDacc::fillChans(Interval Offset, Interval dT) {
    int rc = 0;
    size_type N = mIn.size();
    for (size_type i=0; i<N; ++i) {
	rc = mIn[i]->fillChans(Offset, dT);
    }
    return rc;
}

int
MultiDacc::getDaccIndex(const char* name) {
    chan_iter i=findChannel(name); 
    if (i != mChanList.end()) {
	int index=i->getIndex();
	if(index==-1) {
	    // find a way to detect where is the channel automatically
	    // not so easy, may do it later
	    /*
	      int j;
	      for(j=0;j<(int)mIn.size();j++) {
	      cout<<"entering loop"<<endl;
	      const FrameCPP::FrAdcData* abc;
	      cout<<"before find"<<endl;
	      abc=mIn[j]->nextAdc(name);
	      cout<<"after find"<<endl;
	      if(abc) {
	      cout<<"before set"<<endl;
	      i->setIndex(j);
	      cout<<"after set"<<endl;
	      return j;
	      }
	      const FrameCPP::FrProcData* proc;
	      proc = mIn[j]->nextProc(name);
	      if(proc) {
	      i->setIndex(j);
	      return j;
	      }
	      else {
	      const FrameCPP::FrSimData* sim;
	      sim = mIn[j]->nextSim(name);
	      if(sim) {
	      i->setIndex(j);
	      return j;
	      }
	      }
	      }
	      if(j==(int)mIn.size())
	      return -1;
	    */
	}
	else
	    return index;
    }
    return -1;
}

//======================================  Flush the specified data
int 
MultiDacc::flush(Interval Stride) {
    int rc = 0;
    size_type N = mIn.size();
    for (size_type i=0; i < N; ++i) {
	rc = mIn[i]->flush(Stride);
    }
    return rc;
}

//======================================  Open a file for input.
int
MultiDacc::seek(Time STime) {
    int rc = 0;
    size_type N = mIn.size();
    for (size_type i=0; i < N; ++i) {
	rc = mIn[i]->seek(STime);
    }
    return rc;
}


//======================================  Get a reference to the named Channel
TSeries* 
MultiDacc::refData(const char *name) {
    chan_iter i=findChannel(name);
    if (i != mChanList.end()) {
	int index=getDaccIndex(name);
	if(index!=-1)
	    return mIn[index]->refData(name);
    }
    return (TSeries*) 0;
}

FSeries* 
MultiDacc::refFData(const char *name) {
    chan_iter i=findChannel(name);
    if (i != mChanList.end()) {
	int index=getDaccIndex(name);
	if(index!=-1)
	    return mIn[index]->refFData(name);
    }
    return (FSeries*) 0;
}
