/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef FRVECTREF_HH
#define FRVECTREF_HH

#include "FramePointerTypes.hh"

class DVector;


/**  Class %FrVectRef contains a wrapped reference to an FrVect frame 
  *  structure and supplies various methods to access the data.
  *  @memo   FrVect reference class.
  *  @author John Zweizig
  *  @version 1.2; Last Modified: August 18, 2008
  */
class FrVectRef {
public:

    //---------------------------------  Lib-independent types

    /**  Mnemonic name for compression modes specified in the frame 
      *  specification.
      *  @memo Compression modes.
      */
    enum compression_mode {
	///  Uncompressed data
	kUncompressed=0,

	///  Data compressed by gzip.
	kGZip=1,

	///  Difference followed by gzip
	kGZipDifferential=3,

	///  Frame zero-suppression zlgorithm for 2-byte data
	kZeroSuppress2=5,

	///  Frame zero-suppression zlgorithm for 4-byte data
	kZeroSuppress4=8,

	///  Note this is a framecpp compression mode hack. Data are 
	///  compressed using an appropriate zero suppression mode if
	///  available. Otherwise zgip is used.
	kZeroSuppOrGZip=10
    };

    /**  External data type used for length values.
      */
    typedef unsigned long    size_type;

//-------------------------------------  Lib-dependent types
public:
    /**  FrVect data type used by the I/O library.
      */
    typedef FrameCPP::FrVect vect_type;

    /**  FrVect pointer type used by the I/O library.
      */
    typedef frvect_pointer vect_ptr_type;

//-------------------------------------  Class methods
public:
    /**  Null referecnce constructor.
      *  @memo Null constructor.
      */
    FrVectRef(void);

    /**  Construct a reference to the specified FrVect.
      *  \brief Construct reference to the specified FrVect.
      *  \param x Pointer to FrVect.
      */
    FrVectRef(const vect_type* x);

    /**  Construct a reference to the specified FrVect.
      *  \brief Construct reference to the specified FrVect.
      *  \param x Shared pointer to vector.
      */
    FrVectRef(vect_ptr_type x);

    /**  Construct a 1-dimensional FrVect and fill it with the specified data.
      *  \brief Construct an FrVect with the specified data.
      *  \param v   DVector containing data
      *  \param x0  Zeroeth x-bin position (time or frequency).
      *  \param dx  X-bin position increment.
      *  \param unit X units.
      */
    FrVectRef(const DVector& v, double x0=0.0, double dx=1.0, 
	      const std::string& unit="ct");

    /**  Destroy the FrVectRef.
      *  @memo Destructor.
      */
    ~FrVectRef(void);

    /**  Clear the reference pointer. Delete a referenced FrVect structure 
      *  if it is owned by the FrVectRef.
      *  @memo Clear the reference.
      */
    void clear(void);

    /**  Returns true if the data vector is empty or undefined.
      *  @memo test empty.
      *  @return true if no data in the FrVect.
      */
    bool empty(void) const;

    /**  Construct a DVector contining the FrVect data.
      *  @memo Get a Dvector with the FrVect data.
      *  @return Pointer to a DVector containing the data.
      */
    DVector*  getDVector(void) const;

    /**  Get a data string containing the specified coordinate range from 
      *  an FrVector.
      *  @memo Get data substring
      *  @param dim Dimension always 0?
      *  @param x0  Start value of dimension coorsinate.
      *  @param dx  Length of data in dimension coordinate.
      *  @return Pointer to a DVector containing the data.
      */
    DVector*  getDVector(size_type dim, double x0, double dx) const;

    /**  Get a data string containing the specified data range from an FrVector.
      *  @memo Get data substring
      *  @param inx0 Start index value.
      *  @param dinx Length of string.
      *  @return Pointer to a DVector containing the data.
      */
    DVector*  getDVector(size_type inx0, size_type dinx) const;

    /**  Get the starting coordinate value for the ith dimension.
      *  @memo Get ith dimenstion starting coordinate.
      *  @param i Dimension number.
      *  @return Specified start coordinate.
      */
    double    getDimX0(size_type i) const;

    /**  Get the maximum coordinate value for the ith dimension.
      *  @memo Get ith dimension maximum coordinate.
      *  @param i Dimension number.
      *  @return Specified maximum coordinate.
      */
    double    getDimXmax(size_type i) const;

    /**  Get the coordinate increment value for the ith dimension.
      *  @memo Get ith dimension coordinate increment.
      *  @param i Dimension number.
      *  @return Specified coordinate increment.
      */
    double    getDimDx(size_type i) const;

    /**  Get the number of bins in the ith dimension.
      *  @memo Get number of bins in the ith dimension.
      *  @param i Dimension number.
      *  @return Number of bins in the specified dimension.
      */
    size_type getDimNx(size_type i) const;

    /**  Get the index into the data of the specified value in the specified
      *  dimension.
      *  \brief Get index for specified x-bin.
      *  \param dim Dimension number
      *  \param x0  Value in the specified dimension.
      *  \return  Index of specified data into vector.
      */
    size_type getIndex(size_type dim, double x0) const;

    /**  Get the number of bits in a word for the type of data contained 
      *  in the FrVect.
      *  @memo Get data work lengt in bits.
      *  @return Number of bit in a data word.
      */
    size_type getNBits(void) const;

    /**  get the y-units field
     */
    const std::string& getUnits(void) const;

    /**  Compress the FrVect data.
      *  @memo Compress the data.
      *  @param cmode Compression algorithm to be used.
      */
    void      compress(compression_mode cmode);

    /**  Release ownership of the FrVect structure from this FrVectRef.
      *  If the vector is not owned, a new FrVect structure is constructed.
      *  The reference is not cleared although there is no guarentee that 
      *  the reference will remain valid.
      *  @memo Release the FrVect.
      *  @return Pointer to the FrVect structure.
      */
    vect_ptr_type release(void);

    /**  Make a copy of the referenced FrVect structure. Keep the original 
      *  structure.
      *  @memo   Clone the referenced structure.
      *  @return Pointer to the cloned FrVect.
      */
    vect_ptr_type replicate(void) const;

    /**  Set the name field of the referenced vector to the argument string.
      *  @memo Set the vector name.
      *  @param name Vector name.
      */
    void setName(const std::string& name);

    /**  Set the y-units field of the vector.
      *  @memo Set the vector units.
      *  @param unit Vector Y value units.
      */
    void setUnits(const std::string& unit);

    /**  Construct an FrVect containing the the specified data and set the 
      *  reference. The currently referenced FrVect is released and deleted
      *  if owned by the reference.
      *  @memo Set the reference to a structure containing the specified data.
      *  @param v    DVector containing the data for the FrVect.
      *  @param x0   Start corrdinate.
      *  @param dx   Coordinate increment.
      *  @param unit Coordinate units.
      */
    void setVect(const DVector& v, double x0=0.0, double dx=1.0, 
		 const std::string& unit="ct");

    /**  Replace the reference with the specified vector. The ownership
      *  flag is set as indicated by the owner argument. If the FrVectRef
      *  currently points to a owned FrVect structure, the FrVect is 
      *  deleted.
      *  @memo Set reference to the specified vector.
      *  @param v     Pointer to the vector to be referenced.
      *  @param owner If true, the specified vector will be deleted when
      *               the FrVectRef is deleted.
      */
    void setVect(vect_ptr_type v, bool owner=false);

    /**  Number of data elements contined by the FrVect. Size returns 0 if
      *  the reference is null.
      *  @memo Number of data elements.
      *  @return Number of data elements.
      */
    size_type size(void) const;

    /**  Returns true if the reference is undefined.
      *  @memo Test undefined.
      *  @return true if no FrVect is referenced.
      */
    bool null(void) const;

private:
  bool       mOwner;
  vect_ptr_type mVect;
};

//======================================  Inline methods.
inline FrVectRef::vect_ptr_type
FrVectRef::release(void) {
    if (!mOwner) return replicate();
    mOwner = false;
    return mVect;
}

inline void
FrVectRef::clear(void) {
#ifndef FCPP_SHARED_PTRS
    setVect(0);
#else
    mVect.reset();
#endif
}

inline bool
FrVectRef::null(void) const {
    return !mVect;
}

inline bool
FrVectRef::empty(void) const {
    return !size();
}

#endif // FRVECTREF_HH
