/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef TRIGSTREAM_HH
#define TRIGSTREAM_HH

#include "Time.hh"
#include "TrigBase.hh"
#include "Segment.hh"
#include "Process.hh"
#include "LdasDBWriter.hh"
#include "S6SegWriter.hh"
#include <string>
#include <list>

namespace html {
    class document;
}

#define defTableCount 100

std::string getUnique(const Time& t);

/**  The trigger stream class is used by the trigger manager class to output
  *  trigger or segment table rows to the specified datatbase. The trigger 
  *  stream operates in two modes. 
  *
  *  In count more, the stream will accumulate triggers until either 
  *  - the maximum number of triggers or segments has been reached.
  *  - the stream has been idle (no new segments or trigger are added) since
  *    the last row was added.
  *  - the maximum delay was reached.
  *
  *  In timer mode, a file is written periodically with the length and 
  *  alignment of the delay specified.
  *
  *  @memo Trigger stream output class
  *  @author J. Zweizig
  *  @version 1.0; Modified July 6, 2005
  */
class TrigStream {
public:
    typedef std::list<trig::TrigBase> TriggerDB;
    typedef TriggerDB::iterator trig_iter;

    typedef std::list<trig::Segment> SegmentDB;
    typedef SegmentDB::iterator segm_iter;

    typedef unsigned long mask_type;
    typedef unsigned long count_type;

    struct param {
	enum time_mode {
	    kCount,
	    kTimer
	};
	param(void);
	Time align(const Time& t) const;
	void setMode(time_mode m, Interval i, Interval d);
	time_mode   _Mode;
	Interval    _Inactive;
	Interval    _MaxDelay;
	count_type  _TableCount;
	std::string _OutDir;
	std::string _Insert;
    };

    struct stream_stats {
	stream_stats(void);
	void count_seg(void);
	void count_trig(void);
	void count_write(const std::string& file, const Time& end);
	count_type  errTot;
	count_type  fileTot;
	count_type  segTot;
	count_type  trigTot;
	std::string lastFile;
	Time        lastUpdate;
	Time        lastWrite;
    };

public:
    /** Construct a TrigStream.
     */
    TrigStream(const std::string& name, mask_type mask);

    /**  Destroy a TrigStream.
     */
    virtual ~TrigStream(void);

    /**  Add a process to the process database.
     */
    virtual void addProcess(const Process& pdb);

    /**  Add a segment to the segment database.
     */
    virtual bool addSegment(const trig::Segment& seg, mask_type dest);

    /** Add a trigger to the trigger database.
     */
    virtual bool addTrigger(const trig::TrigBase& tr, mask_type dest);

    /**  Clone a stream.
     */
    virtual TrigStream* clone(void) const=0;

    /**  Create directories in a path
     */
    virtual bool createPath(const std::string& path, int depth=1) const;

    /**  Test whether there are any segments or triggers before the cutoff 
      *  time.
      */
    virtual bool empty(const Time& t) const;

    /**  Get the debug printout level.
     */
    int getDebug(void) const;

    /**  Get the TrigStream name.
     */
    virtual const char* getName(void) const;

    /**  Get the number of triggers available for writing at the specified time.
     */
    virtual count_type getNTrigs(const Time& t) const;

    /**  Get the number of segments available for writing at the specified time.
     */
    virtual count_type getNSegs(const Time& t) const;

    /**  Generate the output file path.
     */
    virtual std::string getPath(const Time& start, const Time& end) const;

    /**  Get the number of seconds before the stream will ready for writing.
      */
    Interval getTimeOut(const Time& now) const;

    /**  Get the stream type.
      */
    virtual const char* getType(void) const=0;

    /**  Perform post-processing on the trigger or segment output file. By 
      *  default the post processing resolves any environment variables 
      *  in the _Insert parameter, and executes the resulting command 
      *  as a detached process. If the _Insert parameter string is empty,
      *  no action is taken.
      *  \brief Output file post-processing.
      */
    void postProcess(const std::string& file) const;

    /**  Get a reference to the parameter structure.
      */
    const param& refParams(void) const;

    /**  Get a constant reference to the Process entry database.
      */
    const ProcessDB& refProcDB(void) const;

    /**  Get a constant reference to the stream statistics structure.
     */
    const stream_stats& refStats(void) const;

    /**  Remove the specified process from the process database.
      */
    void removeProc(const std::string& proc);

    /** write segment sytatistics to an html document
     */
    void segStats(html::document& doc) const;

    /**  Set the debug printout level.
     */
    void setDebug(int lvl);

    /**  Set the destination mask
      */
    void setDestMsk(mask_type mask);

    /**  Set the exclusion mask
      */
    void setExcludeMsk(mask_type mask);

    /** Set the parameters.
     */
    void setParam(const param& par);

    /**  Test whether the specified segment is to be written
      *  to this stream.
      */
    bool testWrite(const trig::Segment& seg, mask_type dest) const;

    /**  Test whether the specified trigger is to be written
      *  to this stream.
      */
    bool testWrite(const trig::TrigBase& tr, mask_type dest) const;

    /**  Test whether there are segments/triggers to be written out at the
      *  specified time.
      */
    bool writeNow(bool force) const;

    /**  Write all appropriate segments and perform post- processing.
      */
    virtual void WriteDB(bool force);

    /**  Write the appropriate segments.
      *  @param force Force the function to write out data.
      *  @param path  File name template
      *  @param wrt   TrigWriter to write stream data.
      *  @return File was written successfully.
      */
    virtual bool write(bool force, trig::TrigWriter& wrt);

    /**  Purge the process database.
      */
    virtual void PurgeProcDB(void);

    /**  Set the last update time if the segment and trigger databases 
      *  are empty.
      */
    void tick(const Time& now);

protected:
    void setTimer(const Time& t);

private:
    std::string  mName;
    int          mDebug;
    mask_type    mDestMsk;
    mask_type    mExcludeMsk;

protected:
    param        mParam;
    ProcessDB    mProcDB;
    stream_stats mStats;
};

inline int 
TrigStream::getDebug(void) const {
    return mDebug;
}

inline const char*
TrigStream::getName(void) const {
    return mName.c_str();
}

inline const TrigStream::param& 
TrigStream::refParams(void) const {
    return mParam;
}

inline const ProcessDB& 
TrigStream::refProcDB(void) const {
    return mProcDB;
}

inline const TrigStream::stream_stats&
TrigStream::refStats(void) const {
    return mStats;
}

//======================================  Ldas database stream
/**  @name LDAS stream
  *  The LDAS stream class is used by the trigger manager class to output
  *  trigger or segment table rows to an Ldas database,
  *  @memo LDAS stream output class
  *  @author J. Zweizig
  *  @version 1.0; Modified July 6, 2005
  */
class LdasStream : public TrigStream {
public:
    typedef TrigStream::count_type count_type;
    LdasStream(const std::string& name, mask_type mask);
    ~LdasStream(void);
    virtual LdasStream* clone(void) const;
    bool addSegment(const trig::Segment& seg, mask_type dest);
    bool addTrigger(const trig::TrigBase& tr, mask_type dest);
    count_type getNTrigs(const Time& t) const;
    count_type getNSegs(const Time& t)  const;
    const char* getType(void) const {return "LDAS";}

    /**  Write all appropriate segments and perform post- processing.
      */
    virtual void WriteDB(bool force);
private:
    trig::LdasDBWriter mWriter;
};

//======================================  Segment database stream
/**  @name Segment stream
  *  The Segment stream class is used by the trigger manager class to output
  *  segment table rows to a S6 Segment database,
  *  @memo Segment stream output class
  *  @author J. Zweizig
  *  @version 1.1; Modified January 26, 2009
  */
class SegDBStream : public TrigStream {
public:
    typedef TrigStream::count_type count_type;
    SegDBStream(const std::string& name, mask_type mask);
    ~SegDBStream(void);
    virtual SegDBStream* clone(void) const;
    bool addSegment(const trig::Segment& seg, mask_type dest);
    count_type getNSegs(const Time& t) const;
    const char* getType(void) const {return "Segment";}

    /** write segment sytatistics to an html document
     */
    void segStats(html::document& doc) const;

    /**  Write all appropriate segments and perform post- processing.
      */
    virtual void WriteDB(bool force);
private:
    trig::S6SegWriter mWriter;
};

//======================================  Epics stream
/**  @name Epics stream
  *  The Epics stream class is used by the trigger manager class to output
  *  triggers to an Epics data server,
  *  @memo Epics stream output class
  *  @author J. Zweizig
  *  @version 1.0; Modified July 6, 2005
  */
class EpxStream : public TrigStream {
public:
    using TrigStream::addTrigger;
    EpxStream(const std::string& name, mask_type mask);
    ~EpxStream(void);
    virtual EpxStream* clone(void) const;
    const char* getType(void) const {return "Epics";}
    bool addTrigger(const trig::TrigBase& trg, mask_type dest);
};

#endif  //  TRIGSTREAM_HH
