
#include "TrigCtlTable.hh"
#include "ParseLine.hh"
#include "Segment.hh"
#include "TrigBase.hh"
#include "TrigProc.hh"
#include "html/table.hh"
#include <cstdio>
#include <cstdlib>
#include "PConfig.h"
#include <sstream>

using namespace std;

//===========================================================================
//
//       Trigger control entry implementation
//
//===========================================================================

//======================================  Default constructor
TrigCtlEntry::TrigCtlEntry(void) 
  : mCount(0), mEpoch(0), mDestMask(trig::d_all), mMaxPrty(5), mThreshold(0.1)
{
}

//======================================  Data constructor
TrigCtlEntry::TrigCtlEntry(const string& program, const string& version, 
			   const string& trigid,  const string& subid, 
			   mask_t dest, int maxprio) 
  : mProgram(program), mVersion(version), mTrigID(trigid), mSubID(subid), 
    mCount(0), mEpoch(0), mDestMask(dest), mMaxPrty(maxprio), mThreshold(0.1)
{
}

//======================================  Destructor
TrigCtlEntry::~TrigCtlEntry(void) {
}

//======================================  compare key
bool 
TrigCtlEntry::testKey(const string& program, const string& version, 
		      const string& trigid,  const string& subid) const 
{
    if (program != mProgram && mProgram != "*") return false;
    if (version != mVersion && mVersion != "*") return false;
    if (trigid  != mTrigID  && mTrigID  != "*") return false;
    if (subid   != mSubID   && mSubID   != "*") return false;
    return true;
}
//======================================  compare key
bool 
TrigCtlEntry::testKey(const TrigCtlEntry& t) const 
{
    if (mProgram != t.mProgram) return false;
    if (mVersion != t.mVersion) return false;
    if (mTrigID  != t.mTrigID ) return false;
    if (mSubID   != t.mSubID  ) return false;
    return true;
}


//======================================  compare key
void 
TrigCtlEntry::setDestMask(mask_t dest) {
    mDestMask = dest;
}

//======================================  compare key
void 
TrigCtlEntry::setMaxPrty(int pri) {
    mMaxPrty = pri;
}

//======================================  compare key
void 
TrigCtlEntry::setNotify(const std::string& email) {
    mNotify = email;
}

//======================================  compare key
void 
TrigCtlEntry::setThreshold(double thresh) {
    mThreshold = thresh;
}

//===========================================================================
//
//       Trigger control table implementation
//
//===========================================================================

//======================================  Default constructor
TrigCtlTable::TrigCtlTable(void) {
}

//======================================  Data constructor
TrigCtlTable::TrigCtlTable(const char* file) {
    readFile(file);
}

//======================================  Destructor
TrigCtlTable::~TrigCtlTable(void) {
    clear();
}

//======================================  Find the appropriate entry
const TrigCtlEntry* 
TrigCtlTable::find(const trig::TrigProc& p, const trig::TrigBase& t) const {
    return find(p.getName(), p.getVersion(), t.getID(), t.getSubID());
}

//======================================  Find the appropriate entry
const TrigCtlEntry* 
TrigCtlTable::find(const trig::TrigProc& p, const trig::Segment& t) const {
    stringstream vsn;
    vsn << t.getVersion() << ends;
    return find(p.getName(), p.getVersion(), t.getGroup(), vsn.str());
}

//======================================  Find the appropriate entry
const TrigCtlEntry*
TrigCtlTable::find(const string& program, const string& version, 
		   const string& trigID,  const string& subID) const {
    for (const_iter i=mList.begin() ; i != mList.end() ; i++) {
        if (i->testKey(program, version, trigID, subID)) return &(*i);
    }
    return 0;
}

//======================================  Find the appropriate entry
TrigCtlTable::list_iter
TrigCtlTable::findAfter(const TrigCtlEntry& t, const list_iter& p) {
    for (list_iter i=p ; i != mList.end() ; i++) {
        if (i->testKey(t)) return i;
    }
    return mList.end();
}

//======================================  Reset trigger statistics
std::ostream& 
TrigCtlEntry::display(std::ostream& out) const {
  out << "program: " << mProgram << " version: " << mVersion 
      << " trigger-id: " << mTrigID << " sub-id: "<< mSubID 
      << " dest: " << mDestMask << endl;
  return out;
}

//======================================  Reset trigger statistics
void 
TrigCtlEntry::epoch(void) {
    mEpoch = 0;
}

//======================================  Reset trigger statistics
void 
TrigCtlEntry::reset(void) {
    mCount = 0;
    mEpoch = 0;
}

//======================================  Read a configuration file
void 
TrigCtlTable::readFile(const string& file, bool reset) {

    //----------------------------------  Open the configuration file
    cout << "Reading trigger file: " << file << endl;
    ParseLine pl(file.c_str());
    if (!pl.isOpen()) {
        cout << "TrigCtlTable: Error opening configuration file" << endl;
	return;
    }
    pl.setLog(cout);

    //----------------------------------  Optional reset, set insert pointer 
    if (reset) clear();
    list_iter pInsert = mList.begin();

    //----------------------------------  Loop over lines
    for (int N=pl.getLine() ; N>=0 ; N=pl.getLine()) {
        if (N < 4) {
	    cout << "Too few arguments on line" << endl;
	    continue;
	}
	string name = pl[0];
	string vers = pl[1];
	string trig = pl[2];
	string sbID = pl[3];
	for (const_iter i=mList.begin() ; i != pInsert ; i++) {
	    if (i->testKey(name, vers, trig, sbID)) {
	        cerr << "Warning: Trigger entry for program= " << name
		     << " version= " << vers << " Trigger= " << trig
		     << " sub-ID= " << sbID << " is hidden by previous entry"
		     <<  endl;
	    }
	}
	TrigCtlEntry te(name,vers, trig, sbID);
	bool syntax(false);
	for (int i=4 ; i<N && !syntax; i++) {
	    string argi = pl[i];
	    char* p;
	    if (argi == "-dmask") {
	        TrigCtlEntry::mask_t dmask = strtol(pl[++i], &p, 0);
		if (*p) syntax = true;
		else    te.setDestMask(dmask);
	    } else if (argi == "-maxprty") {
	        int pmax = strtol(pl[++i], &p, 0);
		if (*p) syntax = true;
		else    te.setMaxPrty(pmax);
	    } else if (argi == "-notify") {
	        te.setNotify(pl[++i]);
	    } else if (argi == "-threshold") {
	        double thresh = strtod(pl[++i], &p);
		if (*p) syntax = true;
		else    te.setThreshold(thresh);
	    } else {
	        syntax = true;
	    }
	}

	if (syntax) {
	    cout << "Syntax error on line." << endl;
	    continue;
	}

	list_iter pEnt = findAfter(te, pInsert);
	if (pEnt != mList.end()) {
	    pEnt->setDestMask(te.getDestMask());
	    pEnt->setMaxPrty(te.getMaxPrty());
	    pEnt->setThreshold(te.getThreshold());
	    pEnt->setNotify(te.getNotify());
	    if (pEnt == pInsert) {
	        pInsert++;
	    } else {
	        mList.insert(pInsert, *pEnt);
		mList.erase(pEnt);
	    }
	} else {
	    mList.insert(pInsert, te);
	}
    }
    mList.erase(pInsert, mList.end());
}

//======================================  Clear the configuration table
void 
TrigCtlTable::clear(void) {
    mList.clear();
}

//======================================  Reset the global counts
void 
TrigCtlTable::reset(void) {
    for (list_iter i=mList.begin() ; i!= mList.end() ; i++) i->reset();
    mReset = Now();
    mEpoch = mReset;
}

//======================================  Reset the instantaneous counts
void 
TrigCtlTable::epoch(void) {
    for (list_iter i=mList.begin() ; i!= mList.end() ; i++) i->epoch();
    mEpoch = Now();
}

//======================================  Write current table status
void 
TrigCtlTable::writeStats(ostream& out) const {
    double dT(Now() - mReset);
    if (dT <= 0.0) dT = 1.0;
    out << "Program   Version Trigger    SubID        Number  Rate" << endl;
    for (const_iter i=mList.begin() ; i!= mList.end() ; i++) {
        char buf[2048];
        sprintf (buf, "%-9s %-7s %-10s %-12s %6i %6.3f\n",
		 i->getProgram(), i->getVersion(), i->getTrigger(),
		 i->getSubID(), i->getCount(), double(i->getCount())/dT);
        out << buf;
    }
}

//======================================  Write current table status
html::table
TrigCtlTable::makeHtmlTable(void) const {
    long   totCount(0);
    double totRate(0.0), totEpoch(0.0);


    //----------------------------------  Initialize table
    html::table tab("Trigger Statistics Table");
    tab.addColumn("Program");
    tab.addColumn("Version");
    tab.addColumn("Trigger");
    tab.addColumn("SubID");
    tab.addColumn("Enable");
    tab.addColumn("Number");
    tab.addColumn("Rate");
    tab.addColumn("Recent");
    
    //-----------------------------------  Add rows.
    double dT(Now() - mReset);
    if (dT <= 0.0) dT = 1.0;
    double dTE(Now() - mEpoch);
    if (dTE <= 0.0) dTE = 1.0;
    for (const_iter i=mList.begin() ; i != mList.end() ; i++) {
        int j = tab.addRow();
	tab.insertData(j, 0, i->getProgram());
	tab.insertData(j, 1, i->getVersion());
	tab.insertData(j, 2, i->getTrigger());
	tab.insertData(j, 3, i->getSubID());
	string de;
	if (i->maskDest(trig::d_metaDB)) de = "DB";
	if (i->maskDest(trig::d_alarm)) {
	    if (!de.empty()) de += ",";
	    de += "Epics";
	}
	if (i->maskDest(trig::d_testDB)) {
	    if (!de.empty()) de += ",";
	    de += "Test";
	}
	if (de.empty()) de = "-";
	tab.insertData(j, 4, de);

	totCount += i->getCount();
	tab.insertData(j, 5, double(i->getCount()));

	double Rate = double(i->getCount())/dT;
	totRate += Rate;
	tab.insertData(j, 6, Rate, -1, 4);

	Rate = double(i->getEpoch())/dTE;
	totEpoch += Rate;
	tab.insertData(j, 7, Rate, -1, 4);
    }
    int j = tab.addRow();
    tab.insertData(j, 0, "Totals");
    tab.insertData(j, 5, double(totCount));
    tab.insertData(j, 6, totRate,  -1, 4);
    tab.insertData(j, 7, totEpoch, -1, 4);
    return tab;
}
