/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "TrigTable.hh"
#include "TrigBase.hh"
#include "Segment.hh"
#include <fstream>
#include <stdlib.h>

using namespace xsil;
using namespace std;
using namespace trig;

//======================================  Define the trigger table.
TrigTable::TrigTable(bool alphaPID) 
  : MetaTable("gds_trigger"), binarydata(&binarydata_length, data),
    proc_id_length(nProcessKeyBytes), process_id(&proc_id_length, processID),
    altPID(alphaPID)
{
    if (altPID) defineColumn("process_id", &PIDstr, true);
    else        defineColumn("process_id", &process_id);
    defineColumn("name",              &name);
    defineColumn("subtype",           &subtype);
    defineColumn("ifo",               &ifo);
    defineColumn("start_time",        &start_time);
    defineColumn("start_time_ns",     &start_time_ns);
    defineColumn("duration",          &duration);
    defineColumn("priority",          &priority);
    defineColumn("disposition",       &disposition);
    defineColumn("size",              &size);
    defineColumn("significance",      &significance);
    defineColumn("frequency",         &frequency);
    defineColumn("bandwidth",         &bandwid);
    defineColumn("time_peak",         &peak_time);
    defineColumn("time_average",      &avg_time);
    defineColumn("time_sigma",        &time_sigma);
    defineColumn("freq_peak",         &freq_peak);
    defineColumn("freq_average",      &freq_avg);
    defineColumn("freq_sigma",        &freq_sigma);
    defineColumn("noise_power",       &noise_power);
    defineColumn("signal_power",      &signal_power);
    defineColumn("pixel_count",       &pixel_count);
    defineColumn("confidence",        &confidence);
    defineColumn("binarydata",        &binarydata);
    defineColumn("binarydata_length", &binarydata_length);
    defineColumn("event_id",          this);
}

//======================================  Delete the trigger table
TrigTable::~TrigTable(void) {
    MetaTable::clear();
}

//======================================  Fill the trigger table.
void
TrigTable::addRow(const trig::TrigBase& t) {
    start_time    = t.getTime().getS();
    start_time_ns = t.getTime().getN();
    priority      = 0;
    disposition   = t.getDisposition();
    if (altPID) PIDstr = t.getProcess();
    else        ProcTable::packKey(nProcessKeyBytes, t.getProcess(), processID);
    name          = t.getID();
    subtype       = t.getSubID();
    ifo           = t.getIFOs();
    duration      = t.getDt();
    size          = t.getIntensity();
    significance  = t.getSignificance();
    frequency     = t.getFrequency();
    bandwid       = t.getBandwidth();
    peak_time     = t.getPeakOffset();
    avg_time      = t.getAvgOffset();
    time_sigma    = t.getTimeSigma();
    freq_peak     = t.getFreqPeak();
    freq_avg      = t.getFreqAvg();
    freq_sigma    = t.getFreqSigma();
    noise_power   = t.getNoisePower();
    signal_power  = t.getSignalPower();
    pixel_count   = t.getPixelCount();
    confidence    = t.getConfidence();
    binarydata_length = t.getDataLength();
    t.getData(0, binarydata_length, (char*)data);
    putRow();
}

//======================================  Check that the pointers are OK
void
TrigTable::check(const char* title) const {
    MetaTable::check(title, sizeof(TrigTable));
}

//======================================  Define the trigger table.
SegTable::SegTable(bool alphaPID)
  : MetaTable("segment"), proc_id_length(nProcessKeyBytes),
    process_id(&proc_id_length, processID), altPID(alphaPID)
{
    defineColumn("segment_group", &group);
    defineColumn("version",       &version);
    defineColumn("start_time",    &start_time);
    defineColumn("start_time_ns", &start_time_ns);
    defineColumn("end_time",      &end_time);
    defineColumn("end_time_ns",   &end_time_ns);
    if (altPID) defineColumn("process_id", &PIDstr, true);
    else        defineColumn("process_id", &process_id);
}

//======================================  Delete the trigger table
SegTable::~SegTable(void) {
    MetaTable::clear();
}

//======================================  Fill the trigger table.
void
SegTable::addRow(const trig::Segment& s) {
    group         = s.getGroup();
    version       = s.getVersion();
    start_time    = s.getStartTime().getS();
    start_time_ns = s.getStartTime().getN();
    end_time      = s.getEndTime().getS();
    end_time_ns   = s.getEndTime().getN();
    if (altPID) PIDstr = s.getProcess();
    else        ProcTable::packKey(nProcessKeyBytes, s.getProcess(), processID);
    putRow();
}

//======================================  Check that the pointers are OK
void
SegTable::check(const char* title) const {
    MetaTable::check(title, sizeof(SegTable));
}
