/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "SegAccountant.hh"
#include "Segment.hh"
#include "TrigClient.hh"
#include <stdexcept>
#include <cmath>

using namespace std;
using namespace trig;

//======================================  seg_info default constructor
SegAccountant::seg_id::seg_id(void) 
    : _version(1)
{}

//======================================  seg_info data constructor
SegAccountant::seg_id::seg_id(const string& name, int versn) 
    : _segname(name), _version(versn)
{
    string::size_type inx = name.find(":");
    if (inx != string::npos) {
	_ifo = name.substr(0, inx);
	_segname.erase(0, inx+1);
    }
}

//======================================  seg_info constructor
SegAccountant::seg_id::~seg_id(void) {
}

//======================================  Set the ifo name
void
SegAccountant::seg_id::setIfo(const string& ifo) {
    _ifo = ifo;
}

//======================================  Align a time to an interval
static Time
tAlign(const Time& t, Interval dt) {
    return t - Interval(fmod(t.totalS(), double(dt)));
}

//======================================  seg_info constructor
SegAccountant::seg_info::seg_info(const seg_id& sid, const string& com, 
				  Interval delay) 
  : seg_id(sid), _comment(com),  _maxtime(delay), _start(0), _last(0), 
    _state(false)
{}

//======================================  seg_info destructor
SegAccountant::seg_info::~seg_info(void) {
}

//======================================  Write the current segment 
void 
SegAccountant::seg_info::seg_write(TrigClient* tc, const Time& end) {
    if (!tc) return;
    Time tEnd = end;
    if (!tEnd || _last < tEnd) tEnd = _last;
    if (_start < tEnd) { 
	Segment s(name(), version(), _start, tEnd);
	if (_state) s.setActivity(1);
	else        s.setActivity(0);
	s.setIfos(ifo());
	s.setComment(_comment);
	tc->sendSegment(s);
	_start = tEnd;
    }
}

//======================================  Set a segment
void
SegAccountant::seg_info::set_segment(TrigClient* tc, const Time& start, 
				     const Time& end, bool onoff) {
    if (!_start) {
	_start = start;
	_last  = end;
	_state = onoff;
    } else if (start >= _start && start <= _last && _state == onoff) {
	_last = end;
    }  else if (_last < start || _state != onoff) {
	seg_write(tc);
	_start = start;
	_last  = end;
	_state = onoff;
    } else {
	throw runtime_error("segment not sequential");
    }
}

//======================================  Locate the named segment type
void 
SegAccountant::seg_info::update(TrigClient* tc, const Time& tup) {
    if (!tup) {
	seg_write(tc, tup);
    } else if (_maxtime != Interval(0)) {
	Time tEnd = _start + _maxtime;
	if (tup >= tEnd) seg_write(tc, tup);
    }
}

//======================================  Get mode string
const char* 
SegAccountant::flush_mode_to_char(flush_mode fm) {
    switch (fm) {
    case kNone:
	return "None";
    case kMaxTime:
	return "MaxTime";
    case kAligned:
	return "Aligned";
    case kWriteThrough:
	return "WriteThrough";
    }
    return "Unknown";
}

#define TEST_FLUSH_MODE(txt,mode) if(txt==flush_mode_to_char(mode)) return mode
SegAccountant::flush_mode 
SegAccountant::flush_char_to_mode(const std::string& fc) {
    TEST_FLUSH_MODE(fc,kNone);
    TEST_FLUSH_MODE(fc,kMaxTime);
    TEST_FLUSH_MODE(fc,kAligned);
    TEST_FLUSH_MODE(fc,kWriteThrough);
    throw runtime_error(string("Invalid flush_mode string: ")+fc);
}
#undef TEST_FLUSH_MODE

//======================================  Construct segment accountant
SegAccountant::SegAccountant(void) 
    : mTC(0), mStart(0), mLastUpdate(0), mLatest(0)
{
}

//======================================  Construct segment accountant
SegAccountant::SegAccountant(TrigClient& tc) 
    : mTC(&tc), mStart(0), mLastUpdate(0), mLatest(0)
{
}

//======================================  Destroy segment accountant
SegAccountant::~SegAccountant(void) {
    close();
}

//======================================  Add a segment type to be managed
void 
SegAccountant::addSegment(const string& name, int version, 
			  const string& com, Interval dly) {
    addSegment(seg_id(name, version), com, dly);
}

//======================================  Add a segment type to be managed
void 
SegAccountant::addSegment(const seg_id& sid, const string& com, Interval dly) {
    unsigned int insp = locate(sid);
    if (insp < mList.size() && mList[insp] == sid) {
	string msg("addSegment called for existing segment type: ");
	msg += sid.full_name();
	throw runtime_error(msg);
    }
    mList.insert(mList.begin()+insp, seg_info(sid, com, dly));
}

//======================================  Close the segment accountant.
void
SegAccountant::close(void) {
    //----------------------------------  Note that the Segment accountant 
    //                                    does not own the trigger client. 
    //                                    It will be taken care of later.
    if (mTC) {
	update(Time(0));
	mTC = 0;
    }
}

//======================================  Close the segment accountant.
ostream& 
SegAccountant::dump(std::ostream& out) const {
    out << "Dump of SegAccountant" << endl;
    for (unsigned int i=0; i<mList.size(); ++i) {
	out << static_cast<const seg_id&>(mList[i]) << endl;
    }
    return out;
}

//======================================  Find the named segment type info
SegAccountant::seg_info& 
SegAccountant::find(const seg_id& str) {
    index_type inx = locate(str);
    if (inx >= mList.size() || mList[inx] != str) {
        cerr << "Can't find segment: " << str << endl;
	throw runtime_error("SegAccountant: Named segment not found");
    }
    return mList[inx];
}

//======================================  Locate the named segment type
SegAccountant::index_type
SegAccountant::locate(const seg_id& sid) {
    index_type hi = mList.size();
    index_type lo = 0;
    while (hi > lo) {
	index_type mid = (hi+lo) >> 1;
	if (mList[mid] < sid) {
	    if (lo != mid) lo = mid;
	    else break;
	} else {
	    hi = mid;
	}
    }
    return hi;
}

//======================================  Set flush mode
void
SegAccountant::set_mode(flush_mode m, Interval t) {
    mFlush         = m;
    mFlushInterval = t;
    cout << "Accountant mode is: " << SegAccountant::flush_mode_to_char(m)
	 << " time: " << t << endl;
}

//======================================  Set a segment
void
SegAccountant::set_segment(const string& name, const Time& start, 
			   const Time& end, bool onoff) {
    set_segment(seg_id(name), start, end, onoff);
}

//======================================  Set a segment
void
SegAccountant::set_segment(const seg_id& sid, const Time& start, 
			   const Time& end, bool onoff) {
    find(sid).set_segment(mTC, start, end, onoff);
    if (end > mLatest) mLatest = end;
}

//======================================  Locate the named segment type
void 
SegAccountant::start_stride(const Time& ts) {
    if (!mLastUpdate) {
	if (mFlush == kAligned) {
	    mLastUpdate = tAlign(ts, mFlushInterval);
	} else {
	    mLastUpdate = ts;
	}
    } else if (mLastUpdate < ts) {
	update(ts);
    }
    if (ts > mLatest) mLatest = ts;
    mStart = ts;
}

//======================================  seg_info constructor
Time
SegAccountant::test_write(const Time& t) const {
    Time tNext;
    switch (mFlush) {
    //----------------------------------  Accumulate indefinitely
    case kNone:
	tNext = Time(0);
	break;

    //----------------------------------  Allow a maximum accumulation time
    case kMaxTime:
	tNext = mLastUpdate + mFlushInterval;
	break;

    //----------------------------------  Flush on a spcified boundary
    case kAligned:
	if (!mLastUpdate) {
	    tNext = tAlign(t + mFlushInterval, mFlushInterval);
	} else {
	    tNext = tAlign(mLastUpdate + mFlushInterval, mFlushInterval);
	}
	break;

    //----------------------------------  Force immediate write.
    case kWriteThrough:
	tNext = t;
    }
    return tNext;
}

//======================================  Locate the named segment type
void 
SegAccountant::update(const Time& tup) {
    //----------------------------------  Is it impossible?
    if (!mTC) throw runtime_error("SegAccountant::update: No TrigClient!");

    //----------------------------------  Record completed segments.
    int N = mList.size();
    for (int i=0; i<N; ++i) {
	mList[i].update(mTC, tup);
    }
    if (tup > mLatest) mLatest = tup;

    //----------------------------------  Write to the final point
    Time tFinal = tup;
    if (!tup) tFinal = mLatest;
    Time tNext = test_write(tFinal);
    while (tNext > mLastUpdate && tNext <= tFinal) {
	mTC->flush(mLastUpdate, tNext);
	mLastUpdate = tNext;
	tNext = test_write(tFinal);
    }
    if (!tup && mLastUpdate < tFinal && tNext > mLastUpdate) {
	mTC->flush(mLastUpdate, tNext);
	mLastUpdate = tNext;
    }
}
