/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "LockStatus.hh"
#include "TSeries.hh"

void LockStatus::Check ( const DaccAPI& dacc ) {
    switch ( fSite )
	{
	case kLHO:
	    {
		CheckLockStatus ( dacc, kH1 );
		CheckLockStatus ( dacc, kH2 );
		CheckLockStatus ( dacc, kH1MC );
		CheckLockStatus ( dacc, kH2MC );
		break;
	    }
	case kLLO:
	    {
		CheckLockStatus ( dacc, kL1 );
		CheckLockStatus ( dacc, kL1MC );
		break;
	    }
	default:
	    break;
	}
}

void LockStatus::CheckLockStatus ( const DaccAPI& dacc, LockType ifo ) {
    int indx = ifo;
	
    const TSeries* ts = dacc.refData ( fName[indx].c_str() );
	
    int data[16];
    if ( ts ) {
	ts->getData ( 16, data );
	if ( CheckLockBits ( data, ifo ) ) {
	    if ( fLock[indx] == kNonLock || fLock[indx] == kLost ) {
		fLock[indx] = kGained;
		fStart[indx] = dacc.getFillTime();
	    }
	    else fLock[indx] = kLock;
	}
	else {
	    if ( fLock[indx] == kLock || fLock[indx] == kGained ) {
		fLock[indx] = kLost;
	    }
	    else fLock[indx] = kNonLock;
	}
    }
    else {
	fLock[indx] = kLock; // in case no state vector available
    }
	
    if ( fLock[indx] && dacc.getFillTime() - fStart[indx] > fWait ) {
	fStable[indx] = true;
    } 
    else {
	fStable[indx] = false;
    }
}

bool LockStatus::CheckLockBits ( const int* data, LockType ifo ) {
    int i = 0;

    switch ( ifo ) {
    case kH1:
    case kH2:
    case kL1:
	{ // check bit 3 of state vector
	    while ( (i < 16) && ((data[i] & 65212) == 65212) ) {
		++i;
	    }
	    break;
	}
    case kH1MC:
    case kH2MC:
    case kL1MC:
	{ // check bit 8 of state vector
	    while ( (i < 16) && (data[i] & 256) ) {
		++i;
	    }
	    break;
	}
    default:
	{
	    return false;
	    break;
	}
    }

    if ( i == 16 ) 
	return true;
    else 
	return false;
}

int LockStatus::GetLockStatus ( int ifo ) const {
    int ret = kLock;
	
    switch ( fSite ) {
    case kLHO:
	{
	    switch ( ifo )
		{
		case kH1:
		    {
			ret = fLock[kH1];
			break;
		    }
		case kH2:
		    {
			ret = fLock[kH2];
			break;
		    }
		case kH1MC:
		    {
			ret = fLock[kH1MC];
			break;
		    }
		case kH2MC:
		    {
			ret = fLock[kH2MC];
			break;
		    }
		default:
		    break;
		}
	    break;
	}
    case kLLO:
	{
	    switch ( ifo )
		{
		case kL1:
		    {
			ret = fLock[kL1];
			break;
		    }
		case kL1MC:
		    {
			ret = fLock[kL1MC];
			break;
		    }
		default:
		    break;
		}
	    break;
	}
    default:
	break;
    }

    return ret;
}

bool LockStatus::IsStable ( int ifo ) const {

    bool ret = true;

    switch ( fSite ) {
    case kLHO:
	{
	    switch ( ifo )
		{
		case kH1:
		    {
			ret = fStable[kH1];
			break;
		    }
		case kH2:
		    {
			ret = fStable[kH2];
			break;
		    }
		case kH1MC:
		    {
			ret = fStable[kH1MC];
			break;
		    }
		case kH2MC:
		    {
			ret = fStable[kH2MC];
			break;
		    }
		default:
		    break;
		}
	    break;
	}
    case kLLO:
	{
	    switch ( ifo )
		{
		case kL1:
		    {
			ret = fStable[kL1];
			break;
		    }
		case kL1MC:
		    {
			ret = fStable[kL1MC];
			break;
		    }
		default:
		    break;
		}
	    break;
	}
    default:
	break;
    }

    return ret;
}

void LockStatus::Register(DaccAPI& dacc, int s ) {

    fSite = (SiteType)s;

    // lock acquisition channels
    if ( s == kLHO ) {
	dacc.addChannel( kH1LockChannel );
	dacc.addChannel( kH2LockChannel );
	// dacc.addChannel( kH1MCLockChannel );
	// dacc.addChannel( kH2MCLockChannel );
    }
    else if ( s == kLLO ) {
	dacc.addChannel( kL1LockChannel );
	// dacc.addChannel( kL1MCLockChannel );
    }

    fName[kH1] = kH1LockChannel;
    fName[kH2] = kH2LockChannel;
    fName[kL1] = kL1LockChannel;
    fName[kH1MC] = kH1LockChannel;
    fName[kH2MC] = kH2LockChannel;
    fName[kL1MC] = kL1LockChannel;
    // fName[kH1MC] = kH1MCLockChannel;
    // fName[kH2MC] = kH2MCLockChannel;
    // fName[kL1MC] = kL1MCLockChannel;
}
