// functions to run a set of filters on a channel, where
// the filters may be designed for use at different sampling
// rates

// C includes
#include <stdio.h>
#include <cmath>

// dmt includes
#include "Dacc.hh"
#include "Interval.hh"
#include "Time.hh"
#include "DVector.hh"
#include "TSeries.hh"
#include "DecimateBy2.hh"

// filtering includes
#include "logicals.h"
#include "IIRfilter_structures.hh"
#include "IIRfilter.hh"
#include "IIRfilter_bank.hh"
#include "IIRfilter_instance.hh"
#include "Rms_output.hh"
#include "Chan_filtset.hh"

using namespace std;

Chan_filtset::
Chan_filtset ( )
{
  no_of_iirfilt = 0;
  raw_data_sampling_rate = 0;
  overall_no_of_iirfilt = 0;
  decimated_data = NULL;
  decimator = NULL;
  f_ins = NULL;
  first_filter_run = YES;
}

int
Chan_filtset::
fill_filters_for_channel ( char* channame,
			   int srate,
			   IIRfilter_bank* f_bank )
{
  // cout << "Entering fill_filters_for_channel with channame=" 
  //      << channame << " sampling rate=" << srate << endl;
  // set channel name
  channel_name = channame;

  // get number of filters in bank.
  overall_no_of_iirfilt = f_bank->no_of_filters;

  // ASSUME FOR NOW THAT THE SAMPLING RATES OF THE
  // FILTERS IN THE FILTER BANKS ARE IN DESCENDING ORDER.
  // SO I DON'T NEED TO SORT THE FILTERS BY SAMPLING RATE.

  // skip_filters is the number of filters in the bank which
  // have a design sampling rate too high for the channel. This
  // section of code calculates skip_filters
  int skip_filters = 0;
  int ratebuff;
  for(int filter_count = 0;
       filter_count < overall_no_of_iirfilt;
       ++filter_count ) {

    f_bank->get_sampling_rate( filter_count, &ratebuff);
    // cout << "filter_count=" << filter_count << ": ratebuff=" << ratebuff 
    //      << " vs chan sample rate=" << srate << endl;
    if( ratebuff > srate ) {
      ++skip_filters;
    }
  }

  // get number of filters
  no_of_iirfilt = overall_no_of_iirfilt - skip_filters;
  // cout << "no_of_iirfilt=" << no_of_iirfilt << " overall_no_of_iirfilt="
  //      << overall_no_of_iirfilt << " skip_filters=" << skip_filters << endl;

  // assign memory to array of filters
  f_ins = new IIRfilter_instance [ no_of_iirfilt ];
  decstage = new int [ no_of_iirfilt ]; 
  sampling_rate = new int [ no_of_iirfilt ];
  logbuf = new int [ no_of_iirfilt ];

  // initialize filter contents, ommitting any filters at the 
  // beginning of the set whose sampling rates were too high 
  // for the channel.
  for(int filter_count = 0;
      filter_count < no_of_iirfilt;
      ++filter_count) {
    
    (f_ins + filter_count)->
      init_filter_instance( (filter_count+skip_filters),
			    f_bank );
    f_bank->get_sampling_rate( (filter_count+skip_filters),
			       (sampling_rate+filter_count) );
  }

  // figure out how many decimation stages are needed between each
  // iir filter stage and the next one.

  int log_raw_srate;
  frexp((double)srate,&log_raw_srate);
  for(int filter_count = 0;
      filter_count < no_of_iirfilt;
      ++filter_count ) {

    frexp((double)(sampling_rate[filter_count]),
	  (logbuf + filter_count));

    if(filter_count == 0) {
      decstage[filter_count] = log_raw_srate - logbuf[filter_count];
    }
    else {
      decstage[filter_count] = logbuf[filter_count - 1]
	- logbuf[filter_count];
    }
    // cout << "decstage[" << filter_count << "]=" 
    //      << decstage[filter_count] << endl;
  }

  // initialize the decimation filter array
  decimator = new DecimateBy2 [ no_of_iirfilt ] ();
  for(int filter_count = 0;
       filter_count < no_of_iirfilt;
       ++filter_count ) {

    if( decstage[filter_count] > 0 ) {
   
      // the 1 is the antialiasing filter type applied before
      // the decimator.

      (decimator+filter_count)->setDecimation( decstage[ filter_count ],
						1 );
    }
  }

  // initialize timeseries arrays to hold decimated data

  decimated_data = new TSeries* [ no_of_iirfilt ];

  first_filter_run = YES;

  return 0;
}

void
Chan_filtset::reset( void)
{
  decimator->reset();
}

int 
Chan_filtset::get_filter_name( int filter_index, 
			       char* pfilter_name ) 
{
  sprintf(pfilter_name,"%s",((f_ins + filter_index)->filter_name));
  return 0;
}

int
Chan_filtset::get_filter_srate( int filter_index,
				int* psrate )
{
  *(psrate) = sampling_rate[ filter_index ];
  return 0;
}

int
Chan_filtset::run_filters( TSeries ts,
			   int* pcolumn_count,
			   float* output, 
			   int make_file )
{
  int input_timeseries_length = (int)ts.getNSample();
  int filter_count;
  float* input_data = new float [ input_timeseries_length ];
  exception e;

  // cout << "Starting run_filters with pcolumn_count=" << *pcolumn_count << endl;
  if( first_filter_run == YES ) {
    for( filter_count = 0;
	 filter_count < no_of_iirfilt;
	 ++filter_count ) {
      
      // create timeseries for decimated data whilst applying the
      // decimator
      if( decstage[ filter_count ] > 0 ) {
	if( filter_count == 0 ) {
	  (*(decimated_data + filter_count)) = new TSeries ( 
            ((decimator + filter_count)->apply( ts )) );
	}
	else {
	  (*(decimated_data + filter_count)) = new TSeries (
	     (decimator + filter_count)->
	      apply( **(decimated_data + filter_count - 1) ) );
	}
      }
      // if not necessary to decimate between successive filter
      // stages, initialize by copying the contents of the 
      // previous timeseries array, or the raw data.
      else {
	if( filter_count == 0 ) {
	  *(decimated_data + filter_count) = new TSeries
	    ( ts );
	}
	else {
	  *(decimated_data + filter_count) = new TSeries
	    ( **(decimated_data + filter_count - 1) );
	}
      }
    }
    first_filter_run = NO;
  }
  // if this isn't the first run of raw data through the decimation bank...
  else {
    for( filter_count = 0;
	 filter_count < no_of_iirfilt;
	 ++filter_count ) {
      
      // decimate if necessary
      if( decstage[ filter_count ] > 0 ) {
	// cout << "   checking for decimation with filter_count=" << filter_count << endl;
	// cout << "   current time of decimator is " << decimator->getCurrentTime() << endl;
	if( filter_count == 0 ) {

	  //	  cout << "About to call datacheck for time series with t0 = " << ts.getStartTime() << endl;
	  try {
	    (decimator+filter_count)->dataCheck(ts);
	  } catch (exception& e) {
	    cout << "oops, got caught on dataCheck(1), will reset filters" << endl;

	    if( ((decimator+filter_count)->isDataValid(ts))==false ) {
	      (decimator+filter_count)->reset();
	    }
	    delete [] input_data;
	    return -1;

	  }
	  
	  **(decimated_data + filter_count) = (decimator + filter_count)->
	  apply( ts );
	}
	else {
	  // check timeseries data for validity on decimator
	  
	  try {
	    (decimator+filter_count)->
	      dataCheck( **(decimated_data+filter_count-1) );
	  } catch (exception& f ) {
	    cout << "oops, got caught on dataCheck(2), will reset filters" << endl;

	    if( ((decimator+filter_count)->
		 isDataValid( **(decimated_data + filter_count-1)) )==false ) {
	      (decimator+filter_count)->reset();
	    }
	    delete [] input_data;
	    return -1;

	  }
	  
	  **(decimated_data + filter_count) = (decimator + filter_count)->
	    apply( **(decimated_data + filter_count - 1) );
	}
      }
      // if not necessary to decimate between successive filter
      // stages, initialize by copying the contents of the 
      // previous timeseries array, or the raw data.
      else {
	if( filter_count == 0 ) {
	  **(decimated_data + filter_count) = ts;
	}
	else {
	  **(decimated_data + filter_count) =
	    **(decimated_data + filter_count - 1);
	}
      }
    }
  }
  // run the IIR filters on the decimated data set.
  float dummy_output_data;
  float sum_of_squares;

  for( filter_count = 0;
       filter_count < no_of_iirfilt;
       ++filter_count ) {

    // grab the floating point data from the time series

    (*(decimated_data + filter_count))->
      getData( (  (*(decimated_data + filter_count))->getNSample() ),
	       input_data );

    /*
      cout << "In run_filters, about to call apply_filter for filter_count=" << filter_count << endl;
    
      int nuse = (int)(*(decimated_data+filter_count))->getNSample();
      cout << "   getNSample= " << nuse << " filter_name=" << (f_ins + filter_count)->filter_name << endl;
      cout << "   input_data: " << endl;
      for (int j=0; j<nuse; j++) {
      cout << j << ":" << input_data[j] << "   ";
      }
      cout << endl;
    */

    // apply the IIR filter to the decimated data

    (f_ins + filter_count)->apply_filter
      ( (int)(*(decimated_data + filter_count))->getNSample() ,
	input_data,
	&dummy_output_data,
	&sum_of_squares,
	make_file,
	(f_ins + filter_count)->filter_name,
	NO );

    // update output file
    output[ *pcolumn_count ] += sum_of_squares;
    ++(*pcolumn_count);
    
  }
  // all done, hopefully.

  delete [] input_data;
  return 0;   
}

Chan_filtset::~Chan_filtset()
{
  int filter_count;
  // only free the decimated data timeseries if they were allocated
  // in the first place.
  if( first_filter_run == NO ) {
    for( filter_count = 0; 
	 filter_count < no_of_iirfilt;
	 ++filter_count ) {
      delete *(decimated_data + filter_count);
    }
  }
  // the remaining arrays were allocated in the constructor, so they
  // should always be deallocated in the destructor.
  delete [] decimated_data;
  delete [] logbuf;
  delete [] decstage;
  delete [] sampling_rate;
  delete [] f_ins;
  delete [] decimator;
  return;
}









