/* -*- mode: c++; c-basic-offset: 3; -*- */

//-->  The next three lines are needed if you are going to generate triggers.
//     The text in PIDTITLE should describe the monitor function.
#define PIDCVSHDR "$Header: https://redoubt.ligo-wa.caltech.edu/svn/gds/trunk/Monitors/blrms/BandMonitor.cc 7131 2014-07-18 00:01:15Z john.zweizig@LIGO.ORG $"
#define PIDTITLE  "Data monitor trigger template"
#include "ProcIdent.hh"
#include "BandMonitor.hh"
#include <time.h>
#include <iostream>
#include <fstream>
#include <cstdlib>
#include <string>
#include "Dacc.hh"
#include "Interval.hh"
#include <sstream>

#include "Time.hh"
#include "DVecType.hh"
#include "scandir.hh"
#include "frame_name.hh"
#include "Trend.hh"
#include "IIRfilter_structures.hh"
#include "IIRfilter.hh"
#include "IIRfilter_bank.hh"
#include "IIRfilter_instance.hh"
#include "Rms_output.hh"
#include "Chan_filtset.hh"
#include "logicals.h"
#include "RmsConfig.hh"
#include "xsil/Xwriter.hh"
#include "xsil/xsil.hh"

//--------------------------------------  Generate the main routine.
EXECDAT(BandMonitor)

using namespace std;
using namespace trig;

//--------------------------------------  Skeleton object constructor.
BandMonitor::BandMonitor(int argc, const char *argv[]) 
   : DatEnv(argc, argv), MaxFrame(999999), monDt(60), uout(NULL),
     mLast_t0(0)
{
   MonServer::setDebug(Debug());
   mTC.setDebug(Debug());
   read_count=0;
   // argument to hold parameter file name, output file name.
   char input_param_file_name[ 100 ];
   sprintf(input_param_file_name,"default");
   char output_param_file_name[ 100 ];
   sprintf(output_param_file_name,"default");
   char update_param_file_name[ 100 ];
   sprintf(update_param_file_name,"none");
  
   //  sprintf(mon_server_name, "Band_Limited_RMS_Dropouts_New");
   //  sprintf(mon_server_name, "Band_Limited_RMS_TEST2");
   sprintf(mon_server_name, "Band_Limited_RMS");
   total_seconds_to_run = 1000000000;
   //  total_seconds_to_run = 400;
   // default to no trend writing
   make_trends = false;
   make_trend_index = false;
   // default trend name
   trend_name = "BLRMSDEF";
   // default to writing text files
   make_text_file = true;
   // KSG 8/26/05 addition of "DMT-BRMS_"after "H0:" etc in trend channel names
   trend_name_new = "DMT-BRMS_";
   // KSG 09/23/05 trend_name is now set equal to trend_name_new
   trend_name = trend_name_new;

   //----------------------------------  Look for arguments
   if( ((argc%2)==0) || (argc < 3) ) {
      DumpHelp();
      exit(0);
   }
   for (int i=1 ; i<argc ; i++) {

      // Input configuration file
      if (!strcmp("-if", argv[i])) {
	 sprintf(input_param_file_name,"%s",argv[++i]);

	 // Output parameter file
      } else if (!strcmp("-of", argv[i])) {
	 sprintf(output_param_file_name,"%s",argv[++i]);
      } else if (!strcmp("-uf", argv[i])) {
	 sprintf(update_param_file_name,"%s",argv[++i]);

	 // Monitor name (for dmt viewer server)
      } else if (!strcmp("-mons", argv[i])) {
	 strcpy(mon_server_name, argv[++i]);

	 // Handle DatEnv arguments
      } else if ( isDatEnvArg(argv[i]) ) {
	 ++i;

	 // Disable making text output file
      } else if (!strcmp("-mtxt", argv[i])) {
	 if ( strcmp(argv[++i],"no") == 0 ) {
	    make_text_file = false;
	 }

	 // Turn on trend output file
      } else if (!strcmp("-midx", argv[i])) {
	 if ( strcmp(argv[++i],"yes") == 0 ) {
	    make_trend_index = true;
	 }

	 // Get trend file name 
      } else if (!strcmp("-trendname",argv[i])) {
	 trend_name = argv[++i];

	 // Oops, unrecognized argument, abort
      } else {
	 DumpHelp();
	 finish();
	 return;
      }
   }

   //----------------------------------  set the server name.
   cout << "Setting monitor name (for name server) = " << mon_server_name 
	<< endl;
   setServerName(mon_server_name);
  
   //----------------------------------  set the time stride.
   float tStep(1.0);
   getDacc().setStride(tStep);
  
   //---------------------------------- get the channel name list
   rmspar = new RmsConfig( input_param_file_name ,
			   output_param_file_name ,
			   update_param_file_name );
  
   // see if rms will be calculated once a minute. If yes, enable
   // generation of minute trends. If no, print a warning that no trends
   // will be generated. 
   if ( rmspar->secs_per_rms_reading == 60 ) {
      make_trends = true;
      pblrmsTrend = new Trend (trend_name.c_str(),Trend::kMinute);
   } else {
      cout << "Warning: rms refresh time (second last line of config" << endl
	   << "file) is not 60 (seconds), so no trends are created" << endl
	   << "by this monitor" << endl;
   }

   //---------------------------------- allocate memory for timeseries array
   ts.resize(rmspar->number_of_channels, NULL);

   //----------------------------------  Specify channels to scan
   for(int j=0; j < (rmspar->number_of_channels); ++j) {
      getDacc().addChannel( rmspar->channel_names[j] );
   }
  
   // set filter state to uninitialized
   are_filters_initialized = 0;

   // set data counter to zero.
   count_seconds = 0;

   // allocate memory for channel samp rates
   srate = new int [ rmspar->number_of_channels ];
  
   // allocate memory for filter instances
   filters = new Chan_filtset [ rmspar->number_of_channels ];
   number_of_filter_outputs = 0;

   // Zero some counters
   segment_counter = 0;
   second_counter = 0;
   time_lost_int_total = 0;
   column_tally = 0;
   first_reading_done = false;
  
   monitor_length = 720; // 720 minutes = 12 hours
   // set maximum length of the name for a monitor
   max_monitor_name_length = 50;

   cout << "Init is finished" << endl;
}


//--------------------------------------  Skeleton object destructor.
BandMonitor::~BandMonitor() 
{
   cout << "Starting cleanup" << endl;
  
   // the conditional on the cleanup determines if the
   // monitor ever started at all. If it did, we probably
   // need to delete the pointers allocated during startup.
   // This kluge will go away as soon as there is a decent
   // way of initializing quantities after reading one set
   // of data to get the data properties.
   if(are_filters_initialized != 0) {
      delete [] srate;
      delete file_output;
      delete [] filters;
      delete mon_name_buf;
      delete rmspar;
   }
   if( make_trends ) {
      delete pblrmsTrend;
   }
}


void 
BandMonitor::DumpHelp(void)
{
   cout << "usage: -if <full path to configuration file and file name>" << endl
	<< "        -of <full path to output data file and file name  >" << endl
	<< "        -uf <file to contain only a single blrms reading >" << endl
	<< "        -seccount <enter finite length (secs) before exit >" << endl
	<< "        -mons <monitor server name (unique!!) >" << endl
	<< "        -mtxt <no/other> disables text file output if no" << endl
	<< "        -midx <yes/other> enables index for trends if yes" << endl
	<< "        -trendname <text string (no spaces) for trend name >" 
	<< endl;
   return;
}

void 
BandMonitor::Attention(void) {
   MonServer::Attention();
}

//======================================  Monitor tseries functions
//
//======================================  Add a list of frames to a dacc
static void
add_frames(const Time& t0, const Interval dT, const std::string& dir, 
	   DaccAPI& dacc) 
{
   scandir sd(dir);
   Time::ulong_t tMin = t0.getS();
   Time::ulong_t tMax = (t0+dT).getS();
   while (sd.next_entry()) {
      if (!sd.is_frame()) continue;
      Time::ulong_t fMin = sd.gps();
      if (fMin >= tMax) continue;
      if (fMin + sd.duration() <= tMin) continue;
      dacc.addPath(sd.file_path());
   }
}

//======================================  Get data from trend frames
static void 
fill_from_trend(TSeries& ts, const Trend& tr, const string& chname) {
   Dacc in;
   Time start = ts.getStartTime();
   Interval dT = ts.getInterval();
   Time tend = start + dT;
   const char* dmtout=getenv("DMTRENDOUT");
   if (dmtout) {
      frame_name frname;
      frname.set_directory(dmtout);
      string start_dir=frname.dir_name(start);
      cout << "add frames from directory: " << start_dir << endl;
      add_frames(start, dT, start_dir, in);
      string end_dir = frname.dir_name(tend);
      if (end_dir != start_dir) add_frames(start, dT, end_dir, in);
   } else {
      add_frames(start, dT, ".", in);
   }

   in.addChannel(chname);
   in.seek(start);
   int rc = 0;
   while (!rc) {
      if (in.synch() || in.getCurrentTime() >= tend) break;
      Interval segDt = tend - in.getCurrentTime();
      rc = in.fillData(segDt);
      if (rc == -3) break;
      if (!in.refData(chname)) continue;
      const TSeries& trend_ts(*(in.refData(chname)));
      if (!trend_ts.refDVect()) continue;
      const DVector& trend_dv(*trend_ts.refDVect());
      size_t fbin = ts.getBin(trend_ts.getStartTime());
      size_t lbin = ts.getBin(trend_ts.getEndTime());
      ts.refDVect()->replace(fbin, lbin-fbin, trend_dv);
      if (rc == -4) rc = 0;
   }
}

//======================================  Extract a sub-series
static void 
fill_from_series(TSeries& out, const TSeries& in) {
   out = in.extract(out.getStartTime(), out.getInterval());
}
 
//======================================  Square root of a vector
static void
sqrt_dvect(DVectF& out) {
   size_t N=out.size();
   for (size_t i=0; i<N; i++) {
      out[i] =  sqrt(out[i]);
   }
}

//--------------------------------------  Frame processing function.
//-------------------------------------- do this step every timestride.
void BandMonitor::ProcessData(void) {
   // Increment overall seconds counter and modulo 60 version
   second_counter++;

   // For testing dropout response, skip a time stride or two...
   // (note that with this method, second_counter gets doubly incremented 
   // for each loss, but segment_counter does not)
   segment_counter++;

   // Get times series of requested channels
   for (int i=0; i < rmspar->number_of_channels; ++i) {
      ts[i] = getDacc().refData( rmspar->channel_names[i] );
   }

   //------- THIS CODE IS ONLY RUN THE FIRST TIME I TAKE DATA --------

   // Init filters the first time I take data
   if( are_filters_initialized == 0 ) {

      are_filters_initialized = 1;
      cout << "First call to ProcessData - more initialization" << endl;

      IIRfilter_bank iir_filters( rmspar->filter_filename.c_str() );
      for(int k=0; k < rmspar->number_of_channels; ++k) {
	 srate[k] = ts[k]->getNSample();
	 (filters+k)->fill_filters_for_channel( rmspar->channel_names[k],
						srate[k], &iir_filters );
	 number_of_filter_outputs += (filters+k)->no_of_iirfilt;
      }
      file_output = new Rms_output(rmspar->output_filename, 
				   number_of_filter_outputs);

      // either allow or inhibit writing of text files
      if( !make_text_file ) {
	 file_output->disableTextOutput();
      }

      // write headers in output file
      // and use header information to name the trend channels
      // to be written to minute trend files. Minute trend files are
      // only created if the blrms update period is set to 60 seconds
      // in the blrms config file.

      char filter_name_buffer[64];
      int filter_srate;
      for(int channel_count = 0; 
	  channel_count < rmspar->number_of_channels;
	  channel_count++ ) {
	 for(int filter_count = 0;  
	     filter_count < ((filters+channel_count)->no_of_iirfilt);
	     filter_count++ ) {
	    (filters + channel_count)->get_filter_name(filter_count,
						       filter_name_buffer);
	    file_output->add_header( ((filters+channel_count)->channel_name),
				     filter_name_buffer,
				     column_tally );

	    (filters + channel_count)->get_filter_srate(filter_count,
							&filter_srate);
	    file_output->update_samp_rate(column_tally,
					  filter_srate );
	    ++column_tally;
	 }
      }
      // allocate memory to array for storing mean square from data.
      data_store.resize(column_tally, 0);

      // Now we know the number of channels, rmspar->number_of_channels,
      // the name of the kth channel, rmspar->channel_names[k],
      // the number of filters for the kth channel, (filters+k)->no_of_iirfilt,
      // the name of the jth filter for the kth data set, 
      // (filters+k)->getFilterName(char* filter_name, j), and
      // the total number of outputs, column_tally
      // So now we can allocate memory to the Timeseries to hold the
      // data to be sent to the monitor.

      // Fill initial 12-hour time series with zeroes from monitor_init array
      float zero(0);
      monDt = double(rmspar->secs_per_rms_reading);
      Time stop = ts[0]->getStartTime();
      stop -=  Interval(stop.getS() % Time::ulong_t(monDt));
      Time start12h = stop - monDt * double(monitor_length);
      TSeries ts12h(start12h, monDt, 1, &zero);
      ts12h.extend(stop);
      ts12h += -1.0;
      Time start48h = stop - monDt * double(4 * monitor_length);
      TSeries ts48h(start48h, monDt, 1, &zero);
      ts48h.extend(stop);
      ts48h += -1.0;
      monitor_tseries.resize(column_tally, ts12h);
      monitor48h_tseries.resize(column_tally, ts48h);

      // fill array of names of monitor channels
      mon_name_buf = new char [max_monitor_name_length];
      int monitor_count(0);
      for(int i=0; i<rmspar->number_of_channels ; ++i) {
	 for(int j=0; j<(filters+i)->no_of_iirfilt ; ++j) {
	    (filters+i)->get_filter_name(j,mon_name_buf);

	    monitor_names.push_back((filters+i)->channel_name + "_" + mon_name_buf);

	    // Tell DMT viewer server where to get data
	    cout << "Pointing name server to channel band " 
		 << monitor_names[monitor_count] << endl;
	    serveData(monitor_names[monitor_count].c_str(), 
		      &(monitor_tseries[monitor_count]), 0);
	    serveData((monitor_names[monitor_count] + "_48h").c_str(), 
		      &(monitor48h_tseries[monitor_count]), 0);

	    //--------------------------  name trend channels if enabled
	    if ( make_trends ) {
	       string trendname = monitor_names[monitor_count].substr(0,3)
		                +  "DMT-BRMS_"
		                + monitor_names[monitor_count].substr(3);
	       size_t hinx = trendname.rfind('-');
	       while (hinx > 6 && hinx != string::npos) {
		  trendname[hinx] = '_';
		  hinx = trendname.rfind('-');
	       } 
	       if (Debug()) cout << "trendname = " << trendname << endl;
	       trend_channel_list.push_back(trendname);
	       pblrmsTrend->addChannel(trendname.c_str());

	       //----------------------------  Fetch data from the trend frames
	       string meanChan = trendname + ".mean";
	       fill_from_trend(monitor48h_tseries[monitor_count],
			       *pblrmsTrend, meanChan);
	       sqrt_dvect(dynamic_cast<DVectF&>
	 		  (*monitor48h_tseries[monitor_count].refDVect()));
	       fill_from_series(monitor_tseries[monitor_count], 
				monitor48h_tseries[monitor_count]);
	    }
	    // end of code to name trend channels
	
	    ++monitor_count;
	 }
      }

      // make index for trends
      if ( make_trends && make_trend_index ) {
	 pblrmsTrend->writeIndex();
      }
      // now we are all ready to start writing time series of monitor data.
   }


   //------------ THIS CODE RUN ONCE EVERY SECOND  ---------------

   // Get start time for this second of data
   tnow = ts.front()->getStartTime();

   // convert timeseries data to floats
   for(int q = 0; q < rmspar->number_of_channels ; ++q) {
      ts[q]->Convert(DVector::t_float);
   }
  
   // Compute expected start time and compare to actual (force equality on 
   //   first second)
   Time expect_t0;
   if (second_counter==1) expect_t0 = tnow;
   else expect_t0 = mLast_t0 + getDacc().getStride();
   mLast_t0 = tnow;

   Interval time_lost = tnow - expect_t0;
   int time_lost_int = int(time_lost);
   segment_counter += time_lost_int;
   second_counter += time_lost_int;
   time_lost_int_total += time_lost_int;

   // Reset filters if more than 30 contiguous seconds lost
   if (time_lost_int > 30) {
      cout << "Severe error: lost >30 seconds of data - Resetting filters" 
	   << endl;
      for(int q = 0; q < rmspar->number_of_channels ; ++q) (filters+q)->reset();
   } 

   if (time_lost_int > 0) {
      cout << "Warning: lost " << time_lost_int 
	   << " seconds - Expected start time = " 
	   << expect_t0 << ", found " << tnow << endl;
   }
    
   // If we had a data dropout that wasn't too long, fill in gap
   // with copies of the current second so that the filter bank doesn't get
   // upset about the gap and reset itself, creating glitches and ringdowns

   Time t0past;
   Interval TimeStep;
   int Nsample;
   float* datacopy;
   for (int iextra=0; iextra<time_lost_int; iextra++) {
  
      cout << "Warning: Running filters extra iteration " << iextra << endl;
      column_count = 0;
      for(int q = 0; q < rmspar->number_of_channels ; ++q) {

	 t0past = tnow - Interval(time_lost_int-iextra);
	 TimeStep = ts[q]->getTStep();
	 Nsample = ts[q]->getNSample();
	 datacopy = new float[Nsample];
	 ts[q]->getData(Nsample,datacopy);

	 TSeries TScopy(t0past, TimeStep, Nsample, datacopy);

	 // run filters on timeseries data
	 if (filters[q].run_filters(TScopy, &column_count, data_store.data(), NO)
	     == -1 ) {
	    cout << "Error: from run_filters attempt at past time t = " 
		 << t0past << endl;
	 }
      
      }
   }

   // Run filters on this second of data

   column_count = 0;
   for(int q=0; q < rmspar->number_of_channels ; ++q) {
      if (filters[q].run_filters(*ts[q], &column_count, 
				 data_store.data(), NO) < 0) {
	 cout << "Error: from run_filters attempt at tnow = " << tnow << endl;
      }
   }

   //------------ THIS SECTION RUN ONCE EVERY secs_per_rms_reading --
   if (!(getDacc().getCurrentTime().getS() % rmspar->secs_per_rms_reading)) {
      Time tbegin = getDacc().getCurrentTime() 
	 - Interval(rmspar->secs_per_rms_reading);

      // If there is an update file, open it
      if (rmspar->update_filename != "none") {
	 uout = fopen(rmspar->update_filename.c_str(),"w");
	 if (!uout) {
	    cerr << "Unable to open update file: " << rmspar->update_filename
		 << endl;
	 }
      }
    
      //send current gps time to output file
      file_output->gps_time_for_entry = tbegin.getS();
    
      // Loop over channels & bands
    
      for(int m=0; m<column_tally; ++m) {
	 // Calculate and store mean square from sum of squares
	 data_store[m] /= float(segment_counter * file_output->samp_rate_array[m]);

	 // Calculate and write ROOT mean square
	 file_output->add_data((float)sqrt((double)data_store[m]),m);	    
  
	 // Loop over data quality flag criteria
	 for (int i=0;i<rmspar->dqflag_count;i++) {
	
	    // Construct threshold string to compare against DQ flag criterion

	    string stringcheck(monitor_names[m]);
	    if (stringcheck[3] == '-') stringcheck[3] = '_';
	    stringcheck = stringcheck.substr(0,3) + "DMT-BRMS_" 
	       + stringcheck.substr(3);

	    // If match to DQ flag criterion, construct segment to send to database
	    bool a=stringcheck == rmspar->dqflag_channel_names[i];
	
	    if (stringcheck == rmspar->dqflag_channel_names[i]) {
	       if (Debug()) cout << stringcheck << " " 
				 << rmspar->dqflag_channel_names[i] 
				 << " " << a << endl;
	       // Parameters to define segment
	  
	       char *groupname=rmspar->dqflag_names[i];	
	       int version=1;	

	       //----------------------------  Get "interferometer name" for DQ 
	       //                              flags (H1, L1 for E14/S6 running)
	       string IFOname;
	       if (stringcheck[0] == 'H') IFOname = "H1";
	       else IFOname = "L1";
	  
	       // Send a segment: 1 = active and 0 = inactive
	       if (data_store[m]>rmspar->dqflag_thresholds[i]){
		  trig::Segment s(groupname,version,tnow-(Interval)60,tnow);
		  s.setComment(rmspar->dqflag_comments[i]);
		  s.setIfos(IFOname.c_str());
		  s.setActivity(1);
		  if (Debug()) cout << "Sending active segment for dqflag " 
				    << groupname << " RMS power = " 
				    << data_store[m] << ", threshold=" 
				    << rmspar->dqflag_thresholds[i] 
				    << ", tnow=" << tnow << endl;
		  sendSeg(s);
		  flag_status[i].active=1;   //Store status for monitor display
		  flag_status[i].time=tnow;
		  flag_status[i].value=data_store[m];
	       }
	       else{
		  trig::Segment s(groupname,version,tnow-(Interval)60,tnow);
		  s.setComment(rmspar->dqflag_comments[i]);
		  s.setIfos(IFOname.c_str());
		  s.setActivity(0);
		  flag_status[i].active=0;
		  flag_status[i].value=data_store[m];
		  if (Debug()) cout << "Sending INactive segment for dqflag " 
				    << groupname << " RMS power = " 
				    << data_store[m] << ", threshold=" 
				    << rmspar->dqflag_thresholds[i]  
				    << ", tnow=" << tnow << endl;
		  sendSeg(s);	 
	       }
	    }
	 }
    
	 // Update standard DMT minute trend file    
      
	 if( make_trends ) {
	    pblrmsTrend->trendData(trend_channel_list[m].c_str(), 
				   tbegin, (double)data_store[m]);
	 }

	 //**** Update time series sent to DMT viewer
	 Interval delta = tbegin - monitor_tseries[m].getEndTime();
	 monitor_tseries[m].eraseStart(delta + monDt);
	 monitor48h_tseries[m].eraseStart(delta + monDt);
	 if (delta != Interval(0)) {
	    Time tMiss = monitor_tseries[m].getEndTime();
	    monitor_tseries[m].extend(tbegin);
	    monitor48h_tseries[m].extend(tbegin);
	    float zero(0);
	    TSeries tmp(tMiss, monDt, 1, &zero);
	    tmp -= 1;
	    monitor_tseries[m]    += tmp;
	    monitor48h_tseries[m] += tmp;
	    cout << "Warning: Inserting -1 (missing data) for "
		 << delta - monDt << " seconds before " << tnow.getS() 
		 << endl;
	 }
	 float sqrt_data_m = sqrt(data_store[m]);
	 monitor_tseries[m].Append(tbegin, monDt, &sqrt_data_m, 1);
	 monitor48h_tseries[m].Append(tbegin, monDt, &sqrt_data_m, 1);

	 // Send latest data point from channel to output file
	 if(rmspar->update_filename != "none" && uout) {
	    fprintf(uout,"%s %e\n", monitor_names[m].c_str(), 
		    sqrt(double(data_store[m])));
	 }
      }      
    
      //reset data_store (and print)
      // cout << "End of minute with data_store=";
      for(int p=0;p<column_tally;++p) {
	 // cout << p << ": " << data_store[p] << " |  ";
	 data_store[p] = 0;
      }
      // cout << endl;
  
      // Reset segment counters
      segment_counter = 0;
      time_lost_int_total = 0;

      //----------------------------------  Update the DMT Monitor Status page
      htmloutput();	 
   }
  
   //if open, close update file
   if (rmspar->update_filename != "none" && uout) {
      fclose(uout);
   }
  
   //----------------------------------  Terminate if requested  
   if ( second_counter == total_seconds_to_run ) {
      cout << "Reached target number of seconds to run: " 
	   << second_counter << " - exiting" << endl;
      //    cout << "Flushing trigger client" << endl;
      //    mTC.flush(tStart,tCur); 
      //    cout << "Closing trigger client" << endl;
      //    mTC.close();
      finish();
   }
  
   return;
}

void BandMonitor::sendSeg(Segment s)
{
   lmsg::error_type rc = mTC.sendSegment(s);
   if (rc) cout << "Error sending segment: error code " << rc << endl;
  
}  /* END: sendSeg() */

//======================================  Creates monitor status html page at 
//                                        DMTHTMLOUT.
void BandMonitor::htmloutput(void) {
   const char* htmlout = getenv("DMTHTMLOUT");
   string htmldir = htmlout ? htmlout : ".";
   string mName=mon_server_name;
   // string filename=htmldir+"/monitor_reports/" + mName + "/index.html";
   string filename=htmldir + "/index.html";
   ofstream htmlfile(filename.c_str());
   htmlfile<<"<head>"<<endl;
   htmlfile<<"</head>"<<endl;
   htmlfile<<"<body>"<<endl;
   htmlfile<<"<center>"<<endl;
   htmlfile<<"<h1>Band Limited RMS Monitor Status</h1><br><br>"<<endl;

   const char* format="%H:%N%:%S";
   const char* format2="%s";
   char utctime[100];
   char utctime_now[100];	
   TimeStr(tnow,utctime_now,format);
   char utctime_next[100];
   char local_now[100];
   char local_next[100];
   LocalStr(tnow,local_now,format);
   LocalStr(tnow+60,local_next,format);
 
   TimeStr(tnow+60,utctime_next,format);
   htmlfile << "This page was last updated on " << tnow << " UTC-> "
	    << utctime_now << " Local-> " << local_now << "<br>" << endl;
   htmlfile << "Next page update will be on " << tnow+60 << " UTC-> "
	    << utctime_next << " Local-> " << local_next << "<br><br><br>"
	    << endl;

   htmlfile<<"<table align=center><tr><td>"<<endl;


   htmlfile<<"<table border=1>"<<endl;
   htmlfile<<"<tr><td>Channels Being Monitored</td></tr>"<<endl;
   for (int i=0;i<rmspar->number_of_channels;i++){
      htmlfile<<"<tr><td>"<<rmspar->channel_names[i]<<"</td></tr>"<<endl;
   }
   htmlfile<<"</table></td><td>"<<endl;
   htmlfile<<"<table border=1>"<<endl;
   htmlfile<<"<tr><th>Filters</th></tr>"<<endl;
   char filter_name_buffer[200];
   for (int filter_count=0; 
	filter_count < filters->no_of_iirfilt; 
	filter_count++ ) {
      filters->get_filter_name(filter_count,filter_name_buffer);
      htmlfile << "<tr><td>" << filter_name_buffer << "</td></tr>" << endl;
   }
   htmlfile<<"</table>"<<endl;

   htmlfile<<"</td></tr></table><br clear=all><br>"<<endl;

   htmlfile << "<table border=2>" << endl;
   htmlfile << "<tr><td>Data Quality Flag</td>" << "<td>Channel</td> " 
	    << "<td>Threshold*</td>" << "<td>Square Mean Value*</td>" 
	    << "<td>RMS Value</td>" << "<td>Status</td>" 
	    << "<td>Time Last On</td>" << "<td>Comments</td></tr>" << endl;
   for (int i=0;i<rmspar->dqflag_count;i++){
      htmlfile << "<tr>"<<endl;
      htmlfile << "<td><center>" << rmspar->dqflag_names[i] 
	       << "</center></td>";
      htmlfile << "<td><center>" << rmspar->dqflag_channel_names[i]
	       << "</center></td>";
      htmlfile << "<td><center>" << rmspar->dqflag_thresholds[i] 
	       << "</center></td>";
      htmlfile << "<td><center>" << flag_status[i].value << "</center></td>";
      TimeStr(flag_status[i].time,utctime,format2);

      htmlfile << "<td><center>" << sqrt(flag_status[i].value) 
	       << "</center></td>";
      htmlfile << "<td><center>" << flag_status[i].active << "</center></td>";
      htmlfile << "<td><center>" << utctime << "</center></td>";
      htmlfile << "<td><center>" << rmspar->dqflag_comments[i] 
	       << "</center></td>";
      htmlfile << "</tr>" << endl;
   }

   htmlfile << "</table>" << endl;
   htmlfile << "*Note: The value is the Square Mean Value. "
	    << "The monitor output shows the root of this, while the "
	    << "threshold is compared with this value.<br>" << endl;
   htmlfile << "</center>" << endl;
   htmlfile << "</body>" << endl;	
   if (Debug()) cout << "OUTPUTTING HTML >>>> " << filename << endl;
}
