#ifndef TRIGSPEC_HH
#define TRIGSPEC_HH

#include "Histogram2.hh"
#include "FSpectrum.hh"
#include "TrigEnv.hh"
#include <string>
#include <list>
#include <vector>
#include <iosfwd>

class Pipe;
class TSeries;
class Histogram1;

/**  TrigSpec is Plots a spectrogram of a specified channel near triggers 
  *  of a specified type. A spectrum is taken before and after the trigger
  *  to get an background estimate which is assumed to be stationary.
  *
  *  COnfiguration file
  *
  *  The configuration file may contain the following statements:
  *  \begin{itemize}
  *  \item Omit - Time stamps of triggers to be omitted
  *  \item TFPlot - Specification of time spans for triggers to be plotted.
  *  \item Lock - Lock condition required.
  */

//======================================  Class HLimit
class HLimit {
public:
    typedef double limit_t;
public:
    HLimit(void);
    HLimit(limit_t low, limit_t hi, std::string name="");
    HLimit(const HLimit& x);
    ~HLimit(void);
    limit_t getLower(void) const {return mLower;}
    const Histogram1* getHisto(void) const {return mHisto;}
    const char* getName(void) const {return mName.c_str();}
    limit_t getUpper(void) const {return mUpper;}
    bool inLimits(limit_t x) const;
    void clear(void);
    void setLimits(limit_t low, limit_t hi);
    void setHisto(int nbin, limit_t binLo, limit_t binHi);
private:
    limit_t     mLower;
    limit_t     mUpper;
    std::string mName;
    Histogram1* mHisto;
};


//======================================  Trigger ID class
class TrigID {
public:
    TrigID(const std::string& ID, const std::string& subID);
    TrigID(const TrigID& t);
    virtual ~TrigID(void);
    const char* getID(void) const;
    const char* getSubID(void) const;
    virtual bool isTrig(const trig::TrigBase& t);

private:
    std::string mTrigID;
    std::string mSubID;
};

inline const char* 
TrigID::getID(void) const {
    return mTrigID.c_str();
}

inline const char* 
TrigID::getSubID(void) const {
    return mSubID.c_str();
}

//======================================  Time/Frequency plot class.
class TFPlot :  public TrigID {
public:
  TFPlot(const char* name, const char* ID, const char* subID);
  TFPlot(const TFPlot& tf);
  ~TFPlot(void);

  void addSlice(const Time& t0, const TSeries& ts);
  void addSigSpec(const TSeries& ts);
  void addBakSpec(const TSeries& ts);

  TSeries Filter(const TSeries& t);

  const char* getChannel(void) const;
  Interval    getEpoch(void) const;
  const char* getName(void) const;
  Interval    getOffset(void) const;
  const Histogram2& getSN(void);
  const Histogram1& getAHist(void) const;
  bool isTrig(const trig::TrigBase& t);

  void setALimits(double aMin, double aMax);
  void setAHisto(double hMin, double hMax);
  void setChannel(const std::string& d);
  void setFAxis(int nbin, double fLo,   double fHi);
  void setFilter(const Pipe* p=0);
  void setBSlice(int nSlice,  Interval Offset);
  void setBSlice(int nSlice1, Interval Offset1, int nSlice2, Interval Offset2);
  void setTAxis(int nbin, Interval tLo, Interval Thi);

  void stats(std::ostream& out) const;

private:
  std::string mName;
  std::string mChannel;
  Pipe*       mFilter;
  Interval    mBakOff1;
  int         mNBSlice1;
  Interval    mBakOff2;
  int         mNBSlice2;
  int         mTBins;
  Interval    mTmin;
  Interval    mTstep;
  int         mFBins;
  double      mFmin;
  double      mFstep;
  int         mSigCount;
  Histogram2  mSigHist;
  FSpectrum   mBakSpect;
  bool        mSNValid;
  Histogram2  mSNHist;
  bool        mBakDefault;
  HLimit      mALimit;
};

#ifndef __CINT__
inline const char*
TFPlot::getName(void) const {
    return mName.c_str();
}

inline const char*
TFPlot::getChannel(void) const {
    return mChannel.c_str();
}

inline void 
TFPlot::setBSlice(int nSlice, Interval Offset) {
    setBSlice(nSlice, -Offset, nSlice, Offset);
}

inline const Histogram1&
TFPlot::getAHist(void) const {
    return *(mALimit.getHisto());
}

#endif

class TrigSpec : public TrigEnv {
public:
  TrigSpec(int argc, const char* argv[]);
  ~TrigSpec(void);
  void ProcessData(void);
  bool ProcessTrigger(const trig::TrigBase& t);
  void readConfig(const char* f);
private:
  typedef std::list<TFPlot>  TrigList;
  typedef TrigList::iterator trig_iter;

  typedef std::vector<Time>  TimeList;
private:
  TrigList    mList;
  std::string mSvChannel;
  trig_iter   mCurTrig;
  std::string mHistFile;
  TimeList    mOmit;
};



#endif  // !defined(TRIGSPEC_HH)
