/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "FilterDesign.hh"
#include "ParseLine.hh"
#include "Station.hh"
#include "TFile.h"
#include "TH1.h"
#include <cmath>
#include <iostream>

EXECDAT(Station)

using namespace std;

//======================================  Histogram conversion
inline TH1*
makeRootHist(const string& name, const Histogram1& h) {
    int nBin = h.GetNBins();
    Histogram1::xbin_t lo = h.GetBinLowEdge(0);
    Histogram1::xbin_t hi = h.GetBinLowEdge(nBin+1);
    auto_ptr<TH1D> rh(new TH1D(name.c_str(), h.GetTitle(), nBin, lo, hi));
    Histogram1::histdata_t* bins = new Histogram1::histdata_t[nBin+2];
    h.GetBinContents(bins);
    rh->SetContent(bins);
    h.GetBinErrors(bins);
    rh->SetError(bins);
    delete[] bins;
    rh->SetEntries(h.GetNEntries());
    return rh.release();
}

//======================================  xy-plot class constructor
xyhist::xyhist(const string& name, int nbin, double xmin, double xmax) 
  : mChannel(name), mSum1(" ", nbin, xmin, xmax), mSumy(" ", nbin, xmin, xmax),
    mSumyy(" ", nbin, xmin, xmax), mXYPlot(" ", nbin, xmin, xmax)
{
}

//======================================  xy-plot class destructor
xyhist::~xyhist(void) {
}

//======================================  Calculate x-y plot
void 
xyhist::calculate(void) {
    int N = mSum1.GetNBins();
    double* dat(new double[N]);
    mSumy.GetBinContents(dat);
    double* err(new double[N]);
    mSumyy.GetBinContents(err);
    double* ent(new double[N]);
    mSum1.GetBinContents(ent);
    for (int i=0; i<N; i++) {
        double nEnt = ent[i];
        if (nEnt > 0) {
	    dat[i] /= nEnt;
	    err[i] /= nEnt;
	    err[i] -= dat[i]*dat[i];
	    if (err[i] > 0.0) err[i] = sqrt(err[i]);
	    else              err[i] = 0;
	}
    }
    mXYPlot.SetBinContents(dat);
    mXYPlot.SetBinErrors(err);
    delete[] dat;
    delete[] ent;
    delete[] err;
}


//======================================  Add an entry
void 
xyhist::fill(double x, double y) {
    mSum1.Fill (x, 1.0);
    mSumy.Fill (x, y);
    mSumyy.Fill(x, y*y);
}

//======================================  Write x-y histogram
void 
xyhist::write(void) const {
}

//======================================  Statistics class constructor
stat_class::stat_class(const string& name, const string& chan, long nSample) 
  : mName(name), mChannel(chan), mNSample(nSample), mFMin(0), mFMax(0), 
    mPsdStride(100.0), mNPsdAvg(12), minPsdAvg(8), mDetrend(nSample), 
    mSigmaHist("", 100, 0, 1.0), mDebug(0)
{
    reset();
}

//======================================  Statistics class destructor
stat_class::~stat_class(void) {
}

//======================================  Copy operator
void 
stat_class::addPlot(const xyhist& xy) {
    mPlotList.push_back(xy);
}

//======================================  Sigma calculation for one sample
double 
stat_class::getVariance(const TSeries& ts) {
    TSeries dts(mDetrend(ts));
    double power;
    if (mFMin >= mFMax) {
	power = dts*dts;
    } else {
        FSeries dfs(dts);
	power = dfs.Power(mFMin, mFMax);
    }
    power /= dts.getNSample();
    return power;
}

//======================================  Sigma calculation for one sample
void
stat_class::print(void) const {
    mPSD.Dump(cout);
}

//======================================  Sigma calculation for one sample
void
stat_class::procStride(const TSeries& ts) {
    if (!mFilter.isDataValid(ts)) {
	mFilter.reset();
	mStride.Clear();
    } else if (!mStride.empty() && mStride.getEndTime() != ts.getStartTime()) {
	mStride.Clear();
    }

    if (!mFilter.null()) {
	mStride.Append(mFilter(ts));
    } else {
	mStride.Append(ts);
    }
    long nPt    = mStride.getNSample();
    long nSlice = nPt/mNSample;
    Time     t0 = mStride.getStartTime();
    Interval dT = double(mNSample)*mStride.getTStep();
    TSeries noise_tmp(t0+0.5*dT, dT, nSlice);
    float*  sig = reinterpret_cast<float*>(noise_tmp.refData());

    //-----------------------------------  Process the first slice
    mSumPower = getVariance(mStride.extract(t0, dT));
    sig[0]    = (mSumPower > 0) ? sqrt(mSumPower) : 0.0;
    mSumAmpl  = sig[0];
    mSigmaHist.Fill(mSumAmpl);

    //-----------------------------------  Process slices 2-N
    for (long i=1; i<nSlice; i++) {
	double pwr = getVariance(mStride.extract(t0+double(i)*dT, dT));
	mSumPower += pwr;
	double sigi = (pwr > 0) ? sqrt(pwr) : 0.0;
	sig[i]     = sigi;
	mSumAmpl  += sigi;
	mSigmaHist.Fill(sigi);
    }
    double avg = mSumAmpl/nSlice;
    float  var = mSumPower/nSlice - avg*avg;
    if (var > 0) var = sqrt(var);
    else         var = 0.0;
    Interval dTot = dT*double(nSlice);
    if (mDebug > 2) cout << "Append t0: " << t0 << " dT: " << dTot 
			 << " sigma: " << var << endl;
    mHistory.fixedAppend(t0, dTot, &var);
    if (mNoise.isEmpty() || mNoise.getEndTime() != noise_tmp.getStartTime()) {
        mNoise = noise_tmp;
    } else {
        mNoise.Append(noise_tmp);
    }
    if (mNoise.getInterval() >= mPsdStride) {
        Time tStart = mNoise.getStartTime();
        mPSD += FSpectrum(FSeries(mNoise.extract(tStart, mPsdStride)));
	mNoise.eraseStart(mPsdStride);
	if (mPSD.getCount() >= mNPsdAvg) {
	    mAvgPsd.copy(mPSD);
	    mAvgPsd /= double(mPSD.getCount());
	    mPSD.clear();
	}
    }
    mStride.eraseStart(dTot);
}

//======================================  Set the debug level
void
stat_class::setDebug(int nLvl) {
    mDebug = nLvl;
}

//======================================  reset accumulators
void
stat_class::reset(void) {
    mPSD.clear();
    mHistory.Clear();
    mHistory.setMaxLen(7200.0);
}

//======================================  Set filter
void
stat_class::setFilter(const Pipe& pipe) {
    mFilter.set(pipe);
}

//======================================  Set band maximum
void
stat_class::setFMax(double fmax) {
    mFMax = fmax;
}

//======================================  Set band minimum
void
stat_class::setFMin(double fmin) {
    mFMin = fmin;
}

//======================================  Set substride width
void
stat_class::setNSample(long nSample) {
    mNSample = nSample;
    mDetrend = Quadetrend(nSample);
}

//======================================  Set substride width
void
stat_class::setOscCond(const string& cond) {
    mOscCond = cond;
}

//======================================  Sigma calculation for one sample
void
stat_class::update(const TSeries& ts) {
    mStride.Append(ts);
    long nPt    = mStride.getNSample();
    long nSlice = nPt/mNSample;
    Time     t0 = mStride.getStartTime();
    Interval dT = double(mNSample)*mStride.getTStep();
    Interval dTot = dT*double(nSlice);
    float var(0.0);
    mHistory.fixedAppend(t0, dTot, &var);
    mStride.eraseStart(dTot);

    //----------------------------------  Display the average Psd if data is OK
    if (mPSD.getCount() >= minPsdAvg) {
        mAvgPsd.copy(mPSD);
	mAvgPsd /= double(mPSD.getCount());
    }

    //----------------------------------  Clear the PSD.
    mPSD.clear();
}

//======================================  Stationarity monitor constructor
Station::Station(int argc, const char* argv[]) 
  : DatEnv(argc, argv), MonServer("Station"), mStartFrame(0), mEndFrame(0), 
    mOsc(getDacc())
{
    string cfile;
    for (int i=1; i<argc; ++i) {
        string argi=argv[i];
	if (argi == "-conf") {
	    cfile = argv[++i];
	} else if (argi == "-histo") {
	    mHistoFile = argv[++i];
	} else if (argi == "-osc") {
	    mOsc.readConfig(argv[++i]);
	} else if(isDatEnvArg(argv[i])) {
	    i++;
	} else {
	    cerr << "Unrecognized argument: " << argi << endl;
	}
    }

    if (cfile.empty()) {
        cerr << "Config file not specified." << endl;
	finish();
	return;
    }

    ReadConfig(cfile);

    getDacc().setIgnoreMissingChannel(true);
}

//======================================  Stationarity monitor destructor
Station::~Station(void) {
    write();
    mList.clear();
}

//======================================  Process a stride
void 
Station::Attention(void) {
    MonServer::Attention();
}

//======================================  Process a stride
void 
Station::ProcessData(void) {
    if (!mStartFrame) mStartFrame = getDacc().getFillTime();
    for (stat_iter i=mList.begin(); i!=mList.end(); ++i) {
        const char* chan = i->getChannel();
	const TSeries* ts = getDacc().refData(chan);
	if (!ts || ts->isEmpty()) {
	    cerr <<"Unable to find channel: " << chan << endl;
	    continue;
	}

        const char* cond = i->getOscCond();
        if (*cond && !mOsc.satisfied(cond)) {
	    if (Debug() > 2) cout << "Skip tool: " << i->getName() 
				  << " condition: " << cond 
				  << " not satisfied." << endl;
	    i->update(*ts);
	} else {
	    if (Debug() > 2) cout << "Process tool: " << i->getName() << endl;
	    i->procStride(*ts);
	}
    }
    mEndFrame = getDacc().getCurrentTime();
}

//======================================  Read configuration.
void 
Station::ReadConfig(const std::string& file) {

    //----------------------------------  Read configuration file
    ParseLine pl(file.c_str());
    while (pl.getLine() >= 0) {
        int nArg = pl.getCount();
	if (!nArg) continue;
        bool syntax=false;

	//------------------------------  Construct stat entry from name, chan
	const char* pName = pl[0];
	const char* pChan = 0;
	int inx=1;
	if (nArg > 1 && *pl[1] != '-') pChan = pl[inx++];
	else                           pChan = pName;
	stat_class sc(pName, pChan);
	sc.setDebug(Debug());

	//------------------------------  Parse arguments
	char* p=0;
	for (int i=inx; i<nArg; ++i) {
	    string argi(pl[i]);
	    if (argi == "-filter") {
	        FilterDesign fd(pl[++i], 16384.0);
		sc.setFilter(fd.get());
	    } else if (argi == "-fmin") {
		sc.setFMin(strtod(pl[++i], &p));
		syntax = (*p != 0);
	    } else if (argi == "-fmax") {
		sc.setFMax(strtod(pl[++i], &p));
		syntax = (*p != 0);
	    } else if (argi == "-nsample") {
	        sc.setNSample(strtol(pl[++i], &p, 0));
		syntax = (*p != 0);
	    } else if (argi == "-while") {
	        const char* cond = pl[++i];
	        if (mOsc.find(cond) == mOsc.end()) {
		    cerr << "Condition: " << cond << " not found." << endl;
		    syntax = true;
		}
	        sc.setOscCond(cond);
	    } else if (argi == "-xyplot") {
	        string argj = pl[++i];
		unsigned int inx = argj.find(";");
		if (inx > argj.size()) inx = argj.size();
	        string xyname = argj.substr(0,inx);
		int nbin = 100;
		double xmin = 0;
		double xmax = 10.0;
		const char* p=argj.c_str() + inx;
		for (int f=0; f<3; f++) {
		    if (*p++ != ';') break;
		    switch(f) {
		    case 0:
		        nbin = strtol(p, const_cast<char**>(&p), 0);
			break;
		    case 1:
		        xmin = strtod(p, const_cast<char**>(&p));
			break;
		    case 2:
		        xmax = strtod(p, const_cast<char**>(&p));
			break;
		    }
		}
		sc.addPlot(xyhist(xyname,nbin, xmin, xmax));
		getDacc().addChannel(xyname.c_str());
	    } else {
	        cerr << "Unrecognized keyword value: " << argi << endl;
		syntax = true;
		break;
	    }
	}
	if (!syntax) {
	    if (!getDacc().isChannelRead(pChan)) getDacc().addChannel(pChan);
	    mList.push_back(sc);
	}
    }

    //----------------------------------  Book DMT viewer objects
    int nChan = mList.size();
    for (int i=0; i<nChan; ++i) {
	string chan_i = mList[i].getName();
	string obj = chan_i + "_History";
	serveData(obj.c_str(), &(mList[i].refHistory()));
	obj = chan_i + "_SigmaHist";
	serveData(obj.c_str(), &(mList[i].refSigmaHist()));
	obj = chan_i + "_PSD";
	serveData(obj.c_str(), &(mList[i].refPSD()));
        int nPlot = mList[i].getNXrefChans();
	for (int j=0; j<nPlot; ++j) {
	    obj  = chan_i + "_vs_";
	    obj += (mList[i].getXrefChan(j) + 7);
	    serveData(obj.c_str(), &(mList[i].refXrefHisto(j)));
	} 
    }
}

//======================================  Stationarity monitor destructor
void
Station::write(void) {
    Interval dT = mEndFrame - mStartFrame;
    if (dT <= Interval(0)) return;
    FrWriter w("Stationarity", 666);
    w.buildFrame(mStartFrame,  dT);
    for (stat_iter i=mList.begin(); i!=mList.end(); ++i) {
        w.addRawSeries(i->getChannel(), i->refHistory());
    }
    w.addHistory("Station", Now(), "Stationarity Monitor");
    w.addWriterHistory();
    w.open("Station.gwf");
    w.writeFrame();
    w.close();

    //----------------------------------  Store histograms if requested.
    if (!mHistoFile.empty()) {
        TFile f(mHistoFile.c_str(), "RECREATE", "Stationarity Histograms");
        for (const_dict_iter i=refDictionary().begin(); 
	     i != refDictionary().end(); ++i) {
	    const DataDesc& dd(i->second);
	    if (dd.getType() == DataDesc::t_Histogram1) {
	        if (Debug()) cout << "Writing histo: " << i->first <<endl;
	        const Histogram1* hist = reinterpret_cast<const Histogram1*>(dd.getAddr());
		TH1* rhist = makeRootHist(i->first, *hist); 
		if (rhist) {
		    const char* hname = rhist->GetName();
		    if (hname) rhist->Write(hname);
		}
	    }
	}
        f.Close();
    }
}


