/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "PSLBand.hh"
#include "PSLChan.hh"
#include "MonServer.hh"
#include "TrigBase.hh"
#include "TrigClient.hh"
#include <iostream>
#include <iomanip>
#include <cstdlib>
#include <float.h>

using namespace std;
using namespace trig;

//======================================  Frequency Band constructor
PSLBand::PSLBand(const string& name, const PSLChan* chan)
  : PSLTool(name, chan), mAmin(0.0), mAmax(0.0), mDmax(0.0), mFmin(0.0),
    mFmax(0.0), mAvgTime(0.0), mHisto(false), mBand(-1.0), mAvgFreq(0.0), 
    mPeakF(0.0), mTrigTot(0), mLastValid(0),
    mPSDHist(name.c_str(), 100, 0.0, 1.0)
{
    setComment("PSLBand");
}

//======================================  Band tool destructor
PSLBand::~PSLBand(void) {}


//======================================  Band tool clone operator
PSLBand*
PSLBand::clone(void) const {
    return new PSLBand(*this);
}

//======================================  Process frequency band
bool 
PSLBand::crunch(void) {
    mTriggered = false;

    //----------------------------------  Check that the data exist.
    if (!refChan().exists()) {
        if (!mLastValid) cout << "PSLmon - Channel " << getChannel() 
			      << " not available, Band " << getName() 
			      << " ignored." << endl;
	return false;
    }
    if (!refChan().isValid()) return false;
    if (!mLastValid) {
        cout << "Band tool: " << getName() << " frequencies: ";
	int N=mBandLim.size();
        for (int i=0; i<N; i++) {
	    if (i) cout << ", ";
	    cout << mBandLim[i].first << "-" << mBandLim[i].second;
	}
	cout << endl;
    }

    //----------------------------------  Get the band power
    const FSeries& fs(refChan().refFSeries());
    if (mBandLim.empty()) {
        mBand = fs.Power(0, fs.getHighFreq());
    } else {
	int N=mBandLim.size();
        mBand = 0.0;
        for (int i=0; i<N; i++) {
	    mBand += fs.Power(mBandLim[i].first, mBandLim[i].second);
	}
    }
    if (mBand < FLT_MIN) mBand = 0.0;

    //----------------------------------  Invalidate mLast if data skipped
    Time t0 = fs.getStartTime() + refChan().getOverlap();
    if (mLastValid != t0) mLast = 0.0;

    //----------------------------------  Look for a trigger
    bool trig(false);
    if (mAmin != mAmax && mBand >= mAmin && (mAmax == 0 || mBand < mAmax)) {
        trig = true;
    } else if (mLast <= 0.0) {

    } else if (mDmax != 0 && fabs(mBand-mLast) > mDmax) {
        trig = true;
    } else if (mBand < mLast*mFmin) {
        trig = true;
    } else if (mFmax != 0 && mBand > mLast*mFmax) {
        trig = true;
    }
    if (trig) mTrigTot++;
    mTriggered = trig;

    //----------------------------------  Keep the history
    mLastValid = fs.getEndTime();
    Interval dT = mLastValid - t0;
    mOutChan.fillData(t0, mBand, dT);

    //----------------------------------  Set up the histogram
    if (mHisto) {
        const long Navg(64);       // Number of points for hist average.
	const double AvgMult(5.0); // Histogram upper limit in average units.
        if (mOutChan.refStats().getN() > Navg) {
            mPSDHist.Fill(mBand);
	} else if (mOutChan.refStats().getN() == Navg) {
	    double xmax = AvgMult*mOutChan.refStats().getAvg();
	    mPSDHist.SetBinLowEdges(100, 0, xmax);
	    mPSDHist.Fill(mBand);
	}
    }

    //----------------------------------  Calculate a running average power.
    
    if (mAvgTime <= 0.0) {
        mLast = mBand;
    } else {
        double avgFact = exp(-double(dT)/mAvgTime);
	mLast = mBand + (mLast - mBand)*avgFact;
	if (mLast < FLT_MIN) mLast = 0;
    }
    return trig;
}

//======================================  Get average frequency
void 
PSLBand::params(void) {
}

//======================================  Perform post-configuration processing
void
PSLBand::postConfig(MonServer& v) {
    if (getDebug() > 1) cout << "Serving band: " << getName() << endl;
    mOutChan.addTrendChannel();
    mOutChan.serveViewerSeries(v, "Band RMS");
    string hTitle = refName() + "_Histo";
    if (mHisto) v.serveData(hTitle.c_str(), &mPSDHist);
}

//=======================================  Send a trigger to TrigMgr
void 
PSLBand::confSegment(SegAccountant& sa) {
    if (!mSegID.refName().empty()) sa.addSegment(mSegID, comment(), 1);
}

//=======================================  Send a trigger to TrigMgr
void 
PSLBand::sendSegment(SegAccountant& sa) {
    if (!mSegID.refName().empty()) {
	if (!refChan().exists()) return;
	const  FSeries& fs(refChan().refFSeries());
	if (fs.empty()) return;
	sa.set_segment(mSegID, fs.getStartTime(), fs.getEndTime(), mTriggered);
    }
}

//=======================================  Set the segment name
void 
PSLBand::setSegment(const std::string& seg) {
    string name(seg);
    string ifo;
    int version = 1;
    string::size_type inx = name.find(':');
    if (inx != string::npos) {
	ifo = name.substr(0, inx);
	name.erase(0, inx+1);
    }
    inx = name.find(':');
    if (inx != string::npos) {
	version = strtol(name.c_str()+inx+1, 0, 0);
	name.erase(inx);
    }
    mSegID = SegAccountant::seg_id(name, version);
    if (!ifo.empty()) mSegID.setIfo(ifo);
}

//=======================================  Send a trigger to TrigMgr
void 
PSLBand::sendTrigger(TrigClient& tc) {
    if (!getTrigEnable()) return;

    //-----------------------------------  Calculate trigger parameters
    mAvgFreq = 0;
    mPeakF   = 0;

    //----------------------------------  Check that the data exist.
    if (!refChan().exists()) return;
    const  FSeries& fs(refChan().refFSeries());
    if (fs.empty()) return;

    //----------------------------------  Get the frequency domain parameters
    double sump  = 0.0;
    double sumpf = 0.0;
    double peakp = 0.0;
    FSeries::size_type pinx = 0;
    const fComplex* x = reinterpret_cast<const fComplex*>(fs.refData());
    if (mBandLim.empty()) {
        FSeries::size_type iMax = fs.getLength();
        for (FSeries::size_type i=0; i<iMax; ++i) {
	    double p = x[i].MagSq();
	    sump  += p;
	    sumpf += p * fs.getBinF(i);
	    if (p > peakp) {
	        peakp = p;
		pinx  = i;
	    }
	}
    } else {
        int N = mBandLim.size();
	for (int j=0; j<N; ++j) {
	    pinx = fs.getBin(mBandLim[j].first);
	    FSeries::size_type iMax = fs.getBin(mBandLim[j].second);
	    for (FSeries::size_type i=pinx; i<iMax; ++i) {
	        double p = x[i].MagSq();
		sump  += p;
		sumpf += p * fs.getBinF(i);
		if (p > peakp) {
		    peakp = p;
		    pinx  = i;
		}
	    }
	}
    }
    if (sump != 0.0) mAvgFreq = sumpf / sump;
    mPeakF = fs.getBinF(pinx);

    //------------------------------------  Get time domain parameters
    Time tStart = fs.getStartTime();
    Interval dT = fs.getEndTime() - tStart;
    double ampl = sqrt(2.0*mBand);
    trig::TrigBase t("RMSBand", getName().c_str(), tStart, dT, ampl, 1.0);
    t.setPriority(trig::p_error);
    t.setDisposition(trig::d_metaDB);
    t.setIfos(getIFO().c_str());
    double noise = (mAvgTime > 0) ? mLast : 0;
    t.setSignalPower(mBand-noise);
    t.setNoisePower(noise);
    double fHigh = getFHigh();
    double fLow  = getFLow();
    t.setFrequency(0.5*(fHigh+fLow));
    t.setFreqAvg(mAvgFreq);
    t.setFreqPeak(mPeakF);
    t.setBandwidth(fHigh - fLow);

    //------------------------------------  Send the trigger.
    int rc = tc.sendTrigger(t);
    if (rc && rc!=12 ) {
        cout << "Error " << rc << " logging Band " << getName() 
	     << " trigger" << endl;
    }
}

//======================================  Get Limits
double
PSLBand::getLowerLimit(void) const {
    if (mAmin != mAmax) return mAmin;
    if (mLast < mDmax)  return 0.0;
    if (mDmax != 0.0)   return mLast-mDmax;
    return mLast*mFmin;
}

double
PSLBand::getUpperLimit(void) const {
    if (mAmin != mAmax) return mAmax;
    if (mDmax != 0.0)   return mLast+mDmax;
    return mLast*mFmax;
}

//======================================  Print statistics
void 
PSLBand::printStats(ostream& out) const {
    const OutChan::stats_type& stats(mOutChan.refStats());
    out << setw(20) << left << getName();
    out << setw(20) << left << getChannel() << right;
    out << setprecision(3) << setw(6) << getFLow() << setw(6) << getFHigh();
    out << setw(10) << scientific << setprecision(3) << stats.getAvg()<< fixed;
    if (mAmin != mAmax) {
        out << " Absolute " << setw(8) << mAmin << setw(8) << mAmax;
    } else if (mDmax) {
        out << " Delta        -   " << setw(8) << mDmax;
    } else if (mFmax) {
        out << " Fraction " << setw(8) << mFmin << setw(8) << mFmax;
    } else {
        out << "     -        -       -   ";
    }
    double Rate = !stats.getDt() ? 0 : mTrigTot/double(stats.getDt());
    out << setw(10) << Rate;
    out << endl;
}

void 
PSLBand::printStatHdr(ostream& out) const {
    out.setf(ios::left);
    out << setw(66) << " ";
    out << setw(26) << "          Limits";
    out << setw(10) << " Trigger";
    out << endl;

    out << setw(20) << "Tool Name";
    out << setw(20) << "Channel Name";
    out.setf(ios::right);
    out << setw(6)  << "  fLow";
    out << setw(6)  << " fHigh";
    out << setw(10) << "   <RMS>";
    out << setw(10) << "   Type";
    out << setw(8)  << "  Low";
    out << setw(8)  << "  High";
    out << setw(10) << " Rate";
    out << endl;
}

//======================================  Reset
void 
PSLBand::reset(void) {
}

//======================================  Set averaging time.
void 
PSLBand::setAvgTime(double tAvg) {
    mAvgTime = tAvg;
}

//======================================  Specify frequency band
void 
PSLBand::setBand(double Flow, double Fhigh) {
    mBandLim.clear();
    if (Flow >= Fhigh) return;
    mBandLim.push_back(limit_node(Flow, Fhigh));
}

//======================================  Specify frequency band
void 
PSLBand::excludeBand(double Flow, double Fhigh) {
    if (Flow <  getFLow())  Flow  = getFLow();
    if (Fhigh > getFHigh()) Fhigh = getFHigh();
    if (Fhigh <= Flow) return;
    for (band_iter i=mBandLim.begin(); i<mBandLim.end(); ++i) {
        if (Flow >= i->second) continue;
        if (Fhigh <= i->first) break;
	if (Fhigh >= i->second) {
	    i->second = Flow;
	} else {
	    double temp = i->first;
	    i->first = Fhigh;
	    if (Flow > temp) mBandLim.insert(i, limit_node(temp, Flow));
	    break;
	}
    }
}

//======================================  Set absolute limits
void 
PSLBand::setAbsLimit(double Alow, double Ahigh) {
    mAmin = Alow;
    mAmax = Ahigh;
}

//======================================  Set difference limit
void 
PSLBand::setDeltLimit(double Dmax) {
    mDmax = Dmax;
}

//======================================  Set fractional limit(s)
void 
PSLBand::setFracLimit(double Fmax) {
    mFmax = Fmax;
    mFmin = 1.0/mFmax;
}

void 
PSLBand::setFracLimit(double Fmin, double Fmax) {
    mFmax = Fmax;
    mFmin = Fmin;
}

//======================================  Set the histogram enable flag
void 
PSLBand::setHistEnable(bool he) {
    mHisto = he;
}
