/* -*- mode: c++; c-basic-offset: 3; -*- */
#ifndef WPIPE_FRAMECACHE_HH
#define WPIPE_FRAMECACHE_HH
#include <string>

namespace wpipe {

   /**  Maintain a list of frames and other data sources for use by the 
    *  readdata functions.
    */
   class wframecache {
   public:
      /** GPS time data type
       */
      typedef unsigned long gps_type;

      /**  Construct an empty frame cache.
        *  \brief Empty frame cache constructor.
	*/
      wframecache(void);

      /**  Construct a frame cache from either a frame cache file or from a 
       *  directory. The \a file argument is a comma separated list of file 
       *  or directory paths. If the  path is a directory, the directory is
       *  scanned for all files that match the stadard file name template,
       *  \e i.e. "<prefix>-<gps>-<dt>.gwf". Each file with a matching name 
       *  is added to the frame cache.
       *  \brief Frame cache constructor.
       *  \param file Frame cache file or directory path(s).
       */
      explicit wframecache(const std::string& file);

      /**  Add a frame group to the cache list. The frame group eill define
        *  either a group of successive frames or an alternative source such
	*  as an NDS2 port.
        *  \brief Add a frame group
	*  \param prefix Prefix with frame observatory and type (e.g. L-R)
	*  \param start  Start time of first frame in group.
	*  \param stop   End time of last frame in group.
	*  \param delta  Length in seconds of all frames in group.
	*  \param dir    Directory or nds port. 
        */
      void add_group(const std::string& prefix, gps_type start, gps_type stop, 
		     gps_type delta, const std::string& dir);

      /** Coalesce the frame cache vector.
       */
      void coalesce(void);

      /**  List the frame cache contents to the stdout.
        *  \brief List frame cache.
	*/ 
      void display(void) const;

      /**  Test if cache list is empty.
        *  \return True if cache list is empty.
	*/
      bool empty(void) const;

      /**  Get a list of frames for the given gps segment and frame type.
       *  \brief Get frame file list.
       *  \param prefix Combined observatory and frame type code. 
       *  \param start Start time of required data.
       *  \param stop  End time of required data.
       *  \param frames String vector to receive frame paths.
       *  \return Number of frame IDs.
       */
      int get_list(const std::string& prefix, gps_type start, 
		   gps_type stop, str_vect& frames) const;

      /**  Read a list of frame types and source directories from a specified 
       *  file. The cache file contains the following columns:
       *  - Observatory ID (\e e.g. "H", "L" or "V")
       *  - Frame Type \e e.g. "R"
       *  - Start gps
       *  - End gps
       *  - File length in seconds
       *  - Frame file directory path
       *
       *  For special data sources (\e e.g. nds) the file length is specified
       *  as zero and the path specifier does not need to be modified.
       *
       *  \brief Parse a frame cache file.
       *  \param file List file name.
       */
      void parse_cacheFile(const std::string& file);

      /**  Generate a frame cache from a directory name that contains all 
       *  appropriate frames.
       *  \brief Generate frame cache from directory contents.
       *  \param file List file name.
       */
      void parse_directory(const std::string& file);

      /**  Set the debug print level to the specified value.
        *  \brief Set the debug level.
	*  \param lvl New debug level.
	*/
      void set_debug(int lvl);

   private:
      class frame_group {
      public:
	 frame_group(const std::string& ftype, gps_type start, gps_type stop,
		     gps_type frame_len, const std::string& dir);
	 bool adjacent(const frame_group& grp) const;
	 void combine(const frame_group& grp);

	 /**  List the frame group to the stdout.
	   *  \brief List frame group.
	   */ 
	 void display(void) const;
	 bool valid(const std::string& ftype, gps_type start, 
		    gps_type stop) const;
	 std::string frame(gps_type gps) const;
	 int get_frames(gps_type start, gps_type stop, str_vect& frames) const;

	 /**  Comparison for sort ordering. THe group prefix is the most 
	   *  significant field, followed by the start time.
	   *  \brief Compare frame_groups
	   *  \param x frame_group instance to be compared to this instance.
	   *  \return True if this instance comes before the argument.
	   */
	 bool operator<(const frame_group& x) const;

      private:
	 std::string _prefix;
	 std::string _directory;
	 gps_type _start;
	 gps_type _length;
	 gps_type _end;
      };

      /**  Find an adjacent entry.
       *  \brief Find an adjacent entry
       *  \param prefix Group to test for adjacency
       *  \return Adjancent entry index or -1.
       */
      int find_adjacent(const frame_group& grp) const;

   private:
      typedef std::vector<frame_group> cache_list;

   private:
      int        debugLevel;
      cache_list frameCache;
   };

   //===================================  inline methods.
   inline bool
   wframecache::empty(void) const {
      return frameCache.empty();
   }

   inline bool
   wframecache::frame_group::operator<(const frame_group& x) const {
      return _prefix < x._prefix || (_prefix == x._prefix && _start < x._start);
   }

}  // namespace wpipe
#endif // !defined(WPIPE_FRAMECACHE_HH)
