/* -*- mode: c++; c-basic-offset: 3; -*- */
// $Id$
#ifndef WPIPE_WEVENTLIST_HH
#define WPIPE_WEVENTLIST_HH 1

#include "wtypes.hh"
#include "Time.hh"
#include <string>
#include <iosfwd>

class TrigClient;

namespace wpipe {

   class  wtransform;
   struct qTransform;
   class  wtile;
   class  wouttype;

   /**  The wevent structure contains all the event properties used in the 
     *  various Omega pipeline event manipulation functions, such as:
     *  wthreshold, wcluster, wselect, wveto, etc.
     */
   struct wevent {
      /**  Contruct an empty event.
       */
      wevent(void);
    
      /**  Compare two events according to normalized energy.
        *  \brief Compare event normalized energy
	*  \param evt WEvent instance to be compared to this.
	*  \return True if this instance has a lesser normalized energy. 
	*/
      bool operator<(const wevent& evt) const;

      /**  Compare two events according to normalized energy.
        *  \brief Compare event normalized energy
	*  \param evt WEvent instance to be compared to this.
	*  \return True if this instance has a greater normalized energy. 
	*/
      bool operator>(const wevent& evt) const;

      /**  Display the contents of an event.
       */
      void dump(std::ostream& out) const;

      double t_offset;           ///< Center time offset of the event
      double frequency;          ///< Central frequency of the event.
      double q;                  ///< Central Q value of the event.
      double duration;           ///< Event duration.
      double bandwidth;          ///< Event band-width
      double normalizedEnergy;   ///< Normalized event energy
      double amplitude;          ///< Event amplitude.
      double incoherentEnergy;   ///< Event incoherent energy
      size_t cluster_size;       ///< Number of tiles composing event
      size_t cluster_id;         ///< ID of cluster this event belongs to.
   };

   //====================================  wevent inline functions.
   inline bool 
   wevent::operator<(const wevent& evt) const {
      return normalizedEnergy < evt.normalizedEnergy;
   }

   inline bool 
   wevent::operator>(const wevent& evt) const {
      return normalizedEnergy > evt.normalizedEnergy;
   }

   /**  The weventlist class is an attempt to unify the Omega pipeline event
     *  manipulation functions in a single class. The wevent structure
     *  contains the basic event parameters. Multiple wevent structures from
     *  a single channels collected in an weventlist. The events are unordered
     *  by default, but a sort() method is included to make the list be sorted
     *  by decreasing normalized energy of each event.
     *  \brief List of event from a single channel.
     *  \author John Zweizig (john.zweizig@ligo.org)
     *  \version $Id$
     */
   class weventlist {
   public:
      /**  Construct an empty event list with a specified class name. The
        *  event list class is optional and is used only in printing out 
	*  the lists.
	*/
      weventlist(const std::string& list_class="");

      /**  Add an event to the list.
       */
      void addEvent(const wevent& evt);

      /**  Return the channel name for which this list was created.
        *  \brief Name of the channel that contained these events.
	*  \return Channel name string.
	*/
      const std::string& channelName(void) const;

      /**  Display the contents of the event list.
       */
      void display(std::ostream& out) const;

      /**  wcluster identifies clusters of related significant Q transform 
        *  tiles to improve the detection of gravitational-wave bursts that 
	*  are extended in time and/or frequency. The events in the argument
	*  list \a significants are clustered and the results are added to 
	*  current list. Note tha the argument list is modified (sorted)
	*  by the wcluster method.
	*
	*  wcluster permits two choices of clustering methods: density and 
	*  hierarchical. The input arguments depends upon which method is 
	*  requested. 
	*  The current event list instance is filled with the clusters
	*  found and the argument event list is updated to reflect the
	*  cluster number to which each original event was assigned.
	*
	*  <b> Density based clustering <\b>
	*
	*  Density based clustering first identifies the number of tiles within
	*  a specified clustering radius of a given tile. If the number of 
	*  tiles within this radius meets a threshold cluster density, all of  
	*  the tiles within this radius become part of the same cluster.  
	*  Clusters are constructed by recursively applying this same test to 
	*  all of the other tiles within the clustering radius, until a 
	*  complete set of clusters have been formed.  By default, both 
	*  clusterRadius and clusterDensity are set to unity.
	*
	*  Density based clustering does not in general assign all tiles to 
	*  clusters, but if the cluster singles argument is non zero, single
	*  tiles are included in the output cluster list.  Otherwise, single
	*  tiles are not included, and zeros are reported for their cluster 
	*  properties in the output significants structure.  By default,
	*  clusterSingles is false.
	*
	*  The significant tile structure is updated to contain the cluster 
	*  identification number and number of tiles in cluster
	*  \brief Density -based cluster finding.
	*  \param significants   Event list of significant tile properties
	*  \param distanceMetric Choice of metric to compute distance (see 
	*                        distance())
	*  \param durationInflation Multiplicative scale factor for duration 
	*                           (see distance())
	*  \param bandwidthInflation Multiplicative scale factor for bandwidth
	*                           (see distance())
	*  \param debugLevel Verbosity of debug output
	*/
      void wcluster(weventlist& significants, double clusterRadius=1.0, 
		    double clusterDensity=1.0, bool clusterSingles=false, 
		    const std::string& distanceMetric="",
		    double durationInflation=1.0, 
		    double bandwidthInflation=1.0, 
		    int debugLevel=1);


      /**  wcluster identifies clusters of related significant Q transform 
        *  tiles to improve the detection of gravitational-wave bursts that are
	*  extended in time and/or frequency. The events in the argument
	*  list \a significants are clustered and the results are added to 
	*  current list. Note that the argument list is modified (sorted)
	*  by the wcluster method.
	*
	*  <b> Hierarchical clustering <\b>
	*
	*  In hierarchical clustering, the distances returned by WDISTANCE are
	*  used by the linkage class to construct a hierarchical cluster tree 
	*  by successively linking together the next closest pair of tiles or 
	*  existing clusters.  Finally, clusters are identified by cutting the
	*  hieararchical cluster tree based on the specified criterion and
	*  threshold.
	*
	*  The cluster linkage method defines how intercluster distance is 
	*  computed, and may be any of the following methods.  By default,
	*  'single' linkage is assumed.
	*
	*  <table>
	*  <tr><td>'single'</td>
	*      <td> nearest distance between cluster members</td></tr>
	*  <tr><td>'complete'</td>
	*      <td>furthest distance between cluster members</td></tr>
	*  <tr><td>'average'</td>
	*      <td>average distance between cluster members</td></tr>
	*  <tr><td>'centroid'</td>
	*      <td>center of mass distance between clusters</td></tr>
	*  <tr><td>'ward'</td>
	*      <td>inner squared distance between clusters</td></tr>
	*  </table>
	*
	*  The cluster criterion specifies how to cut the hierarchical cluster 
	*  tree to form clusters, and may be either of the two methods 
	*  understood by the CLUSTER function.  By default, 'inconsistent' is 
	*  assumed.
	*  <table>
	*  <tr><td>'inconsistent'</td>
	*      <td>threshold on inconsistency between of cluster distance and
	*          average subcluster difference</td></tr>
	*  <tr><td>'distance'</td>
	*      <td>threshold on absolute distance between clusters</td></tr>
	*  </table>
	*
	*  The clusterThreshold corresponds to the specified clusterCriterion, 
	*  and has a default threshold of unity for both criteria.
	*/
      void wcluster(weventlist& significants, 
		    const std::string& clusterLinkage, 
		    const std::string& clusterCriterion, 
		    double clusterThreshold, 
		    const std::string& distanceMetric,
		    double durationInflation, double bandwidthInflation, 
		    int debugLevel);

      /**  Dump out the contents of this event list to the specified output
        *  stream.
	*/
      void dump(std::ostream& out) const;

      /**  Get the event class name for the events included in this list.
        *  \brief Event class name.
	*  \return Event class name string.
	*/
      const std::string& eventClass(void) const;

      /**  Compare the current list size to the specified maximum size. If 
        *  list is longer than the specified maximum, sort the list by 
	*  decreasing normalized energy and remove the lowest NE events 
	*  to reduce the size to the specified maximum.
	*  \brief Limit the list size to the specified maximum
	*  \param maxEvents Mximum number of events allowed
	*/
      void limit(size_t maxEvents);

      /**  Get the reference time for the events in this list. The event 
        *  time offset is added to this time to get the central GPS of
	*  the event.
        *  \brief Event list reference time.
	*  \return Reference time for events in the list.
	*/
      const Time& refTime(void) const;

      /**  Return the number of events in the list.
        *  \brief Event list size.
	*  \return Number of events in the list.
	*/
      size_t size(void) const;

      /**  Return a constant reference to the specified event number.
        *  \brief Reference event \a inx.
	*  \param inx requested event index.
	*  \return Constant reference to specified event.
        */
      const wevent& operator[](size_t inx) const;

      /**  Return a reference to the specified event number.
        *  \brief Reference event \a inx.
	*  \param inx requested event index.
	*  \return Reference to specified event.
        */
      wevent& operator[](size_t inx);

      /**  Sort the events in the list in the order of decreasing normalized
        *  energy.
	*  \brief Sort the event list.
	*/
      void sort(void);

      /** Swap the contents of two lists.
       */
      void swap(weventlist& el);

      /**  Sort the events in the list in the order of increasing center time.
	*  \brief Sort the event list by time.
	*/
      void tsort(void);

      /**  WSELECT selects statistically significant events from the set of 
        *  statistically significant Q transform tiles. Events are defined by 
	*  the properties of their most significant tile and are identified 
	*  by excluding the less significant of any overlapping tiles.  The 
	*  input significant tiles must be sorted by decreasing normalized 
	*  energy. If the debugLevel argument is > 2, the ordering of the list 
	*  will be verified on entry and a logic_error exception is thrown if 
	*  the order is incorrect. Starting with the most significant tile, 
	*  tiles are discarded if they overlap with a more significant tile.  
	*  The remaining set of tiles comprises a minimal set of tiles that 
	*  describes an event. The selected events are entered in the current 
	*  event list instance.
	*
	*  The optional durationInflation and bandwidthInflation arguments are
	*  multiplicative scale factors that are applied to the duration and 
	*  bandwidth of significant tiles prior to testing for overlap.  If  
	*  not specified, these parameters both default to unity such that the 
	*  resulting tiles have unity time-frequency area.  The normalized 
	*  energy of the resulting tiles are scaled by the product of the 
	*  duration and bandwidth inflation factors to avoid over counting the 
	*  total energy of clusters of tiles.  Likewise, the amplitude of the 
	*  resulting tiles is scaled by the square root of the product of the
	*  duration and bandwidth inflation factors.
	*
	*  The optional maximumEvents argument provides a safety mechanism to 
	*  limit the total number of events returned by WSELECT.  If this 
	*  maximum number of events is exceeded, an overflow flag is set, only 
	*  the maximumEvents most significant events are returned, and a warning
	*  is issued if debugLevel is set to unity or higher.  By default, 
	*  maximumEvents is set to infinity and debugLevel is set to unity.
	*
	* \param significants      event list tiles properties. This is assumed
	*                          to be ordered by decreasing normalizedEnergy.
	* \param durationInflation multiplicative scale factor for duration
	* \param bandwidthInflation multiplicative scale factor for bandwidth
	* \param maximumEvents     maximum allowable number of events
	* \param debugLevel        verbosity of debug output
	*/
      void wselect(const weventlist& significants, double durationInflation=1,
		   double bandwidthInflation=1, size_t maximumEvents=0, 
		   int debugLevel=1);

      /**  wthreshold identifies discrete Q transform coefficients whose 
       *  magnitudes exceed a normalized energy threshold. The single channel 
       *  false rate calculation assuming ideal white noise is moved upstream.
       *  Each such tile is used to create an event which is added to the 
       *  list.
       *
       *  The user can focus on a subset of the times and frequencies available
       *  in the transform data by specifying a desired range of central times,
       *  central frequencies, and Qs to threshold on.  Ranges should be 
       *  specified as a two component vector, consisting of a minimum and 
       *  maximum value. Alternatively, if only a single Q is specified, 
       *  WTHRESHOLD is only applied to the time-frequency plane which has the 
       *  nearest value of Q in a logarithmic sense to the requested value.
       *
       *  To determine the range of central times to threshold on, WTHRESHOLD
       *  requires the start time of the transformed data in addition to a
       *  reference time and a relative time range.  Both the start time and
       *  reference time should be specified as absolute quantities, while the
       *  range of times to analyze should be specified relative to the 
       *  requested reference time.
       *
       *  By default, WTHRESHOLD is applied to all available frequencies and Qs,
       *  and the reference time and relative time range arguments are set to
       *  exclude data potentially corrupted by filter transients as identified 
       *  by the transient duration field of the tiling structure.  The default 
       *  value can be obtained for any argument by passing an empty vector.
       *
       *  If provided, the optional analysisMode string is used by WTHRESHOLD 
       *  to determine which channels are signal channels, which channels are 
       *  null channels, and which channels to report results for.  For 
       *  coherent analysis modes, a desired veto threshold, squared 
       *  calibration uncertainy factor, and required signal correlation factor
       *  must also be specified.
       *
       *  The optional maximumSignificants argument provides a safety mechanism 
       *  to limit the total number of events returned by WTHRESHOLD.  If this 
       *  maximum number of significants is exceeded, the overflow flag is set, 
       *  only the maximumSignificants most significant tiles are returned, and
       *  a warning is issued if debugLevel is set to 1 or higher.  By default,
       *  maximumSignificants is set to 1e9-1 and debugLevel is set to unity.
       *
       *  \brief Generate list of significant tiles.
       *  \param signal         Input signal Q transform structure
       *  \param tiling         Discrete Q transform tiling structure from WTILE
       *  \param eventThreshold desired Normalized energy event threshold
       *  \param referenceTime  reference time for time range to threshold on
       *  \param timeRange      vector range of relative times to threshold on
       *  \param frequencyRange vector range of frequencies to threshold on
       *  \param qRange         scalar Q or vector range of Qs to threshold on
       *  \param maximumSignificants maximum allowable number of events
       *  \param uncertaintyFactor squared calibration uncertainty factor
       *  \param debugLevel        verbosity of debug output
       */
      void wthreshold(const qTransform& signal, const wtile& tiling, 
		      double eventThreshold, 
		      const Time& referenceTime, const dble_vect& timeRange, 
		      const dble_vect& frequencyRange, const dble_vect& qRange, 
		      size_t maximumSignificants, double uncertaintyFactor=0.0,
		      int debugLevel=1);

      /**  wthreshold identifies discrete Q transform coefficients whose 
        *  magnitudes exceed a specified threshold. The calculation of a 
	*  threshold that approximately yields a specified single channel 
	*  false rate assuming ideal white noise.  has been moved upstream.
	*  Each such tile is used to create an event which is added to the 
	*  list.
	*
	*  The user can focus on a subset of the times and frequencies 
	*  available in the transform data by specifying a desired range 
	*  of central times, central frequencies, and Qs to threshold on.
	*  Ranges should be specified as a two component vector, consisting
	*  of a minimum and maximum value. Alternatively, if only a single
	*  Q is specified, WTHRESHOLD is only applied to the time-frequency
	*  plane which has the nearest value of Q in a logarithmic sense to
	*  the requested value.
	*
	*  To determine the range of central times to threshold on, WTHRESHOLD
	*  requires the start time of the transformed data in addition to a
	*  reference time and a relative time range.  Both the start time and
	*  reference time should be specified as absolute quantities, while the
	*  range of times to analyze should be specified relative to the 
	*  requested reference time.
	*
	*  By default, WTHRESHOLD is applied to all available frequencies and
	*  Qs, and the reference time and relative time range arguments are 
	*  set to exclude data potentially corrupted by filter transients as
	*  identified by the transient duration field of the tiling structure.
	*  The default value can be obtained for any argument by passing an 
	*  empty vector.
	*
	*  If provided, the optional analysisMode string is used by WTHRESHOLD 
	*  to determine which channels are signal channels, which channels are 
	*  null channels, and which channels to report results for.  For 
	*  coherent analysis modes, a desired veto threshold, squared 
	*  calibration uncertainy factor, and required signal correlation 
	*  factor must also be specified.
	*
	*  The optional maximumSignificants argument provides a safety 
	*  mechanism to limit the total number of events returned by 
	*  WTHRESHOLD.  If this maximum number of significants is exceeded,
	*  the overflow flag is set, only the maximumSignificants most 
	*  significant tiles are returned, and a warning is issued if 
	*  debugLevel is set to 1 or higher. By default, maximumSignificants
	*  is set to 1e9-1 and debugLevel is set to unity.
	*
	*  \brief Generate list of significant tiles.
	*  \param signal         Input signal Q transform structure
	*  \param refer          Input null-stream Q transform structure
	*  \param tiling         Discrete Q transform tiling structure from 
	*                        WTILE
	*  \param eventThreshold desired event normalized enerygy threshold
	*  \param referenceTime  reference time for time range to threshold on
	*  \param timeRange      vector range of relative times to threshold on
	*  \param frequencyRange vector range of frequencies to threshold on
	*  \param qRange         scalar Q or vector range of Qs to threshold on
	*  \param maximumSignificants maximum allowable number of events
	*  \param uncertaintyFactor squared calibration uncertainty factor
	*  \param correlationFactor fractional correlated energy threshold
	*  \param debugLevel        verbosity of debug output
	*/
      void wthreshold(const qTransform& signal, const qTransform& refer, 
		      const wtile& tiling, double eventThreshold, 
		      const Time& referenceTime, const dble_vect& timeRange, 
		      const dble_vect& frequencyRange, const dble_vect& qRange,
		      size_t maximumSignificants, double uncertaintyFactor=0.0,
		      double correlationFactor=0.0, int debugLevel=1);

      /**  Write the specified fields of the event list to the output file.
       */
      void writeEvent(const std::string& path, const str_vect& fields, 
		      const std::string& format) const;

      /**  Write the specified fields of the event list to a flat text file
        *  a l'Omega. The fields specified in the field list are printed out.
	*  \brief Write events to a flat text file.
	*  \param path Output text file path.
	*  \param flds List of fields to be printed out.
	*/
      void writeEvent_txt(const std::string& path, const str_vect& flds) const;

      /**  Write the specified fields of the event list to a flat text file
        *  a l'Omega. The fields specified in the field list are printed out.
	*  \brief Write events to a flat text file.
	*  \param ostr Output text stream.
	*  \param flds List of fields to be printed out.
	*/
      void writeEvent_txt(std::ostream& ostr, const str_vect& flds) const;

      /**  Write all entries in the event list to the specified xml file.
        *  Events are first used to generate a gds TrigBase instance and
	*  then written out using a TrigClient instance.
	*  \brief Write events to an xml file.
	*  \param path LIGO_LW output file path.
	*/
      void writeEvent_xml(const std::string& path) const;

      /**  Write all entries in the event list to the specified xml file.
        *  Events are first used to generate a gds TrigBase instance and
	*  then written out using a TrigClient instance.
	*  \brief Write events to an xml file.
	*  \param client Trigger client.
	*/
      void writeEvent_xml(TrigClient& client) const;

   private:
      /// Event vector data type
      typedef std::vector<wevent>  event_vect;

      /// Event vector iterator
      typedef event_vect::iterator event_iter;

   private:
      /**  For both density and hierarchical clustering, distances between all 
        *  pairs of tiles are first computed using the wdistance function and 
	*  the specified distance metric and inflation parameters.
	*
	*  The distance metric specified how distances are computed between 
	*  tiles, and may be any of the methods understood by the wdistance 
	*  function.  By default, 'integratedMismatch' is assumed.
	*  <table>
	*  <tr><td>'pointMismatch'</td>
	*      <td>Second order approximation to mismatch function evaluated 
	*          at center point between tiles</td></tr>
	*  <tr><td>'integratedMismatch'</td>
	*      <td>Second order approximation to mismatch function integrated 
	*          between tiles</td></tr>
	*  <tr><td>'logMismatch'</td>
	*      <td>negative log of the overlap between two tiles</td></tr>
	*  <tr><td>'euclidean'</td>
	*      <td>normalized time frequency distance between tiles</td></tr>
	*  <tr><td>'modifiedEuclidean'</td>
	*      <td>normalized time frequency distance between tiles modified 
	*          to give greater weight to frequency</td></tr>
	* </table>
	*
	*  The specified duration and bandwidth inflation factors are used by 
	*  some distance metrics to compute the distance between tiles.  They 
	*  are also used when computing the total energies of clusters.  By 
	*  default, they are set to unity.
	*
	* \param distances Pointer to pre-allocated output array of dimension
	*                  N*(N-1)/2.
	* \param method Name of method to be used to calculate distances
	* \param durationInflation  Inflation factor for tile duration.
	* \param bandwidthInflation Inflation factor for tile bandwidth.
	*/
      void wdistance(double distances[], const std::string& method, 
		     double durationInflation=1.0, 
		     double bandwidthInflation=1.0);

      /**  Fill the event list with clustered events from the argument 
        *  list.
	*/
      void clusterFill(const weventlist& ev, double dtInflation,
		       double bwInflation);

      /**  Merge two vectors into one sorted vector. The resulting vector 
        *  must contain \a maxLength or fewer events.
	*  \memo Sort and merge the argument vector into _events.
	*  \param ev Vector containing events to be merged into \c _events.
	*  \param maxLength The maximum length of the result vector.
	*/
      void sorted_merge(event_vect& ev, size_t maxLength);

   private:
      std::string _channelName;  ///< Channel name for this list
      std::string _list_class;   ///< Event class (e.g. raw, clustered, etc)
      int         _overflowFlag; ///< more than the maximum number of events.
      event_vect  _events;       ///< list of events
      Time        _refTime;      ///< Reference time for all events in list.
      bool        _sorted;       ///< List is sorted by decreasing normE
   };

   //================================  weventlist inline accessors.
   inline const std::string& 
   weventlist::channelName(void) const {
      return _channelName;
   }

   inline const std::string& 
   weventlist::eventClass(void) const {
      return _list_class;
   }

   inline const Time&
   weventlist::refTime(void) const {
      return _refTime;
   }

   inline size_t 
   weventlist::size(void) const {
      return _events.size();
   }

   inline const wevent& 
   weventlist::operator[](size_t inx) const {
      return _events[inx];
   }

   inline wevent& 
   weventlist::operator[](size_t inx) {
      return _events[inx];
   }

   /**  The eventstack class contains a list of several event lists for multiple
    *  channels.
    */
   class weventstack {
   public:
      /**  Empty event stack constructor
        *  \brief Default constructor.
	*/
      weventstack(void) {}

      /**  Create an event stack an reserve space for the specified number 
        *  of event lists (Channels).
	*  \brief create an event stack and allocat space.
	*  \param N Number of channel slots (event lists) to reserve.
	*/
      weventstack(size_t N);

      void dump(std::ostream& out) const;
      void display(std::ostream& out) const;
      const weventlist& operator[](int i) const;

      /**  Push an empty event list on this stack corresponding to each list 
        *  in the argument stack and swap the contents with the argument 
        *  list.
	*  \brief Move lists from argument stack to this stack.
	*  \param el Event stack containing lists to be moved.
	*/
      void moveLists(weventstack& el);

      /** Number of lists (channels) in this stack.
        * \brief number of channels
	* \return number of channels.
        */
      int numberOfChannels(void) const;

      void status(std::ostream& out) const;

      /**  Swap the contents of two event stacks.
        *  \brief Swap stack contents.
	*  \param wstak Argument %weventstack.
	*/
      void swap(weventstack& wstak);

      /**  Sort the events in the all the lists in the order of increasing 
        *  center time.
	*  \brief Sort the event list by time.
	*/
      void tsort(void);

      /**  Return the total number of events in this event stack.
        *  \brief total events.
	*  \return Total number of events.
	*/
      size_t totalEvents(void) const;

      /**  wcluster identifies clusters of related significant Q transform 
        *  tiles to improve the detection of gravitational-wave bursts that 
	*  are extended in time and/or frequency.
	*
	*  wcluster permits two choices of clustering methods: density and 
	*  hierarchical. The input arguments depends upon which method is 
	*  requested. 
	*  The current event list instance is filled with the clusters
	*  found and the argument event list is updated to reflect the
	*  cluster number to which each original event was assigned.
	*
	*  <b> Density based clustering <\b>
	*
	*  Density based clustering first identifies the number of tiles within
	*  a specified clustering radius of a given tile. If the number of 
	*  tiles within this radius meets a threshold cluster density, all of  
	*  the tiles within this radius become part of the same cluster.  
	*  Clusters are constructed by recursively applying this same test to 
	*  all of the other tiles within the clustering radius, until a 
	*  complete set of clusters have been formed.  By default, both 
	*  clusterRadius and clusterDensity are set to unity.
	*
	*  Density based clustering does not in general assign all tiles to 
	*  clusters, but if the cluster singles argument is non zero, single
	*  tiles are included in the output cluster list.  Otherwise, single
	*  tiles are not included, and zeros are reported for their cluster 
	*  properties in the output significants structure.  By default,
	*  clusterSingles is false.
	*
	*  The significant tile structure is updated to contain the cluster 
	*  identification number and number of tiles in cluster
	*  \brief Density -based cluster finding.
	*  \param significants   Event list of significant tile properties
	*  \param distanceMetric Choice of metric to compute distance (see 
	*                        distance())
	*  \param durationInflation Multiplicative scale factor for duration 
	*                           (see distance())
	*  \param bandwidthInflation Multiplicative scale factor for bandwidth
	*                           (see distance())
	*  \param debugLevel Verbosity of debug output
	*/
      void wcluster(weventstack& significants, double clusterRadius=1.0, 
		    double clusterDensity=1.0, bool clusterSingles=false, 
		    const std::string& distanceMetric="",
		    double durationInflation=1.0, 
		    double bandwidthInflation=1.0, 
		    int debugLevel=1);


      /**  wcluster identifies clusters of related significant Q transform 
        *  tiles to improve the detection of gravitational-wave bursts that are
	*  extended in time and/or frequency.
	*
	*  <b> Hierarchical clustering <\b>
	*
	*  In hierarchical clustering, the distances returned by WDISTANCE are
	*  used by the linkage class to construct a hierarchical cluster tree 
	*  by successively linking together the next closest pair of tiles or 
	*  existing clusters.  Finally, clusters are identified by cutting the
	*  hieararchical cluster tree based on the specified criterion and
	*  threshold.
	*
	*  The cluster linkage method defines how intercluster distance is 
	*  computed, and may be any of the following methods.  By default,
	*  'single' linkage is assumed.
	*
	*  <table>
	*  <tr><td>'single'</td>
	*      <td> nearest distance between cluster members</td></tr>
	*  <tr><td>'complete'</td>
	*      <td>furthest distance between cluster members</td></tr>
	*  <tr><td>'average'</td>
	*      <td>average distance between cluster members</td></tr>
	*  <tr><td>'centroid'</td>
	*      <td>center of mass distance between clusters</td></tr>
	*  <tr><td>'ward'</td>
	*      <td>inner squared distance between clusters</td></tr>
	*  </table>
	*
	*  The cluster criterion specifies how to cut the hierarchical cluster 
	*  tree to form clusters, and may be either of the two methods 
	*  understood by the CLUSTER function.  By deafult, 'inconsistent' is 
	*  assumed.
	*  <table>
	*  <tr><td>'inconsistent'</td>
	*      <td>threshold on inconsistency between of cluster distance and
	*          average subcluster difference</td></tr>
	*  <tr><td>'distance'</td>
	*      <td>threshold on absolute distance between clusters</td></tr>
	*  </table>
	*
	*  The clusterThreshold corresponds to the specified clusterCriterion, 
	*  and has a default threshold of unity for both criteria.
	*/
      void wcluster(weventstack& significants, 
		    const std::string& clusterLinkage, 
		    const std::string& clusterCriterion, 
		    double clusterThreshold, 
		    const std::string& distanceMetric,
		    double durationInflation, double bandwidthInflation, 
		    int debugLevel);

      /**  WSELECT selects statistically significant events from the set of 
        *  statistically significant Q transform tiles. Events are defined by 
        *  the properties of their most significant tile and are identified 
	*  by exluding the less significant of any overlapping tiles.  The 
	*  input significant tiles must be sorted by decreasing normalized 
	*  energy. If the debugLevel argument is > 2, the ordering is tested
	*  and a logic_error exceptoion is thrown if the order is incorrect. 
	*  Starting with the most significant tile, tiles are discarded if 
	*  they overlap with a more significant tile.  The remaining set of 
	*  tiles comprises a minimal set of tiles that describes an event.
	*  the results are left in the current eventlist instance.
	*
	*  The optional durationInflation and bandwidthInflation arguments are
	*  multiplicative scale factors that are applied to the duration and 
	*  bandwidth of significant tiles prior to testing for overlap.  If not 
	*  specified, these parameters both default to unity such that the 
	*  resulting tiles have unity time-frequency area.  The normalized 
	*  energy of the resulting tiles are scaled by the product of the 
	*  duration and bandwidth inflation factors to avoid over counting the 
	*  total energy of clusters of tiles.  Likewise, the amplitude of the 
	*  resulting tiles is scaled by the square root of the product of the
	*  duration and bandwidth inflation factors.
	*
	*  The optional maximumEvents argument provides a safety mechanism to 
	*  limit the total number of events returned by WSELECT.  If this 
	*  maximum number of events is exceeded, an overflow flag is set, only 
	*  the maximumEvents most significant events are returned, and a warning
	*  is issued if debugLevel is set to unity or higher.  By default, 
	*  maximumEvents is set to infinity and debugLevel is set to unity.
	*
	*  \param significants       event list tiles properties. This must be
	*                            ordered by decreasing normalizedEnergy.
	*  \param durationInflation  multiplicative duration scale factor
	*  \param bandwidthInflation multiplicative bandwidth scale factor
	*  \param maximumEvents      maximum allowable number of events
	*  \param debugLevel         verbosity of debug output
	*/
      void wselect(const weventstack& significants, double durationInflation=1,
		   double bandwidthInflation=1.0, size_t maximumEvents=0, 
		   int debugLevel=1);

      /**  wthreshold identifies discrete Q transform coefficients whose 
       *  magnitudes exceed a normalized energy threshold. The calculation of
       *  a thresold that yields a specified single channel false rate assuming
       *  ideal white noise has been moved upstream.
       *  Each such tile is used to create an event which is added to the 
       *  list.
       *
       *  The user can focus on a subset of the times and frequencies available 
       *  in the transform data by specifying a desired range of central times,
       *  central frequencies, and Qs to threshold on.  Ranges should be 
       *  specified as a two component vector, consisting of a minimum and 
       *  maximum value. Alternatively, if only a single Q is specified, 
       *  WTHRESHOLD is only applied to the time-frequency plane which has the 
       *  nearest value of Q in a logarithmic sense to the requested value.
       *
       *  To determine the range of central times to threshold on, WTHRESHOLD
       *  requires the start time of the transformed data in addition to a
       *  reference time and a relative time range.  Both the start time and
       *  reference time should be specified as absolute quantities, while the
       *  range of times to analyze should be specified relative to the 
       *  requested reference time.
       *
       *  By default, WTHRESHOLD is applied to all available frequencies and Qs,
       *  and the reference time and relative time range arguments are set to
       *  exclude data potentially corrupted by filter transients as identified 
       *  by the transient duration field of the tiling structure.  The default 
       *  value can be obtained for any argument by passing an empty vector.
       *
       *  If provided, the optional analysisMode string is used by WTHRESHOLD 
       *  to determine which channels are signal channels, which channels are 
       *  null channels, and which channels to report results for.  For 
       *  coherent analysis modes, a desired veto threshold, squared 
       *  calibration uncertainy factor, and required signal correlation factor
       *  must also be specified.
       *
       *  The optional maximumSignificants argument provides a safety mechanism 
       *  to limit the total number of events returned by WTHRESHOLD.  If this 
       *  maximum number of significants is exceeded, the overflow flag is set, 
       *  only the maximumSignificants most significant tiles are returned, and 
       *  a warning is issued if debugLevel is set to 1 or higher.  By default,
       *  maximumSignificants is set to 1e9-1 and debugLevel is set to unity.
       *
       *  \brief Generate list of significant tiles.
       *  \param signal         Input signal Q transform structure
       *  \param tiling         Discrete Q transform tiling structure from WTILE
       *  \param eventThreshold desired event Normalized-E threshold
       *  \param referenceTime  reference time for time range to threshold on
       *  \param timeRange      vector range of relative times to threshold on
       *  \param frequencyRange vector range of frequencies to threshold on
       *  \param qRange         scalar Q or vector range of Qs to threshold on
       *  \param maximumSignificants maximum allowable number of significant tiles
       *  \param analysisMode      string name of analysis mode to implement
       *  \param vetoThreshold     desired veto Normalized-E threshold
       *  \param uncertaintyFactor squared calibration uncertainty factor
       *  \param correlationFactor fractional correlated energy threshold
       *  \param debugLevel        verbosity of debug output
       */
      void wthreshold(const wtransform& transforms, const wtile& tiling, 
		      double eventThreshold, 
		      const Time& referenceTime, const dble_vect& timeRange, 
		      const dble_vect& frequencyRange, const dble_vect& qRange, 
		      size_t maximumSignificants, 
		      const std::string& analysisMode="independent", 
		      double vetoThreshold=1e100, double uncertaintyFactor=0.0, 
		      double correlationFactor=0.0, int debugLevel=1);

      /**  Write the specified fields of the event list to the output file.
       */
      void writeEvents(const wouttype& out, const str_vect& fields, 
		      const std::string& format) const;
 
   private:
      typedef  std::vector<weventlist> eventlist_vect;
      eventlist_vect _lists;
   };

   //====================================  Inline methods.
   inline const weventlist& 
   weventstack::operator[](int i) const {
      return _lists[i];
   }

   inline int 
   weventstack::numberOfChannels(void) const {
      return _lists.size();
   }

}

#endif // !defined(WPIPE_WEVENTLIST_HH)
