/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "GDSPlot.hh"
#include "TCanvas.h"
#include "TColor.h"
#include "TError.h"
#include "TH1.h"
#include "TH2.h"
#include "TList.h"
#include "TBox.h"
// #include "TPaletteAxis.h"
#include "TStyle.h"
#include "fSeries/PSD.hh"
#include "TSeries.hh"
#include "DVector.hh"
#include "lcl_array.hh"

#include <iostream>
#include <stdexcept>
using namespace std;

//======================================  Constructor
GDSPlot::GDSPlot(TCanvas* tc) 
  : mCanvas(tc), mTrace(0), mXmin(0), mXmax(0), mYmin(0), mYmax(0),
    mZmin(0), mZmax(0), mLogX(false), mLogY(false), mPalletID(0),
    mNextColor(1), mNSmooth(0)
{
    gErrorIgnoreLevel=1500;
    set_style(0);
    if (!mCanvas && gPad) set_canvas(gPad->GetCanvas());
}

//======================================  Plot a data vector.
void
GDSPlot::boxes(size_t nbox, const double* x0, const double* x1, 
	       const double* y0, const double* y1, const double* z) {
    TH2D slate("slate", mTitle.c_str(), 10, mXmin, mXmax, 10, mYmin, mYmax);
    set_palette();
    slate.SetMinimum(mZmin);
    slate.SetMaximum(mZmax);
    for (size_t i=0; i<10; i++) {
	double x = mXmin + (i+0.5)/10.0 * (mXmax - mXmin);
	for (size_t j=0; j<10; j++) {
	    double y = mYmin + (j+0.5)/10.0 * (mYmax - mYmin);
	    slate.Fill(x, y, 0.0001);
	}
    }

    set_axes(slate);
    Int_t ncolors = TColor::GetNumberOfColors();
    slate.SetContour(ncolors, 0);
    if (!mTitle.empty())  slate.SetTitle(mTitle.c_str());
    TH1* cpy = slate.DrawCopy("colz");
    Int_t ndivz = abs(cpy->GetContour());
    if (ndivz == 0) throw runtime_error("GDSPlot:boxes No contours");
    Double_t scale = ndivz / (mZmax - mZmin);
    mCanvas->Update();
    //    TPaletteAxis* zax=(TPaletteAxis*)(cpy->GetListOfFunctions()
    //				      ->FindObject("palette"));
    // if (!zax) {
    //	cerr << "Yikes! Can't find the z-axis palette" << endl;
    //	throw runtime_error("GDSPlot:boxes No Z palette");
    // }
    for (size_t i=0; i<nbox; i++) {
	double z0 = z[i];
	TBox* box = new TBox(x0[i], y0[i], x1[i], y1[i]);
	box->SetBit(kCanDelete);
	//------------------------------ Note no log z values!
	if (z0 >= mZmax) z0 = mZmin + 0.99 * (mZmax - mZmin);
	if (z0 <  mZmin) z0 = mZmin;

	// box->SetFillColor(zax->GetValueColor(z0));
	//------------------------------  Replacement code stolen from 
	//                                TPaletteAxis::GetValueColor
	Int_t color = Int_t(0.01 + (z0 - mZmin) * scale);

	Int_t theColor = Int_t((color + 0.99) * Double_t(ncolors) 
			       / Double_t(ndivz));
	Int_t colorIndex = gStyle->GetColorPalette(theColor);
	box->SetFillColor(colorIndex);
	//-------------------------------------------------------
	box->Draw();
    }
    mTrace++;
}

//======================================  Plot a data vector.
void
GDSPlot::new_plot(void) {
    if (mCanvas) mCanvas->Clear();
    mTrace = 0;
    mXmin = 0;
    mXmax = 0;
    mYmin = 0;
    mYmax = 0;
    mZmin = 0;
    mZmax = 0;
    mLogX = false;
    mLogY = false;
    mNextColor = 1;
}

//======================================  Plot a PSD (log-log)
void 
GDSPlot::plot(const containers::PSD& psd) {
    xlog();
    ylog();
    plot(psd.refDVect(), psd.getLowFreq(), psd.getFStep());
}

//======================================  Plot a PSD (log-log)
void 
GDSPlot::plot(const TSeries& ts) {
    xlog(false);
    plot(*ts.refDVect(), 0.0, ts.getTStep());
}

//======================================  Plot a data vector.
void 
GDSPlot::plot(const DVector& dvec, double xmin, double dx,
	      const string& title) {
    int bMin = 0;
    int bMax = dvec.size();
    if (mLogX && xmin <= 0) {
	bMin = int(xmin / dx) + 1;
    }

  //------------------------------------  Set mXmin, mXmax, bMin, bMax
  if (mXmax <= mXmin) {
    mXmin = xmin + bMin * dx;
    mXmax = xmin + bMax * dx;
  } 
  else {
    if (mXmin > xmin + bMin * dx) bMin = int((mXmin - xmin) / dx + 0.5);
    if (mXmax < xmin + bMax * dx) bMax = int((mXmax - xmin) / dx + 0.5);
  }

  //------------------------------------  get the data to be plotted
  int nBin = bMax - bMin;
  lcl_array<double> ytemp(nBin);
  nBin = dvec.getData(bMin, nBin, ytemp);
  bMax = nBin + bMin;

  //------------------------------------  Fill a histogram
  TH1D histo("dvect", title.c_str(), nBin, xmin+dx*bMin, xmin+dx*bMax);
  for (int i=0; i<nBin; ++i) {
    histo.SetBinContent(i+1, ytemp[bMin + i]);
  }

  //------------------------------------  Optional smoothing
  if (mNSmooth) histo.Smooth(mNSmooth, mSmoothOpt.c_str());

  //------------------------------------  Draw it
  histo.SetLineColor(mNextColor);
  if (!mTrace) {
    if (mLogX) mCanvas->SetLogx(1);
    else       mCanvas->SetLogx(0);

    if (mLogY) mCanvas->SetLogy(1);
    else       mCanvas->SetLogy(0);

    mCanvas->SetTicks(1);

    set_axes(histo);
    histo.DrawCopy();
  }
  else {
    histo.DrawCopy("Same");
  }
  mTrace++;
  mNextColor = mTrace + 1;
}

//======================================  Print canvas to the specified file
void
GDSPlot::print(const std::string& file) const {
    if (!mTrace) return;
    mCanvas->Print(file.c_str());
}


//======================================  Set the canvas
void 
GDSPlot::set_axes(TH1& slate) {
    if (!mXlabel.empty()) {
	TAxis* axis = slate.GetXaxis();
	axis->CenterTitle(kTRUE);
	slate.SetXTitle(mXlabel.c_str());
    }

    if (!mYlabel.empty()) {
	TAxis* axis = slate.GetYaxis();
	axis->CenterTitle(kTRUE);
	slate.SetYTitle(mYlabel.c_str());
    }

    if (!mZlabel.empty()) {
	TAxis* axis = slate.GetZaxis();
	axis->CenterTitle(kTRUE);
	slate.SetZTitle(mZlabel.c_str());
    }
}

//======================================  Set the canvas
void 
GDSPlot::set_canvas(TCanvas* tc) {
  mCanvas = tc;
}

//======================================  Set color of the next plotted trace
void 
GDSPlot::set_color(int tc) {
    mNextColor = tc;
}

//======================================  Set the color palette
void
GDSPlot::set_palette(void) {
    if (!mPalletID) {
	//  John's original guess
	//UInt_t Number = 6;
	//Double_t Red[6]    = { 0.00, 0.00, 0.70, 1.00, 1.00, 0.50};
	//Double_t Green[6]  = { 0.00, 0.00, 0.90, 1.00, 0.00, 0.25};
	//Double_t Blue[6]   = { 0.40, 1.00, 1.00, 0.40, 0.00, 0.00};
	//Double_t Length[6] = { 0.00, 0.10, 0.32, 0.60, 0.99, 1.00};
	//  Based on matlab rgb definitions
	UInt_t Number = 6;
	Double_t Red[6]    = {0.000, 0.000, 0.000, 1.000, 1.000, 0.500};
	Double_t Green[6]  = {0.000, 0.000, 1.000, 1.000, 0.000, 0.000};
	Double_t Blue[6]   = {0.500, 1.000, 1.000, 0.000, 0.000, 0.000};
	Double_t Length[6] = {0.000, 0.125, 0.375, 0.625, 0.875, 1.000};
	Int_t nb=255;
	TColor::CreateGradientColorTable(Number,Length,Red,Green,Blue,nb);
	mPalletID = 1;
    }
}

//======================================  Set the Axis ranges
void 
GDSPlot::set_range(double xmin, double xmax, double ymin, double ymax) {
    mXmin = xmin;
    mXmax = xmax;
    mYmin = ymin;
    mYmax = ymax;
}

//======================================  Set the print size
void 
GDSPlot::set_size(double xsize, double ysize) const {
    mCanvas->Size(xsize, ysize);
}

//======================================  Set the smoothing options
void 
GDSPlot::set_smooth(int nSmooth, const std::string& smoothOpt) {
    mNSmooth   = nSmooth;
    mSmoothOpt = smoothOpt;
}

//======================================  Set the plot style
void 
GDSPlot::set_style(int id) {
    switch (id) {

    //----------------------------------  Standard plot style
    case 0:
	//------------------------------  Turn off plot statistics
	gStyle->SetOptStat(0);

	//------------------------------  Canvas
	gStyle->SetCanvasBorderMode(0); //  No borders, white background
	gStyle->SetCanvasColor(0);      //  white background

	//------------------------------  Pads:
	gStyle->SetPadBorderMode(0);    //  No borders, white background
	gStyle->SetPadColor(0);         //  White background
	gStyle->SetPadRightMargin(0.15);//  Leave space for z-axis title.

	//------------------------------  Plot Title
	gStyle->SetTitleX(0.5);         //  Centered title
	gStyle->SetTitleAlign(23);      //  
	gStyle->SetTitleBorderSize(0);  //  No borders/shadows
	gStyle->SetTitleFillColor(0);   //  White background.
	gStyle->SetTitleSize(0.08, "u");// Size = 0.08 for title

	//------------------------------  Label text
	gStyle->SetTitleSize(0.05, "xyz"); // Size = 0.05 for all labels
	break;
    default:
	cerr << "Unrecognized graphics style: " << id << endl;
	break;
    }
}

//======================================  Set the Axis ranges
void 
GDSPlot::set_zrange(double zmin, double zmax) {
    mZmin = zmin;
    mZmax = zmax;
}

//======================================  Plot a surface
void 
GDSPlot::surf(int nx, const double* x, int ny, const double* y, 
	      const double* z) {
    TH2D xyplot("xyplot", mTitle.c_str(), nx, x, ny, y);
    int inx = 0;
    for (int j=0; j<ny; j++) {
	double y0 = 0.5 * (y[j] + y[j+1]);
	for (int i=0; i<nx; i++) {
	    double x0 = 0.5 * (x[i] + x[i+1]);
	    xyplot.Fill(x0, y0, z[inx++]);
	}
    }

    //---------------------------------- Optional smoothing
    if (mNSmooth) xyplot.Smooth(mNSmooth, mSmoothOpt.c_str());

    if (mLogX) mCanvas->SetLogx(1);
    else       mCanvas->SetLogx(0);
    if (mLogY) mCanvas->SetLogy(1);
    else       mCanvas->SetLogy(0);
    mCanvas->SetTicks(1, 1);

    set_palette();
    if (mZmax > mZmin) {
	xyplot.SetMinimum(mZmin);
	xyplot.SetMaximum(mZmax);
    }
    if (!mTitle.empty())  xyplot.SetTitle(mTitle.c_str());
    set_axes(xyplot);
    xyplot.SetContour(TColor::GetNumberOfColors(), 0);
    xyplot.DrawCopy("colz");
    mTrace++;
}

//======================================  Set canvas title
void 
GDSPlot::title(const std::string& titl) {
    mTitle = titl;
}

//======================================  Set the X-axis label
void 
GDSPlot::xlabel(const std::string& titl) {
    mXlabel = titl;
}

//======================================  Set X-axis log scale
void 
GDSPlot::xlog(bool logx) {
    mLogX = logx;
}

//======================================  Time unit ranges
static const double millisecondThreshold = 0.5;
static const double secondThreshold = 3 * 60;
static const double minuteThreshold = 3 * 60 * 60;
static const double hourThreshold = 3 * 24 * 60 * 60;
static const double dayThreshold = 365.25 * 24 * 60 * 60;

//======================================  Set x time scale in reasonable units
double 
GDSPlot::xTimeScale(double dT, const std::string& xttl) {
    double tScale = 1.0;
    string tUnits;
    if (dT < millisecondThreshold) {
      tScale = 0.001;
      tUnits = " [milliseconds]";
    }
    else if (dT < secondThreshold) {
      tScale = 1.0;
      tUnits = " [seconds]";
    }
    else if (dT < minuteThreshold) {
      tScale = 60;
      tUnits = " [minutes]";
    }
    else if (dT < hourThreshold) {
      tScale = 3600;
      tUnits = " [hours]";
    }
    else if (dT < dayThreshold) {
      tScale = 86400;
      tUnits = " [days]";
    }
    else {
      tScale = 31557600;
      tUnits = "years";
    }
    xlabel(xttl + tUnits);
    return tScale;
}


//======================================  Set the Y-axis label
void 
GDSPlot::ylabel(const std::string& titl) {
    mYlabel = titl;
}

//======================================  Set Y-axis log scale
void 
GDSPlot::ylog(bool logy) {
    mLogY = logy;
}

//======================================  Set the Z-axis label
void 
GDSPlot::zlabel(const std::string& titl) {
    mZlabel = titl;
}
