//
//    Usage: DEnvCorr configuration_file
//
//    For details of the configuration file structure see DInitCorr.cc
//    Note DCorrInit must be run before running DEnvCorr
//

#include <time.h>
#include "DEnvCorr.hh"

const int CINT_MAX_CHANS=10;
const int CINT_MAX_PLOT_POINTS=1024;
// We need to pick out reasonably distinct colors!
// const int N_ROOT_COLORS=18
// const int root_colors[18]={2,3,4,5,6,7,8,9,11,25,28,30,35,38,41,43,46,50}; 
// NB This lines which should logically appear here actually
// appear below to overcome a shortcoming of CINT

class DEnvCorr : public DatEnv {
  public:
  DEnvCorr(int argc, const char* argv[]);
  ~DEnvCorr();
  void ProcessData(void);
private:
  int correlationWidth;
  int nPlot;
  float detailMin;
  float signalRate;
  TCanvas *c;
  TPad *corrPad[3];
  TPad *spectrumPad[2];
#ifndef __CINT__
  TH1F **corrGraph;
#else 
  TH1F *corrGraph[CINT_MAX_CHANS];
#endif
  TH1F *spectrumGraph[2];
};     

void
EnvCorr(const char* config) {
    DEnvCorr xxx(1, &config);
    xxx.MainLoop();
}

using namespace std;

//--------------------------------------  Define the main program in background
#ifndef __CINT__
EXEROOT(DEnvCorr);
#endif

DEnvCorr::DEnvCorr(int argc, const char* argv[]) 
  :DatEnv(argc, argv), correlationWidth(64), detailMin(0.75) {
  char descriptor[256];
  float threshold,modx2sum,delta_f,freq,time;
  int i,clean,offset,nChan,nEnvChan,length;


#ifndef __CINT__
  gErrorIgnoreLevel=999;  
#endif


  //****************************************************************//
  //                       Read the configuration file              //
  //****************************************************************//
 
  const char* config;

#ifndef __CINT__
  if (argc !=2 ) {
    cerr<< "# EnvCorr: interactive usage DEnvCorr configuration_file " << endl;
    return;
  }
#endif


  config = argv[argc-1];
  if (!config || !*config) {
    cerr<< "# EnvCorr: configuration file not specified" << endl;
    return;
  }

  //---------------------------------  Get parameters.
  ifstream ifs(config, ios::in);
  ifsopen_check(ifs,config);

  cerr << "# EnvCorr: reading parameters from config file: " << config << endl;
   
  //   Comments in the configuration file start with a # 
  //   The first line after any comments should contain 
  //   a string characterising the run
  while (1) {
    ifs.getline(descriptor, sizeof(descriptor));
    if (descriptor[0] != '#') break;
  }  

  //  the next line gives the total number of channels including the `signal'
  ifs >> nChan; 
  cerr << "# EnvCorr: total number of channels (including signal)  " << nChan <<endl;
  //  then the signal rate
  ifs >> time;
  cerr << "# EnvCorr: Sample time = " << time << endl;


#ifndef __CINT__
  Chan *chanv;
  chanv=new Chan[nChan];
#else
  if (nChan > CINT_MAX_CHANS) {
    cerr << "# EnvCorr: CINT can only handle a maximum of CINT_MAX_CHANS = " <<
	 CINT_MAX_CHANS << " channels" << endl;
    cerr << "         Run outside root or increase CINT_MAX_CHANS at head of DEnvCorr.cc" << endl;
    abort();
  }  
  Chan chanv[CINT_MAX_CHANS];
#endif

 
  // the next nChan lines of the configuration file should contain 3 columns
  //                  - if not print an error message           
  // the 3 columns are the channel name,the data type and the number of points
  //     the data type is important for corr_init but is irrelevant here   
  for (int id=0;id<nChan;id++) {
    ifs >> chanv[id].mName >> chanv[id].mRate;
    chanv[id].mPoint=int(ceil(time*chanv[id].mRate));
    cerr << "# EnvCorr: Added channnel " << chanv[id].mName << "\t " << chanv[id].mPoint << endl;
  }

  //      do we want to calculate (the fft of) the `cleaned' signal? 
  //    clean should be set to 1 if we want to calculate the cleaned signal  
  ifs >> clean;

  ifs.close();

  //******************************************************************//
  //        Determine the number of frequency bins we will use        // 
  //******************************************************************//

  length=chanv[0].mPoint;
  for(int id=1;id<nChan;id++) {
     if (chanv[id].mPoint<length) length=chanv[id].mPoint;
  }

  length/=2;   // because we are creating separate arrays for the real and imaginary parts

  if ( (length%correlationWidth) != 0 ) {
      length-=length%correlationWidth;
  }
  cerr << "# EnvCorr: Using " << length << 
    " FFT values (correlationWidth = " << correlationWidth << ")" << endl;

  nPlot=length/correlationWidth;

  nEnvChan=nChan-1;   // because the configuration file includes the signal channel  

  //***************************************************************//
  //                         Allocate memory                       //
  //***************************************************************//

  // Allocate memory for |x_i|^2,|y_i|^2 and x_i y_i*   
  float *rp_signal,*ip_signal;
  rp_signal=new float[length];
  ip_signal=new float[length];
  typedef float* float_ptr;
#ifndef __CINT__
  float **rp_env,**ip_env;
  rp_env=new float_ptr[nEnvChan];
  ip_env=new float_ptr[nEnvChan];
#else
  float *rp_env[CINT_MAX_CHANS-1],*ip_env[CINT_MAX_CHANS-1];
#endif

  rp_env[0]=new float[nEnvChan*length];
  for (int i=1;i<nEnvChan;i++)
    rp_env[i]=rp_env[i-1]+length;

  ip_env[0]=new float[nEnvChan*length];
  for (int i=1;i<nEnvChan;i++)
    ip_env[i]=ip_env[i-1]+length;

  float *rp_clean(0), *ip_clean(0);
  if (clean) { 
    rp_clean=new float[length];
    ip_clean=new float[length];
  }

  float *rho2;
  rho2=new float[length];
#ifndef __CINT__
  float **rho2_pairwise;
  rho2_pairwise=new float_ptr[nPlot];
#else
  if (nPlot <= CINT_MAX_PLOT_POINTS) {
    float *rho2_pairwise[CINT_MAX_PLOT_POINTS];
   }
  else
    {
	 cerr << "# EnvCorr" << ": CINT can only handle a maximum of CINT_MAX_PLOTS_POINTS = " <<	 CINT_MAX_PLOT_POINTS << " points" << endl;
	 cerr << "         Note: Number of plot points = (number of data points)/(correlation width)" << endl;
	 cerr << "         Run outside root or increase CINT_MAX_PLOT_POINTS at head of DEnvCorr.cc" << endl;
	 abort();
    }
#endif

  for (int i=0;i<nPlot;i++) rho2_pairwise[i]=new float[nEnvChan];

  //***************************************************************//
  //                 Allocate memory for lapack arrays             //
  //***************************************************************//


  complx *A=NULL, *B=NULL; 
  A=new complx[nEnvChan*nEnvChan];
  B=new complx[nEnvChan];


  complx *R=NULL,*work=NULL;
  integer lwork=0, *ipivot=NULL;
  if (clean) {
    R=new complx[nEnvChan];
    lwork=(integer) nEnvChan;
    work=new complx[lwork];  
    ipivot=new integer[nEnvChan];
  } 

  //****************************************************************//
  //****************************************************************//
  //                    Read data from file                         //
  //****************************************************************//
  //****************************************************************//

  //****************************************************************//
  //              Determine name of data directory                  //
  //****************************************************************//
  char fft_dir[256];
  sprintf(fft_dir,"%s_fft",descriptor);

  //****************************************************************//
  //                      Read data from signal channel             //   	  
  //****************************************************************//
  char fname[256];
  sprintf(fname,"%s/%s_fft.b",fft_dir,chanv[0].mName);
  read_binary_fft(fname,length,rp_signal,ip_signal,&delta_f);

  cerr << "# EnvCorr: delta_f = " << delta_f << endl;

  //*****************************************************************//
  //                 Read data from environmental channels           //
  //*****************************************************************//
  for (int id=0;id<nEnvChan;id++) { 
    sprintf(fname,"%s/%s_fft.b",fft_dir,chanv[id+1].mName);
    read_binary_fft(fname,length,rp_env[id],ip_env[id],&delta_f);
  } 

  threshold=((0.1) < (5.0/correlationWidth) ? (0.1) : (5.0/correlationWidth) ); // MIN - see discussion in Hua et al. 

  char rho2_fname[256];
  sprintf(rho2_fname,"%s/rho2_%s_%d.dat",fft_dir,chanv[0].mName,correlationWidth);

  ofstream ofs(rho2_fname, ios::out);
  if (ofs.bad()) {
    cerr << "# EnvCorr: unable to open output file " << rho2_fname  << endl;
    return;
  }
  ofs.setf(ios::fixed,ios::floatfield);
  ofs.precision(3);
  cerr << "# EnvCorr: Writing " << rho2_fname << endl;

  //***************************************************************//
  //         Step through the range of offsets and call the        // 
  //       function calc_rho where the major calculation is done   //
  //***************************************************************//

  for (int iOffset=0;iOffset<nPlot;iOffset++) {

    offset=iOffset*correlationWidth;
    calc_rho(offset,correlationWidth,threshold,rp_signal,ip_signal,nEnvChan,
		   rp_env,ip_env,rho2_pairwise[iOffset],A,B,&modx2sum);

    //*************************************************************//
    //             Calculate the `cleaned' signal                  //
    //*************************************************************//

    rho2[iOffset]=0.0; // set rho2=0.0 if we don't clean 
    if (clean) {
	 chan_clean(offset,correlationWidth,threshold,&(rho2[iOffset]),rp_signal,ip_signal,nEnvChan,
			  rp_env,ip_env,rp_clean,ip_clean,A,B,modx2sum,R,work,lwork,ipivot);
    }

    //*************************************************************//
    // Print out |rho|^2 and the level of *pairwise* correlation   //
    // between the signal and each environmental channel in turn   //
    //*************************************************************//

    freq=(offset+0.5*correlationWidth)*delta_f;
    write_rho2(ofs,freq,rho2[iOffset],nEnvChan,rho2_pairwise[iOffset]);

  } // end of iOffset loop 

  ofs.close();


    
  //****************************************************************//
  //           Set up and display ROOT graphics                     //
  //****************************************************************//

  gROOT->Reset();
  gStyle->SetOptTitle(0); //Don't show title
  gStyle->SetOptStat(0); // Don't show statistics
  gROOT->SetInterrupt(kTRUE);

  c = new TCanvas(descriptor, "Interchannel Correlations",10,10,788,750); 
  c->SetHighLightColor(2);
  c->SetFillColor(33); // Blue-grey in the default ROOT palette
  c->SetBorderSize(2); 
  c->SetLeftMargin(0.12); 
  c->SetRightMargin(0.0);
  c->SetTopMargin(0.08); 
  c->SetBottomMargin(0.12); 
  c->SetEditable();
 
  corrPad[0]=new TPad("Pad0","Fullscale",0.025,0.6,0.975,0.99,21);
  corrPad[1]=new TPad("Pad1","Detail",0.025,0.2,0.975,0.57,21);
  corrPad[2]=new TPad("Pad2","Description",0.0,0.0,1.0,0.19,10);
  corrPad[0]->Draw();
  corrPad[1]->Draw();
  corrPad[2]->Draw();

  corrPad[2]->cd();
  TPaveText *corrTitle = new TPaveText(0.025, 0.07, 0.975, 0.33,"NDC");
  corrTitle->SetFillColor(42);
  corrTitle->SetMargin(0.025);
  corrTitle->SetTextAlign(12);
  char title[128]="Interchannel Correlations with ";
  strcat(title,chanv[0].mName);
  corrTitle->AddText(title);       
  corrTitle->Draw();

  const int N_ROOT_COLORS=18;
  const int root_colors[N_ROOT_COLORS]={2,3,4,5,6,7,8,9,11,25,28,30,35,38,41,43,46,50}; 


  {Float_t xText,yText;
  for (int id=0;id<nEnvChan;id++) {
    xText=0.05+0.24*(id-id%4)/4;
    yText=0.85-0.15*(id%4);
    TText *tt = new TText(xText,yText,chanv[id+1].mName);
    tt->SetTextColor(root_colors[id]%N_ROOT_COLORS);
    tt->SetTextSize(0.125);
    tt->Draw();
  }
  }

#ifndef __CINT__
  typedef TH1F* TH1F_ptr;
  corrGraph=new TH1F_ptr[nEnvChan]; 
#endif 

  for (int iPad=0;iPad<2;iPad++) {
    corrPad[iPad]->cd(); 
    corrPad[iPad]->SetFillColor(10); 
    corrPad[iPad]->SetBorderMode(0);
    corrPad[iPad]->SetGrid(); 
    corrPad[iPad]->SetTicks();
    corrPad[iPad]->SetFrameFillColor(10);
    corrPad[iPad]->Draw();

    for (int id=0;id<nEnvChan;id++) {
	 corrGraph[id]=new TH1F(chanv[id+1].mName,chanv[id+1].mName,nPlot,0.0,length*delta_f);
	 corrGraph[id]->SetMaximum(1.0);
	 if (iPad==0)
	   corrGraph[id]->SetMinimum(0.0);
	 else	 
	   corrGraph[id]->SetMinimum(detailMin);
	 corrGraph[id]->SetLineColor(id+2);     
	 for (i=0;i<nPlot;i++) {
	   corrGraph[id]->Fill((i+0.5)*correlationWidth*delta_f,rho2_pairwise[i][id]);
	 }
	 if (id==0) 
	   corrGraph[id]->Draw("L"); // Draw points
	 else
	   corrGraph[id]->Draw("Lsame"); // Draw points
    }

    corrPad[iPad]->Modified();
    corrPad[iPad]->Draw();
    c->Update(); 
  }

  //  Leave graph up until signalled to quit
  char term[128]="n";
  while (strcmp(term,"q")) {
    cerr << "Type q<CR> to quit" << endl << flush;
    cin >> term;
  }

  if (clean) {
    sprintf(fname,"%s/fftclean_%s_%d.dat",fft_dir,chanv[0].mName,correlationWidth);
    write_fft(fname,length,rp_clean,ip_clean,delta_f); 
  
    c->cd();
    spectrumPad[0]=new TPad("SpectrumPad","Fullscale",0.025,0.1,0.975,0.99,21);
    spectrumPad[1]=new TPad("SpectrumPad","Title",0.0,0.0,1.0,0.1,21);
    spectrumPad[0]->Draw();
    spectrumPad[1]->Draw();

    spectrumPad[0]->cd(); 
    spectrumPad[0]->SetLogy();
    spectrumPad[0]->SetFillColor(10); 
    spectrumPad[0]->SetFrameFillColor(10);
    spectrumPad[0]->SetGrid(); 
    spectrumPad[0]->SetTicks();

    spectrumGraph[0]=new TH1F(chanv[0].mName,chanv[0].mName,length,0.0,length*delta_f);
    spectrumGraph[1]=new TH1F(chanv[0].mName,chanv[0].mName,length,0.0,length*delta_f);
    for (i=0;i<length;i++) {
	 spectrumGraph[0]->Fill((i+0.5)*delta_f,rp_signal[i]*rp_signal[i]+ip_signal[i]*ip_signal[i]);
	 spectrumGraph[1]->Fill((i+0.5)*delta_f,rp_clean[i]*rp_clean[i]+ip_clean[i]*ip_clean[i]);
    }

    spectrumGraph[0]->SetLineColor(4);     
    spectrumGraph[1]->SetLineColor(2);     
    spectrumGraph[0]->Draw("L");
    spectrumGraph[1]->Draw("Lsame");
    spectrumPad[0]->Modified();
    spectrumPad[0]->Draw();

    spectrumPad[1]->cd();
    spectrumPad[1]->SetFillColor(10); 
    spectrumPad[1]->SetFrameFillColor(10);
    spectrumPad[1]->SetBorderMode(0);
    TPaveText *spectrumTitle = new TPaveText(0.025, 0.25, 0.975, 0.75,"NDC");
    spectrumTitle->SetFillColor(42);
    spectrumTitle->SetMargin(0.025);
    spectrumTitle->SetTextAlign(12);
    sprintf(title,"Cleaned spectrum for %s",chanv[0].mName);
    spectrumTitle->AddText(title);       
    spectrumTitle->Draw();    // tt->Delete();
    spectrumPad[1]->Modified();
    spectrumPad[1]->Draw();

    c->Update();
  }

#ifndef __CINT__
  //  Leave graph up until signalled to quit
  strcat(term,"n");
  while (strcmp(term,"q")) {
    cerr << "Type q<CR> to quit" << endl << flush;
    cin >> term;
  }

  //    Free memory            

  delete [] chanv;

  for (int id=0;id<nPlot;id++) {
    delete [] rho2_pairwise[id];
  }
  delete [] rho2_pairwise;
  
  delete [] rp_signal;
  delete [] ip_signal;
  delete [] rp_env[0];
  delete [] rp_env;
  delete [] ip_env[0];
  delete [] ip_env;
  delete [] rp_clean;
  delete [] ip_clean;
 
  // free lapack arrays 
  delete [] A;
  delete [] B;
  delete [] R;
  delete [] work;
  delete [] ipivot; 
#endif

}

void DEnvCorr::ProcessData() {
 finish();  // we have put DEnvCorr into DatEnv class only for uniformity and
            // to allow for future developments
}

DEnvCorr::~DEnvCorr() {

   //---------------------------------  Clean up at the end
    cout << "EnvCorr finished" << endl;
}
