/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "seg_stack.hh"
#include <sstream>
#include <iomanip>
#include <stdexcept>
#include <cstdlib>

using namespace std;

//======================================  Segment token
seg_token::seg_token(void) 
  : mValue("<empty>"), mNumeric(0), mType(kNone)
{
}

//======================================  Segment constructor
seg_token::seg_token(const LockSegList& s)
  : mValue(s), mNumeric(0), mType(kSegment)
{}

//======================================  Numeric constructor
seg_token::seg_token(double d)
  : mValue("<empty>"), mNumeric(d), mType(kNumeric)
{}

//======================================  String stack class
seg_token::~seg_token(void) 
{}

//======================================  String stack class
seg_token* 
seg_token::clone(void) const {
    return new seg_token(*this);
}

eval_token::datype
seg_token::getType() const {
    return mType;
}

double
seg_token::getNumeric() const {
    if (mType != kNumeric) throw runtime_error("Not a valid number");
    return mNumeric;
}

string
seg_token::getString() const {
    ostringstream ostr;
    switch (mType) {
    case kNumeric:
      ostr << mNumeric;
      break;
    case kSegment:
      {
	ostr << mValue.getListID() << " {";
	unsigned int N = mValue.size();
	if (N > 10) N=10;
	for (unsigned int i=0; i<N; ++i) {
	  const LockSegment& seg(mValue[i]);
	  ostr << " " << seg.getStartTime().getS() << ":" << seg.getDuration();
	}
	if (N < mValue.size()) ostr << " ... ";
	ostr << "}" << endl;
      }
      break;
    default:
      break;
    }
    return ostr.str();
}

void 
seg_token::setNumeric(double val) {
    mNumeric = val;
    mType  = kNumeric;
}

void
seg_token::setString(const string& str) {
    mValue.clear();
    char* strip=const_cast<char*>(str.c_str());
    LockSegment::id_type id(0);
    while (*strip) {
        long gps_start = strtol(strip, &strip, 0);
	if (!gps_start || *strip != ':') break;
        long duration  = strtol(strip, &strip, 0);
	if (!duration) break;
	mValue.insert(LockSegment(id++, Time(gps_start), Interval(duration)));
    }
}

//======================================  String stack class
seg_stack::seg_stack(void) {
}

//======================================  Eval stack Dump
void 
seg_stack::dump(std::ostream& out) const {
    out << "-----------------  Stack Dump, size=" << size() << endl;
    for (unsigned int inx=size(); inx; ) {
	out << --inx << "  ";
	unsigned int ioff = size() - inx;
        switch (type(ioff)) {
	case seg_token::kNone:
	  out << "None    " << endl;
	  break;
	case seg_token::kSegment:
	  out << "Segment  " << endl;
      	  break;
	case seg_token::kNumeric:
	  out << "Numeric " << peek(ioff).getNumeric() << endl;
      	  break;
	}
    }
}

//======================================  Perform operation on the top 
//                                        element(s) of the stack.
void 
seg_stack::evaluate(OpsEnum op) {
    switch (op) {

    //----------------------------------  Segment Or.
    case kOr:
      if (numeric(1) || numeric(2)) {
	  throw runtime_error("Operation not defined for numeric arguments");
      } else {
	  seg_token* t = dynamic_cast<seg_token*>(pop_token());
	  dynamic_cast<seg_token&>(top()).refSegment() |= t->refSegment();
	  delete t;
      }
      break;

    //----------------------------------  Segment And.
    case kAnd:
      if (numeric(1) || numeric(2)) {
	  throw runtime_error("Operation not defined for numeric arguments");
      } else {
	  seg_token* t = dynamic_cast<seg_token*>(pop_token());
	  dynamic_cast<seg_token&>(top()).refSegment() &= t->refSegment();
	  delete t;
      }
      break;

    //----------------------------------  Numeric operations.
    case kAdd:
      if (!numeric(1) || !numeric(2)) {
	  throw runtime_error("Operation not defined for segment arguments");
      } else {
	  push(pop_numeric() + pop_numeric());
      }
      break;

    case kSub:
      if (!numeric(1) || !numeric(2)) {
	  throw runtime_error("Operation not defined for segment arguments");
      } else {
	  double arg = pop_numeric();
	  push(pop_numeric() - arg);	  
      }
      break;

    //----------------------------------  Undefined operations.
    case kMpy:
    case kDiv:
    case kModulo:
    case kLogAnd:
    case kLogOr:
    case kCmpEq:
    case kCmpNE:
    case kCmpLt:
    case kCmpLE:
    case kCmpGt:
    case kCmpGE:
      throw runtime_error(string("Undefined operation: ")+getOpSym(op));
      break;

    //----------------------------------  Invalid opcode.
    default:
        throw logic_error("Invalid opcode in evaluate");
    }
}

//======================================  Push a numeric value
void 
seg_stack::push(double val) {
    seg_token* t = new seg_token(val);
    push(t);
}

//======================================  Push a string value.
void
seg_stack::push(const string& str) {
    seg_token tk;
    tk.setString(str);
    push(tk);
}
