/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef XSIL_PARAM_HH
#define XSIL_PARAM_HH

#include "xsil/xobj.hh"

namespace xsil {
    class Xwriter;

/**  xsil::param defines parameter objects in an xsil structure.
  *  @memo %XSIL parameter class.
  *  @author  J. Zweizig
  *  @version 1.1; Modified December 13, 1999
  *  @ingroup IO_xsil
  */
class param : public xobj {
public:
    /**  Create an unnamed, untyped parameter.
      *  @memo Default constructor.
      */
    param(void);

    /**  Create a constructor with the specified name, type, and units. 
      *  Any of the arguments may be set to a null string by coding 
      *  0 instead of a character string pointer.
      *  @memo Param constructor.
      *  @param Name Pointer to parameter name string.
      *  @param Type Pointer to parameter type string.
      *  @param Unit Pointer to parameter unit string.
      *  @param Value Pointer to the parameter value string.
      */
    param(const char* Name, const char* Type=0, const char* Unit=0, 
	  const char* Value = 0);

    /**  Create a constructor with the specified name, type, and units,
      *  and a numeric value. Any of the string arguments may be set to a 
      *  null by coding 0 instead of a character string pointer.
      *  @memo Param constructor.
      *  @param Name Pointer to parameter name string.
      *  @param Unit Pointer to parameter unit string.
      *  @param Value Parameter value.
      */
    param(const char* Name, const char* Unit, double Value);

    /**  Create a constructor with the specified name, type, and units,
      *  and a numeric value. Any of the string arguments may be set to a 
      *  null by coding 0 instead of a character string pointer.
      *  @memo Param constructor.
      *  @param Name Pointer to parameter name string.
      *  @param Unit Pointer to parameter unit string.
      *  @param Value Parameter value.
      */
    param(const char* Name, const char* Unit, int Value);

    /**  Parameter destructor.
      *  @memo Destructor.
      */
    ~param(void);

    /**  The parameter object is translated to XML and written to an STL 
      *  output stream.
      *  @memo Write parameter definition to an XML output file.
      *  @param xout XML output writer on which the parameter is to be written.
      */
    void Spew(Xwriter& xout) const;

    /**  An new parameter object with identical contents to the current 
      *  %param is constructed and a pointer to the object is returned.
      *  @memo Clone a parameter definition.
      *  @return Pointer to the clone.
      */
    param* Clone(void) const;

    /**  Mandatory xobj identification function.
      *  @return Pointer to the object type string.
      */
    const char* getObjType(void) const;

    /**  The parameter unit attribute is returned as a string.
      *  @memo Get the parameter units.
      *  @return Pointer to the parameter units string.
      */
    const char* getUnit(void) const;

    /**  The parameter value is returned as a string.
      *  @memo Get the parameter value.
      *  @return Pointer to the parameter value string.
      */
    const char* getValue(void) const;

    /**  The parameter units attribute is set to the argument string.
      *  No action is taken if \a Unit is a null pointer.
      *  @memo Set the parameter units.
      *  @param Unit Pointer to the parameter unit string.
      */
    void setUnit(const char* Unit);

    /**  The parameter value string is set to the value of the argument.
      *  @memo Set the parameter to a string value.
      *  \param Value New value for string parameter.
      */
    void setValue(const std::string& Value);

    /**  The argument is converted to a string and copied in to the 
      *  parameter value string.
      *  @memo Set the parameter to a numeric value.
      *  \param Value New value for numeric parameter.
      */
    void setValue(double Value);

    /**  The argument is converted to a string and copied in to the 
      *  parameter value string.
      *  @memo Set the parameter to an unsigned integer value.
      *  \param Value New value for unsigned int parameter.
      */
    void setValue(unsigned int Value);

    /**  The argument is converted to a string and copied in to the 
      *  parameter value string.
      *  @memo Set the parameter to an integer value.
      *  \param Value New value for int parameter.
      */
    void setValue(int Value);

    /**  The parameter value string is set to the value of the argument.
      *  \brief Set the parameter to a string value.
      *  \param Value New value for string parameter.
      */
    void setValue(const char* Value);

private:
    /**  Parameter units stored as a string.
      */
    std::string mUnit;

    /**  Parameter value stored as a string.
      */
    std::string mValue;
};

}

#endif  //  XSIL_PARAM_HH
