/* -*- mode: c++; c-basic-offset: 4; -*- */
#include <sys/types.h>
#include "xsil/MetaIO.hh"

#include <cstdio>
#include <iostream>
#include <sstream>
#include <unistd.h>

extern "C" {
#if HAVE_CPLUSPLUS_COMPATABLE_METAIO_H
  #include "metaio.h"
#else
  #include "metaio-cpp.h"
#endif /* HAVE_METAIO_H */
}

//======================================  Set a reasonable default version
//
//   A major change was made to the api
#ifndef METAIO_VSN
#warning libmetaio version not specified, 7.3 assumed
#define METAIO_VSN 703
#endif

using namespace std;

//======================================  Default  constructor
xsil::MetaIO::MetaIO(void) 
  : mParseEnv(0)
{
}

//======================================  File constructor
xsil::MetaIO::MetaIO(const char* file, const char* table) 
  : mParseEnv(0)
{
    if (table && !*table) table = 0;
    open(file, table);
}

//======================================  MetaIO destructor
xsil::MetaIO::~MetaIO(void) {
    close();
}

//======================================  Open an xml file
int
xsil::MetaIO::open(const char* file, const char* table) {
    //----------------------------------  Does file exist?
    if (access(file, R_OK)) {
	return -1;
    }
    if (!mParseEnv) {
        mParseEnv = new MetaioParseEnvironment;
	if (!mParseEnv) return -1;
    }
    int rc = MetaioOpenFile(mParseEnv, file);
    if (!rc) rc = selectTable(table);
    if (rc) {
        MetaioAbort(mParseEnv);
        delete mParseEnv;
	mParseEnv = 0;
	//cerr << "MetaIO:open Failed to open file: " << file << ", rc=" 
	//     << rc << endl;
    }
    return rc;
}

//======================================  Close the xml table file
int
xsil::MetaIO::close(void) {
    if (!is_open()) return 1; 
    int rc = MetaioClose(mParseEnv);
    delete mParseEnv;
    mParseEnv = 0;
    // cerr << "Closed MetaIO file" << endl;
    return rc;
}

//======================================  get float value
int
xsil::MetaIO::getNColumn(void) const {
    if (!is_open()) return 0;
    return mParseEnv->ligo_lw.table.numcols;
}

//======================================  get float value
const char*
xsil::MetaIO::getTypeName(int inx) const {
    if (!is_open()) return 0;
    const char* r(0);
    switch (mParseEnv->ligo_lw.table.col[inx].data_type) {
    case METAIO_TYPE_INT_4S:
	r = "int_4s";
	break;
    case METAIO_TYPE_INT_8S:
	r = "int_8s";
	break;
    case METAIO_TYPE_INT_8U:
	r = "int_8u";
	break;
    case METAIO_TYPE_INT_2S:
	r = "int_2s";
	break;
    case METAIO_TYPE_REAL_4:
	r = "real_4";
	break;
    case METAIO_TYPE_REAL_8:
	r = "real_8";
	break;
    case METAIO_TYPE_ILWD_CHAR:
        r = "ilwd:char";
	break;
    case METAIO_TYPE_ILWD_CHAR_U:
        r = "ilwd:char_u";
	break;
    case METAIO_TYPE_LSTRING:
        r = "string";
	break;
    default:
        r = 0;
    }
    return r;
}

//======================================  get float value
const char*
xsil::MetaIO::getColumnName(int inx) const {
    if (!is_open()) return 0;
    return MetaioColumnName(mParseEnv, inx);
}

//======================================  get float value
double
xsil::MetaIO::getFloat(const char* cname, double def) const {
    if (!is_open()) return def;
    int irow = MetaioFindColumn(const_cast<MetaioParseEnvironment*>(mParseEnv), 
				cname);
    if (irow < 0 || !mParseEnv->ligo_lw.table.elt[irow].valid) return def;

    double r;
    switch (mParseEnv->ligo_lw.table.col[irow].data_type) {
    case METAIO_TYPE_REAL_4:
        r = mParseEnv->ligo_lw.table.elt[irow].data.real_4;
	break;
    case METAIO_TYPE_REAL_8:
        r = mParseEnv->ligo_lw.table.elt[irow].data.real_8;
	break;
    default:
        r = def;
    }
    return r;
}

//======================================  Get long value
long
xsil::MetaIO::getInt(const char* cname, long def) const {
    if (!is_open()) return def;
    int irow = MetaioFindColumn(const_cast<MetaioParseEnvironment*>(mParseEnv), 
				cname);
    if (irow < 0 || !mParseEnv->ligo_lw.table.elt[irow].valid) return def;

    long r;
    switch (mParseEnv->ligo_lw.table.col[irow].data_type) {
    case METAIO_TYPE_INT_4S:
        r = mParseEnv->ligo_lw.table.elt[irow].data.int_4s;
	break;
    case METAIO_TYPE_INT_8S:
        r = mParseEnv->ligo_lw.table.elt[irow].data.int_8s;
	break;
    case METAIO_TYPE_INT_2S:
        r = mParseEnv->ligo_lw.table.elt[irow].data.int_2s;
	break;
    default:
        r = def;
    }
    return r;
}

//======================================  Get string value
std::string
xsil::MetaIO::getString(const char* cname, const char* def) const {
    if (!is_open()) return def;
    int irow = MetaioFindColumn(const_cast<MetaioParseEnvironment*>(mParseEnv), 
				cname);
    if (irow < 0 || !mParseEnv->ligo_lw.table.elt[irow].valid) return def;
    std::string r(def);
    std::ostringstream ss;
    switch (mParseEnv->ligo_lw.table.col[irow].data_type) {
    case METAIO_TYPE_INT_4S:
        ss << mParseEnv->ligo_lw.table.elt[irow].data.int_4s;
	r = ss.str();
	break;
    case METAIO_TYPE_INT_8S:
        ss << mParseEnv->ligo_lw.table.elt[irow].data.int_8s;
	r = ss.str();
	break;
    case METAIO_TYPE_INT_2S:
        ss << mParseEnv->ligo_lw.table.elt[irow].data.int_2s;
	r = ss.str();
	break;
    case METAIO_TYPE_REAL_4:
        ss << mParseEnv->ligo_lw.table.elt[irow].data.real_4;
	r = ss.str();
	break;
    case METAIO_TYPE_REAL_8:
        ss << mParseEnv->ligo_lw.table.elt[irow].data.real_8;
	r = ss.str();
	break;
    case METAIO_TYPE_ILWD_CHAR:
      {
#if METAIO_VSN < 800
	const char* p = (const char*)mParseEnv->ligo_lw.table.elt[irow].data.ilwd_char.data;
	int N = mParseEnv->ligo_lw.table.elt[irow].data.ilwd_char.len;
#else
	const char* p = (const char*)mParseEnv->ligo_lw.table.elt[irow].data.lstring.data;
	int N = mParseEnv->ligo_lw.table.elt[irow].data.lstring.len;
#endif
	r = string(p, N);
	break;
      }
    case METAIO_TYPE_ILWD_CHAR_U:
      {
#if METAIO_VSN < 800
	const char* p = (const char*)mParseEnv->ligo_lw.table.elt[irow].data.ilwd_char_u.data;
	int N = mParseEnv->ligo_lw.table.elt[irow].data.ilwd_char_u.len;
#else
	const char* p = (const char*)mParseEnv->ligo_lw.table.elt[irow].data.blob.data;
	int N = mParseEnv->ligo_lw.table.elt[irow].data.blob.len;
#endif
	r = string(p,N);
	break;
      }
    case METAIO_TYPE_LSTRING:
        r = std::string(mParseEnv->ligo_lw.table.elt[irow].data.lstring.data,
			mParseEnv->ligo_lw.table.elt[irow].data.lstring.len);
	break;
    default:
        break;
    }
    int l = r.size();
    if (l > 0 && r[l-1] == '\n') r.erase(l-1,1);
    return r;
}

//======================================  Get a pointer to the table name
const char*
xsil::MetaIO::getTableName(void) const {
    if (!is_open()) return 0;
    return mParseEnv->ligo_lw.table.name;
}

//======================================  Read the next row from the file.
int 
xsil::MetaIO::getRow(void) {
    if (!is_open()) {
        //cerr << "MetaIO::getRow: Environment is closed!" << endl;
	return -1;
    }
    return MetaioGetRow(mParseEnv);
}

//======================================  Get a pointer to the table name
int
xsil::MetaIO::selectTable(const char* table) {
    if (!is_open()) return -1;
    return MetaioOpenTableOnly(mParseEnv, table);
}
