/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: xsilHistogram     					*/
/*                                                         		*/
/* Module Description: Classes for LIGO-LW input/output to DMT objects.	*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0   4/1/02   D. Sigg       Initial revision                        */
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: none 						*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 2001.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#ifndef _GDS_XSILHISTOGRAM_HH
#define _GDS_XSILHISTOGRAM_HH

#include <string>
#include <iosfwd>
#include "xml/Xsil.hh"
#include "Histogram1.hh"
#include "Histogram2.hh"

namespace xml {

/** Converts a DMT Histogram1 or Histogram2 object to an 
    output stream in LIGO_LW format as described in LIGO-T000067.
   
    Example:
    \verbatim
    ofstream out ("myfile.xml");
    Histogram1 h1 (...);
    Histogram2 h2 (...);
    // Fill histogram...
    out << xsilHeader() << endl;
    out << xsilHistogram (&h1) << endl;
    out << xsilHistogram (&h2) << endl;
    out << xsilTrailer() << endl;
    \endverbatim
 
    @memo Writes HistogramX in LIGO_LW format as described in LIGO-TOOOO67
    @author Written September 2001 by Daniel Sigg
    @version 0.1
    @ingroup IO_handlers
  *************************************************************************/
   class xsilHistogram {
   protected:
      /// Indent level
      int            fLevel;
      /// Reference to Hostogram1 object
      const Histogram1* fH1;
      /// Reference to Hostogram2 object
      const Histogram2* fH2;
   public:
      /// Constructor
      xsilHistogram (int level = 1) 
      : fLevel(level), fH1 (0), fH2 (0) {
      }
      /// Constructor for 1D histogram
      xsilHistogram (const Histogram1* h1, int level = 1) 
      : fLevel(level), fH1 (h1), fH2 (0) {              
      }
      /// Constructor for 1D histogram
      xsilHistogram (const Histogram1& h1, int level = 1) 
      : fLevel(level), fH1 (&h1), fH2 (0) {              
      }
      /// Constructor for 2D histogram
      xsilHistogram (const Histogram2* h2, int level = 1) 
      : fLevel(level), fH1 (0), fH2 (h2) {              
      }
      /// Constructor for 2D histogram
      xsilHistogram (const Histogram2& h2, int level = 1) 
      : fLevel(level), fH1 (0), fH2 (&h2) {              
      }
      /// Write the histogram data to output stream
      std::ostream& write (std::ostream &os) const;
   };


/** Write histogram to an output stream.
    @param os output stream
    @param h xsilHistogram object
    @memo XML output of histograms
    @return output stream
    @ingroup xml_handlers
  *************************************************************************/
   inline std::ostream& operator<< (std::ostream &os, 
                     const xsilHistogram &h) {
      return h.write(os); }

/** @name XML handlers for Histogram data
    @memo XML handlers for parsing the LIGO_LW format of data into
    DMT objects following the xml representation described in
    LIGO-T000067.

    Example for reading a few histograms:
    \begin{verbatim}
    vector<Histogram1> hlist1;
    vector<Histogram2> hlist2;
    xsilHandlerQueryHistogram histQ (hlist1, hlist2);
    xsilParser parser;
    parser.AddHandler (histQ);
    parser.Parse (filename);
    // Look at histograms in hlist...
    \end{verbatim}

    @author Written September 2001 by Daniel Sigg
    @version 0.1
    @ingroup xml_handlers
  *************************************************************************/
//@{

/** Handler for Histogram data types
    @memo Histogram datatype handler
    @author Written September 2001 by Daniel Sigg
    @version 0.1
    @ingroup xml_handlers
  *************************************************************************/
   class xsilHandlerHistogram : public xsilHandler {
   protected:
      /// Pointer to DMT Histogram1 vector class
      std::vector<Histogram1>*	  fDat1;
      /// Pointer to DMT Histogram2 vector class
      std::vector<Histogram2>*	  fDat2;
   
      /// Name of data object
      std::string                 fTitle;
      /// X-axis/Y-axis/Z-Axis and bin-count label
      std::string		  fLabel[4];
      /// Data subtype
      int                         fSubtype;
      /// Time (sec)
      unsigned long               fSec;
      /// Time (nsec)
      unsigned long               fNsec;
      /// Number of x/y/z bins
      int			  fNBin[3];
      /// Number of data points
      int			  fNData;
      /// Sum of the weights
      double			  fSumWeight;
      /// Sum of the weight squares
      double			  fSumWeightSqr;
      /// Sum of the x*weights/y*weights/z*weights
      double			  fSumWeightDim[3];
      /// Sum of the x*weights/y*weights/z*weights squares
      double			  fSumWeightDimSqr[3];
      /// Sum of the x*y*weights
      double			  fSumWeightXY;
      /// low edge (x, y and z)
      double			  fLowEdge[3];
      /// spacing (x, y and z)
      double			  fSpacing[3];
      /// Bins (x, y and z)
      double*			  fBins[3];
      /// Length of bin arrays
      int			  fBinsLen[3];
      /// Errors
      double*			  fErrors;
      /// Length of error array
      int			  fErrorsDim[3];
      /// Data pointer
      double*			  fData;
      /// Length of data array
      int			  fDataDim[3];
   
   public:
      /// Constructor
      explicit xsilHandlerHistogram (std::vector<Histogram1>* objs1, 
                        std::vector<Histogram2>* objs2,
                        const attrlist *attr=0, bool ignore=false);
      /// Destructor
      virtual ~xsilHandlerHistogram();
      /// bool parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const bool& p, int N = 1) {
         return false; }
      /// byte parameter callback (must return true if handled)
      virtual bool HandleParameter(const std::string& name, 
                        const attrlist& attr,
                        const char& p, int N = 1) {
         return false; }
      /// short parameter callback (must return true if handled)
      virtual bool HandleParameter(const std::string& name, 
                        const attrlist& attr,
                        const short& p, int N = 1) {
         return false; }
      /// int parameter callback (must return true if handled)
      virtual bool HandleParameter(const std::string& name, 
                        const attrlist& attr,
                        const int& p, int N=1);
   #ifndef __CINT__
      /// long parameter callback (must return true if handled)
      virtual bool HandleParameter(const std::string& name, 
                        const attrlist& attr,
                        const long long& p, int N=1) {
         return false; }
   #endif //__CINT__
      /// float parameter callback (must return true if handled)
      virtual bool HandleParameter(const std::string& name, 
                        const attrlist& attr, 
                        const float& p, int N=1) {
         return false; }
      /// double parameter callback (must return true if handled)
      virtual bool HandleParameter(const std::string& name, 
                        const attrlist& attr, 
                        const double& p, int N=1);
      /// complex float parameter callback (must return true if handled)
      virtual bool HandleParameter(const std::string& name, 
                        const attrlist& attr,
                        const std::complex<float>& p, int N = 1) {
         return false; }
      /// complex double parameter callback (must return true if handled)
      virtual bool HandleParameter(const std::string& name, 
                        const attrlist& attr,
                        const std::complex<double>& p, int N = 1) {
         return false; }
      /// string parameter callback (must return true if handled)
      virtual bool HandleParameter(const std::string& name, 
                        const attrlist& attr,
                        const std::string& p);
      /// time callback (must return true if handled)
      virtual bool HandleTime (const std::string& name, 
                        const attrlist& attr,
                        unsigned long sec, unsigned long nsec);
      /// data callback (must return true if data is adopted)
      virtual bool HandleData (const std::string& name, float* x, 
                        int dim1, int dim2=0, int dim3=0, int dim4=0) {
         return false; }
      /// data callback (must return true if data is adopted)
      virtual bool HandleData (const std::string& name, 
                        std::complex<float>* x,
                        int dim1, int dim2=0, int dim3=0, int dim4=0) {
         return false; }
      /// data callback (must return true if data is adopted)
      virtual bool HandleData (const std::string& name, double* x, 
                        int dim1, int dim2=0, int dim3=0, int dim4=0);
      /// data callback (must return true if data is adopted)
      virtual bool HandleData (const std::string& name, 
                        std::complex<double>* x,
                        int dim1, int dim2=0, int dim3=0, int dim4=0) {
         return false; }
   };

/** Xsil Histogram handler query class.
    The query will return a handler if the data object type is Histogram.
    @memo Xsil Histogram handler query
    @author Written September 2001 by Daniel Sigg
    @version 0.1
    @ingroup xml_handlers
  *************************************************************************/
   class xsilHandlerQueryHistogram : public xsilHandlerQuery {
   protected:
      /// Pointer to DMT Object vector class
      std::vector<Histogram1>*	  fDat1;
      /// Pointer to DMT Object vector class
      std::vector<Histogram2>*	  fDat2;
   
   public:
      /// Constructor for 1D histograms
      explicit xsilHandlerQueryHistogram (std::vector<Histogram1>& h1)
      : fDat1 (&h1), fDat2 (0) {
      }
      /// Constructor for 2D histograms
      explicit xsilHandlerQueryHistogram (std::vector<Histogram2>& h2)
      : fDat1 (0), fDat2 (&h2) {
      }
      /// Constructor for 1D & 2D histograms
      explicit xsilHandlerQueryHistogram (std::vector<Histogram1>& h1,
                        std::vector<Histogram2>& h2)
      : fDat1 (&h1), fDat2 (&h2) {
      }
      /// returns a handler for the specified object (or 0 if not)
      virtual xsilHandler* GetHandler (const attrlist& attr);
   };
//@}

} // namespace xml

#endif // _GDS_XSILHISTOGRAM_HH
