/* -*- mode: c++; c-basic-offset: 4; -*- */
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: webserver						*/
/*                                                         		*/
/* Module Description: a basic web server				*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 0.1	 4Jan02   D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: webserver.html					*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#ifndef _LIGO_WEBSERVER_H
#define _LIGO_WEBSERVER_H


#include <semaphore.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <vector>
#include <map>
#include <string>
#include <iosfwd>
#include "gmutex.hh"
#include "web/webcache.hh"
#include <cstring>

//  The hostname printing caused the web server to fail when a dns was.
//  for this reason, I have slightly reformatted the transaction log
//  printout and mad the host-name translation optional. To turn on the
//  hostname printing you must uncomment the following line:
//
//  #define WEBSRV_HOSTNAME 1
//
//--> JGZ  31.X.2006

namespace web {

   const int kDefaultPort = 80;

/** @page Basic web server
    This header implements a basic web server which supports
    HTTP 1.0.

    A simple web server can be written by deriving the server class
    from basic_server and override the RequestGet method.
   
    @memo Basic web server
    @author Written January 2002 by Daniel Sigg
    @version 1.0
    @ingroup  IO_web
 ************************************************************************/

/** Server request class
   
    @memo Server request
    @author Written January 2002 by Daniel Sigg
    @version 1.0
    @ingroup IO_web
 ************************************************************************/
   class request_t {
   public:
      /// Request type
      enum req_type {
      /// Invalid
      rInvalid, 
      /// GET
      rGet, 
      /// POST
      rPost, 
      /// HEAD
      rHead,
      /// OPTIONS
      rOptions,
      /// PUT
      rPut,
      /// DELETE
      rDelete,
      /// TRACE
      rTrace,
      /// CONNECT
      rConnect
      };
      /// Header sort orde
      class header_sort {
      public:
         /// ignore case
         bool operator () (const std::string& s1, const std::string& s2) const;
      };
      /// header type (name/token)
      typedef std::map <std::string, std::string, header_sort> header_type;
      /** Browser information. Go to 
          http://www.mozilla.org/docs/web-developer/sniffer/browser_type.html
          for the caveats.
        */
      struct browser_t {
	 /// Constructor
         browser_t (const char* p = 0);
         /// Major version
         int	fMajor;
         /// Minor version
         float	fMinor;
         /// Is Navigator?
         bool	fIsNav;
         /// Is Navigator version 2?
         bool	fIsNav2;
         /// Is Navigator version 3?
         bool	fIsNav3;
         /// Is Navigator verison 4?
         bool	fIsNav4;
         /// Is Navigator verison 4 and up?
         bool	fIsNav4up;
         /// Is Navigator only?
         bool	fIsNavOnly;
         /// Is Navigator version 6?
         bool	fIsNav6;
         /// Is Navigator verison 6 or up?
         bool	fIsNav6up;
         /// Is Gecko?
         bool	fIsGecko;
         /// Is Internet Explorer?
         bool	fIsIe;
         /// Is Internet Explorer version 3?
         bool	fIsIe3;
         /// Is Internet Explorer version 4?
         bool	fIsIe4;
         /// Is Internet Explorer version 4 and up?
         bool	fIsIe4up;
         /// Is Internet Explorer verison 5?
         bool	fIsIe5;
         /// Is Internet Explorer version 5.5?
         bool	fIsIe5_5;
         /// Is Internet Explorer verison 5 and up?
         bool	fIsIe5up;
         /// Is Internet Explorer version 5.5 and up?
         bool	fIsIe5_5up;
         /// Is Internet Explorer verison 6?
         bool	fIsIe6;
         /// Is Internet Explorer version 6 and up?
         bool	fIsIe6up;
         /// Is AOL?
         bool	fIsAol;
         /// Is AOL version 3?
         bool	fIsAol3;
         /// Is AOL version 4?
         bool	fIsAol4;
         /// Is AOL version 5?
         bool	fIsAol5;
         /// Is AOL version 6?
         bool	fIsAol6;
         /// Is Opera?
         bool	fIsOpera;
         /// Is Opera version 2?
         bool	fIsOpera2;
         /// Is Opera version 3?
         bool	fIsOpera3;
         /// Is Opera version 4?
         bool	fIsOpera4;
         /// Is Opera version 5?
         bool	fIsOpera5;
         /// Is Opera version 5 and up?
         bool	fIsOpera5up;
         /// Is Web TV?
         bool	fIsWebtv; 
         /// Is TV navigator
         bool	fIsTVNavigator; 
         /// Is AOL TV
         bool	fIsAOLTV;
         /// Is hot java?
         bool	fIsHotjava;
         /// Is hot java version 3?
         bool	fIsHotjava3;
         /// Is hot java verison 3 and up?
         bool	fIsHotjava3up;
         /// Java Script version
         float	fJsVersion;
         /// Is Windows?
         bool	fIsWin;
         /// Is Windows 95?
         bool	fIsWin95;
         /// Is Windows 16bit?
         bool	fIsWin16;  
         /// Is Windows 3.1?
         bool	fIsWin31;
         /// Is Windows Me?
         bool	fIsWinme;
         /// Is Windows 2000?
         bool	fIsWin2k;
         /// Is Windows 98?
         bool	fIsWin98;
         /// Is Windows NT?
         bool	fIsWinnt;
         /// Is Windows 32bit?
         bool	fIsWin32;
         /// Is OS/2?
         bool	fIsOs2;
         /// Is Mac?
         bool	fIsMac;
         /// Is Mac 68k?
         bool	fIsMac68k;
         /// Is Mac PPC?
         bool	fIsMacppc;
         /// Is Sun Os?
         bool	fIsSun;
         /// Is Sun Os 4?
         bool	fIsSun4;
         /// Is Sun Os 5?
         bool	fIsSun5;
         /// Is Sun 86?
         bool	fIsSuni86;
         /// Is Irix?
         bool	fIsIrix;
         /// Is Irix 5?
         bool	fIsIrix5;
         /// Is Irix 6?
         bool	fIsIrix6;
         /// Is HP UX?
         bool	fIsHpux;
         /// Is HP UX 9?
         bool	fIsHpux9;
         /// Is HP UX 10?
         bool	fIsHpux10;
         /// Is AIX?
         bool	fIsAix;
         /// Is AIX 1?
         bool	fIsAix1;    
         /// Is AIX 2?
         bool	fIsAix2;    
         /// Is AIX 3?
         bool	fIsAix3;    
         /// Is AIX 4?
         bool	fIsAix4;    
         /// Is Linux?
         bool	fIsLinux;
         /// Is SCO?
         bool	fIsSco;
         /// Is UnixWare?
         bool	fIsUnixware; 
         /// Is Mpras?
         bool	fIsMpras; 
         /// Is Reliant?
         bool	fIsReliant;
         /// Is DEC?
         bool	fIsDec;
         /// Is Sinix?
         bool	fIsSinix;
         /// Is Free BSD?
         bool	fIsFreebsd;
         /// Is BSD?
         bool	fIsBsd;
         /// Is Unix?
         bool	fIsUnix;
         /// Is VMS?
         bool	fIsVms;
      };
   
      /// Constructor
      request_t() 
      : fHttpVersion (10), fType (rInvalid), fKeepAlive (false), 
      fData (0), fLen (0) {
      }
       /// Constructor
      request_t (int fd)     
      : fHttpVersion (10), fType (rInvalid), fKeepAlive (false), 
      fData (0), fLen (0) {
         read (fd); }
      /// Destructor
      ~request_t() {
         if (fData) delete[] fData; }
      /// Read a request from fd
      bool read (int fd);
      /// Set type
      void SetType (req_type p) {
         fType = p; }
      /// Get type
      req_type GetType() const {
         return fType; }
      /// Set URL
      void SetUrl (const char* p) {
         fUrl = p; }
      /// Get URL
      const char* GetUrl() const {
         return fUrl.c_str(); }
      /// Get demangled URL
      std::string GetDemangledUrl() const;
      /// Get header
      header_type& Header() {
         return fHeader; }
      /// Get header
      const header_type& Header() const {
         return fHeader; }
      ///  Test the KeepAlive flag
      bool KeepAlive() const {
         return fKeepAlive; }
   
      /// Get data
      const char* GetData() const {
         return fData; }
      /// Get data
      char* GetData() {
         return fData; }
      /// Set data
      void SetData (const char* p, int len) {
         Reserve (len); 
         if (len) memcpy (fData, p, len); };
      /// Reserve data array
      void Reserve (int len) {
         if (fData) delete [] fData; 
         fData = (len > 0) ? new char [len] : 0; 
         fLen = (len > 0) ? len : 0; }
      /// Get length
      int GetLength() const {
         return fLen; }
      /// Get HTTP version: 10 (1.0) or 11 (1.1)
      int GetHttpVersion() const {
         return fHttpVersion; }
      /// Get browser type
      const browser_t& GetBrowser() const {
         return fBrowser; }
   private:
      int		fHttpVersion;
      browser_t		fBrowser;
      req_type		fType;
      std::string 	fUrl;
      header_type	fHeader;
      bool		fKeepAlive;
      char*		fData;
      int		fLen;
   
      request_t (const request_t&);
      request_t& operator= (const request_t&);
   };

/** Server response class
    @memo Server response
    @author Written January 2002 by Daniel Sigg
    @version 1.0
    @ingroup IO_web
 ************************************************************************/
   class response_t {
   public:
      /// Header entry (name/token)
      typedef std::pair <std::string, std::string> header_entry;
      /// header type (name/token list)
      typedef std::vector <header_entry> header_type;
   
      /// Constructor
      response_t() : fData (0), fLen (0), fCanCompress (false) {
      }
      /// Destructor
      ~response_t() {
         if (fData) delete[] fData; }
   
      /// Write a response to fd
      bool write (int fd);
      /// Set status line
      void SetStatusLine (const char* p) {
         fStatusLine = p; }
      /// Get status line
      const char* GetStatusLine() const {
         return fStatusLine.c_str(); }
      /// Add header entry
      void AddHeader (const std::string& name, const std::string& token);
      /// Remove header entry
      void RemoveHeader (const std::string& name);
      /// Get Header
      header_type& GetHeader() {
         return fHeader; }
      /// Get Header
      const header_type& GetHeader() const {
         return fHeader; }
   
      /// Get data
      const char* GetData() const {
         return fData; }
      /// Get data
      char* GetData() {
         return fData; }
      /// Set data
      void SetData (const char* p, int len) {
         Reserve (len); 
         if (len) memcpy (fData, p, len); };
     /// Set Data
      void SetData (const std::string& s) {
         SetData(s.c_str(), s.size());}
      /// Reserve data array
      void Reserve (int len) {
         if (fData) delete [] fData; 
         fData = (len > 0) ? new char [len] : 0; 
         fLen = (len > 0) ? len : 0; }
      /// Get length
      int GetLength() const {
         return fLen; }
      /// Try to compress response data
      void SetCanCompress (bool set = true) {
         fCanCompress = set; }
      /// Get try compress value
      bool GetCanCompress() const {
         return fCanCompress; }
      /// Try to compress response data
      bool TryCompress();
   private:
      std::string	fStatusLine;
      header_type 	fHeader;
      char*		fData;
      int		fLen;
      bool		fCanCompress;
   
      response_t (const response_t&);
      response_t& operator= (const response_t&);
   };

   inline void 
   response_t::AddHeader(const std::string& name, const std::string& token) {
     header_entry ent(name, token);
     fHeader.push_back(ent); 
   }


/** This class implements a simple stand-alone web server. It will
    listen at the specified port for requests and spawn a thread
    to process the request. By default it will install a ^C and SIGTERM 
    handler that closes the socket and exits. However, this works only, 
    if only one instance of basic_webserver is created. If the specified
    number of maximum retries is larger than zero, the server tries to
    continue running even after a SIGSEGV, SIGBUS, SIGILL or SIGFPE
    trap. This may lead to unstable operation and inconsistent variable
    states.
   
    @memo Basic web server
    @author Written January 2002 by Daniel Sigg
    @version 1.0
    @ingroup IO_web
 ************************************************************************/
   class basic_server {
   public:
      /** Constructs a web server.
          The following command line arguments are supported:
          \begin{verbatim}
          -p <port>    Port number for listening socket
          \end{verbatim}
          
          @memo Constructor
          @param argn Number of cmd line arguments
          @param argv list of cmd line arguments
          @param max Maximum number of allowed threads
     @param ctrlC If true installs a ^C and SIGTERM handler
     @param maxretry Maximum number of recover attempts
       ******************************************************************/
      basic_server (int argn, char** argv, int max = 20, 
                   bool ctrlC = true, int maxretry = 0);
      /** Destructs the web server. Terminates all threads
          @memo Destructor
       ******************************************************************/
      virtual ~basic_server();
      /** Starts listening at the port. This method will not return
          if successful.
          @memo Listen
       ******************************************************************/
      virtual void Listen();
      /** Parse a server request.
          @memo Parse request
          @param csock client socket
          @param caddr client address
          @param clen address length
          @return True if request could be processed
       ******************************************************************/
      virtual bool Parse (int csock, struct sockaddr* caddr, int clen);
   
      /** Get the port number.
          @memo Get port number
          @return port
       ******************************************************************/
      int getPort() const {
         return fPort; }
   
      /** Set the output log.
          @memo Set log
          @param os Output stream
       ******************************************************************/
      void SetLog (std::ostream* os) {
         fLog = os; }
   
      /** Handles ^C and SIGTERM.
          By default closes the socket and exits
          @memo Handles ^C and SIGTERM
          @param sig Signal
       ******************************************************************/
      virtual void Interrupt (int sig);
   
   protected:
      /// Server mutex
      mutable thread::recursivemutex	fMux;
      /// Web page cache (by defeault caching is disabled)
      webcache				fCache;
   
      /** Spawn a new server thread. This function is called by Listen
          whenever a new request is received.
          @memo Spawn
          @param csock client socket
          @param caddr client address
          @param clen address length
       ******************************************************************/
      virtual void Spawn (int csock, struct sockaddr* caddr, int clen);
      /** Process a request.
          Can be overriden by descendents.
          @memo Process request
          @param request requested page
          @param response Web page (return)
          @return True if request could be processed
       ******************************************************************/
      virtual bool Request (const request_t& request,
                        response_t& response);							
      /** Process a CONNECT request.
          Can be overriden by descendents. Default handler will return
          OK.
          @memo Process CONNECT request
          @param request requested page
          @param response Web page (return)
          @return True if request could be processed
       ******************************************************************/
      virtual bool RequestConnect (const request_t& request,
                        response_t& response);
      /** Process a GET request.
          Can be overriden by descendents. Default handler will return
          NOT FOUND.
          @memo Process GET request
          @param request requested page
          @param response Web page (return)
          @return True if request could be processed
       ******************************************************************/
      virtual bool RequestGet (const request_t& request,
                        response_t& response);
      /** Process a HEAD request.
          Can be overriden by descendents. Default handler will call
          RequestGet and truncate the data field.
          @memo Process HEAD request
          @param request requested page
          @param response Web page (return)
          @return True if request could be processed
       ******************************************************************/
      virtual bool RequestHead (const request_t& request,
                        response_t& response);
      /** Process a POST request.
          Can be overriden by descendents. Default handler will return
          NOT IMPLEMENTED.
          @memo Process POST request
          @param request requested page
          @param response Web page (return)
          @return True if request could be processed
       ******************************************************************/
      virtual bool RequestPost (const request_t& request,
                        response_t& response);
      /** Process a PUT request.
          Can be overriden by descendents. Default handler will return
          NOT IMPLEMENTED.
          @memo Process PUT request
          @param request requested page
          @param response Web page (return)
          @return True if request could be processed
       ******************************************************************/
      virtual bool RequestPut (const request_t& request,
                        response_t& response);
      /** Process a DELETE request.
          Can be overriden by descendents. Default handler will return
          NOT IMPLEMENTED.
          @memo Process DELETE request
          @param request requested page
          @param response Web page (return)
          @return True if request could be processed
       ******************************************************************/
      virtual bool RequestDelete (const request_t& request,
                        response_t& response);
      /** Process a OPTIONS request.
          Can be overriden by descendents. Default handler will return
          NOT IMPLEMENTED.
          @memo Process OPTIONS request
          @param request requested page
          @param response Web page (return)
          @return True if request could be processed
       ******************************************************************/
      virtual bool RequestOptions (const request_t& request,
                        response_t& response);
      /** Process a TRACE request.
          Can be overriden by descendents. Default handler will return
          NOT IMPLEMENTED.
          @memo Process TRACE request
          @param request requested page
          @param response Web page (return)
          @return True if request could be processed
       ******************************************************************/
      virtual bool RequestTrace (const request_t& request,
                        response_t& response);
      /** Check for cached entry.
          @param request requested page
          @param response Web page (return)
          @return True if request was found in cache
       ******************************************************************/
      virtual bool CheckCache (const request_t& request, 
                        response_t& response);
      /** Close server socket.
          @memo Close server socket
       ******************************************************************/
      virtual void Close();
   
      /// Log file
      std::ostream*		fLog;
   
   private:
      typedef std::vector<pthread_t> thread_list;
      /// Thread semaphore
      sem_t			fThreadCount;
      /// Thread count mutex
      mutable thread::mutex	fThreadMux;
      /// List of thread IDs
      thread_list 		fThreads;
      /// Main server socket
      int			fMainSocket;
      /// Port number
      int			fPort;
   };

}

#endif // _LIGO_WEBSERVER_H
