/* -*- mode: c++; c-basic-offset: 3; -*- */
#ifndef _LIGOWEBCLIENT_H
#define _LIGOWEBCLIENT_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: webclient						*/
/*                                                         		*/
/* Module Description: Class to access a web server			*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 0.1	 23Feb05  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: webclient.html					*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#include <string>
#include <map>

namespace web {

/** Class to access a web server.
    @memo web server access class
    @author Written February 2005 by Daniel Sigg
    @version 1.0
    @ingroup IO_web
 ************************************************************************/
   class http_request {
   public:
  
      /** Default constructor.
          @memo Default constructor
       ******************************************************************/
      http_request(void);

      /** Constructor.
          Open connection, send request and wait for answer.
          @memo Open connection, send request and wait for answer
          @param url URL string
          @param addr Web server address
          @param port Web server port number
   	  @return true if successful
       ******************************************************************/
      http_request (const char* url, const char* addr, int port = 80);

      /** Destructor.
          @memo Destructor
       ******************************************************************/
      ~http_request ();

      /** Open connection.
          @memo Open connection
          @param addr Web server address
          @param port Web server port number
   	  @return true if successful
       ******************************************************************/
      bool open (const char* addr, int port = 80);

      /** Clear the input buffer.
          \brief Clear the buffer.
       *****************************************************************/
      void clear(void);

      /** Close connection.
          @memo Close connection
       ******************************************************************/
      void close(void);

      /** Send request and wait for answer.
          @memo Send request and wait for answer
          @param url URL string
   	  @return true if successful
       *****************************************************************/
      bool request (const char* url);

      /** Open connection, send request and wait for answer.
          @memo Open connection, send request and wait for answer
          @param url URL string
          @param addr Web server address
          @param port Web server port number
   	  @return true if successful
       *****************************************************************/
      bool request (const char* url, const char* addr, int port = 80);

      /**  Set the debug printout level
	   \brief Set debug level
	   \param lvl Print level.
       *****************************************************************/
      void set_debug(int lvl);

      /// Get content length
      int size() const {
         return fLen; }

      /// Get content pointer
      char* data() {
         return fData; }
   
      /** Mangle a string to escape control characters.
          @memo Mangle
          @param s input string
   	  @return mangled string
       ******************************************************************/
      static std::string mangle (const std::string& s);

      /** Demangle a string to unescape control characters.
          @memo Demangle
          @param s input string
   	  @return demangled string
       ******************************************************************/
      static std::string demangle (const std::string& s);
     
      /** Write data to the data buffer.
       */
      size_t wbuf(const char* ptr, size_t nBytes);
  
   protected:
      std::string  fServer;   ///<  Server name and port
      char*	   fData;     ///<  Data
      size_t	   fLen;      ///<  Data length
      size_t	   fFill;     ///<  filled data
      void*	   fCurl;     ///<  Curl handle
      int          fDebug;    ///<  Debug print level.

      /// send an http request
      bool sendRequest (const char* url);
      /// receive the header of the response
      bool recvHeader();
      /// receive the content
      bool recvContent();
      /// uncompress the content
      bool uncompress();
  
   private:
      http_request (const http_request&);
      http_request& operator= (const http_request&);
   };


}

#endif
