#ifndef _LIGO_SOCKUTIL_H
#define _LIGO_SOCKUTIL_H

#ifdef __cplusplus
extern "C" {
#endif


/* Header File List: */
#include <inttypes.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/time.h>

/** \file 
    This API provides utility functions to deal with sockets.
    \brief Socket Utilities
    \author Written August 1999 by Daniel Sigg
    \version $Id: sockutil.h 7304 2015-03-07 09:03:55Z john.zweizig@LIGO.ORG $
************************************************************************/

/*@{*/

/** Wait times are specified in seconds with negative numbers indicating
    an infinite time.
    \brief Wait time data type.
************************************************************************/
  typedef double wait_time;

/** This function uses DNS if necessary. If the
    hostname is NULL, the (first) address of the local machine is 
    returned.

    \brief Look up a hostname.
    \param hostname name of the host (NULL for self)
    \param addr IP address (return)
    \return 0 if successful, <0 otherwise
    \author DS, July 98
************************************************************************/
   int nslookup (const char* hostname, struct in_addr* addr);

/**  This function uses DNS if 
    necessary. If the address is NULL, the name of the local 
    machine is returned. Hostname should contain at least 256 chars.

    \brief Look up an address (inverse lookup).
    \param addr IP address
    \param hostname name of the host (return)
    \return 0 if successful, <0 otherwise
    \author DS, July 98
************************************************************************/
   int nsilookup (const struct in_addr* addr, char* hostname);

/**  This function is similar to connect, but implements an additional 
     user specified timeout. If the connextion was not made, the function 
     returns -1 and the reson for failure is indicated by \e errno.

     \brief Connects a socket with a timeout
     \param sock Socket
     \param name Address to which the connectiom will be made.
     \param size Size of socket address
     \param timeout Connection timeout
     \return 0 if successful, <0 otherwise
     \author DS (July 98), JGZ (Mar 2015)
************************************************************************/
   int connectWithTimeout (int sock, struct sockaddr* name, int size, 
			   wait_time timeout);

/** ping() attempts to establish a connection to the echo port of 
    the specified host machine. If a connection is established or refused, 
    ping returns with true. If the connection attempt aborts with a timeout, 
    the function returns false. A non positive timeout value is interpreted 
    as the default (10 sec).

    \brief Ping the given host.
    \param hostname name of the machine to ping
    \param timeout maximum time to wait for response (sec)
    \return true if alive, false otherwise
    \author DS, July 98
************************************************************************/
   int ping (const char* hostname, wait_time timeout);

/** Set the system receive buffer length.
    \brief Set the receive buffer length
    \param sock Socket fd.
    \param len  Length to set.
    \return 0 if successful, -1 if an error occurred.
************************************************************************/
  int setRcvBufLen(int sock, int len);

/** Set the system send buffer length.
    \brief Set the send buffer length
    \param sock Socket fd.
    \param len  Number of bytes to allocate for write buffer.
    \return 0 if successful, -1 if an error occurred.
************************************************************************/
  int setSndBufLen(int sock, int len);

/** Wait condition enumeration.
************************************************************************/
enum wait_mode {
  wm_read,   /**< read ready flag */
  wm_write,  /**< write ready flag */
  wm_ctrl    /**< control ready flag */
};

/** Wait for a condition on the specified socket. The conditions are 
    enumerated by the wm_mode flag.
    \brief Wait for a socket
    \param sock Socket fd.
    \param maxtime Maximum time to wait in seconds. Negative = infinite.
    \param mode conditions to wait for.
    \return 1 if condition was satisfied, 0 if condition was not satisfied 
              in the specified time, -1 if an error occurred.
************************************************************************/
  int socketWait(int sock, wait_time maxtime, enum wait_mode mode);

/*@}*/

#ifdef __cplusplus
}
#endif

#endif /*_LIGO_SOCKUTIL_H */
