/* -*- mode: c++; c-basic-offset: 3; -*- */
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: gdssock							*/
/*                                                         		*/
/* Module Description: socket utility functions				*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

/* Header File List: */
#include "sockutil.h"
#include "PConfig.h"
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <arpa/inet.h>
#if defined (P__SOLARIS)
#include <netdir.h>
#endif
#include <fcntl.h>
#include <sys/file.h>
#include <unistd.h>
#if defined (P__WIN32)
#include <pthread.h>
#endif

#if 0 /* Change to 1 to enable verbose debugging */
#define DEBUG_PRT 1 /* Enable verbose debugging */
#endif /* 0 */
#include <stdio.h>	/* perror */

/*  define obsolete fcntl flag. */
#ifndef FNDELAY
#define FNDELAY O_NONBLOCK
#endif

/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Constants: _PING_TIMEOUT	  timeout for ping (sec)		*/
/*            _SHUTDOWN_DELAY	  wait period before shut down		*/
/*            								*/
/*----------------------------------------------------------------------*/
#define _PING_PORT 		7
#define _PING_TIMEOUT		1.0


/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Internal Procedure Name: copy_hent					*/
/*            								*/
/*----------------------------------------------------------------------*/
#if defined (P__WIN32)
   static struct hostent* copy_hent (struct hostent *result, 
                     const struct hostent* ret,
                     char *buffer, int buflen)
   {
      char** p;
      int n;
      /* copy hent */
      memcpy (result, ret, sizeof (struct hostent));
      /* copy alias list array */
      n = 0;
      for (p = ret->h_aliases; *p; ++p) ++n;
      if (buflen < (n + 1) * sizeof (char*)) {
         return NULL;
      }
//      memcpy (buffer, ret->h_aliases, (n + 1) * sizeof (char*));
      result->h_aliases = (char**)buffer;
      buflen -= (n + 1) * sizeof (char*); 
      buffer += (n + 1) * sizeof (char*);
      result->h_aliases[n] = 0;
      /* copy address list array */
      n = 0;
      for (p = ret->h_addr_list; *p; ++p) ++n;
      if (buflen < (n + 1) * sizeof (char*)) {
         return NULL;
      }
//      memcpy (buffer, ret->h_h_addr_list, (n + 1) * sizeof (char*));
      result->h_addr_list = (char**)buffer;
      buflen -= (n + 1) * sizeof (char*); 
      buffer += (n + 1) * sizeof (char*);
      result->h_addr_list[n] = 0;
      /* copy name */
      if (buflen < strlen (ret->h_name) + 1) {
	 return NULL;
      }
      strcpy (buffer, ret->h_name);
      result->h_name = buffer;
      buflen -= strlen (ret->h_name) + 1; 
      buffer += strlen (ret->h_name) + 1;
      /* copy alias list */
      for (n = 0; ret->h_aliases[n]; ++n) {
         if (buflen < strlen (ret->h_aliases[n]) + 1) {
	    return NULL;
         }
	 strcpy (buffer, ret->h_aliases[n]);
	 result->h_aliases[n] = buffer;
	 buflen -= strlen (ret->h_aliases[n]) + 1;
	 buffer += strlen (ret->h_aliases[n]) + 1;
      }
      /* copy address list */
      for (n = 0; ret->h_addr_list[n]; ++n) {
         if (buflen < ret->h_length) {
	    return NULL;
         }
	 memcpy (buffer, ret->h_addr_list[n], ret->h_length);
	 result->h_addr_list[n] = buffer;
	 buflen -= ret->h_length;
	 buffer += ret->h_length;
      }
      return result;
   }
#endif


/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Internal Procedure Name: __destroy_hostent				*/
/*            								*/
/*----------------------------------------------------------------------*/
#if defined (P__LINUX) || defined(P__DARWIN)
static void
__destroy_hostent(struct hostent *h) {
   if (h->h_name != NULL) {
      free(h->h_name);
      h->h_name = NULL;
   }
   if (h->h_addr_list != NULL) {
      char** p;
      for (p=h->h_addr_list; *p != NULL; ++p) {
	 free(*p);
	 *p = NULL;
      }
      free(h->h_addr_list);
   }
}
#endif

/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Internal Procedure Name: __gethostbyname_r				*/
/*            								*/
/*----------------------------------------------------------------------*/
   static struct hostent* __gethostbyname_r (const  char  *name,  
                     struct hostent *result, char *buffer, int buflen, 
		     int *h_errnop)
   {
   #if defined (P__SOLARIS)
      return gethostbyname_r (name, result, buffer, buflen, h_errnop);
   #elif defined (P__LINUX) || defined(P__DARWIN)
      static struct addrinfo ip4hints = {0, AF_INET, 0, 0, 0, NULL, NULL, NULL};
      struct addrinfo* res = 0;
      int rc = getaddrinfo(name, 0, &ip4hints, &res);
      if (rc || !res) return NULL;
#ifdef DEBUG_PRT
      printf("getaddrinfo was successful. flg: %i fam: %i type: %i len: %i\n",
	     res->ai_flags, res->ai_family, res->ai_socktype, res->ai_addrlen);
      printf("address info for %s: ", name);
      int i;
      unsigned char* p = (unsigned char*)res->ai_addr;
      for (i=0; i<res->ai_addrlen; ++i) printf("%i.", (int)p[i]);
      printf("\n");
#endif
      struct sockaddr_in* paddr_in = (struct sockaddr_in*)(res->ai_addr);
      result->h_name      = strdup(name);
      result->h_aliases   = NULL;
      result->h_addrtype  = paddr_in->sin_family;
      result->h_length    = sizeof(paddr_in->sin_addr);
      result->h_addr_list = (char**)malloc(2*sizeof(char*));
      result->h_addr_list[0] = (char*)malloc(result->h_length);
      memcpy(result->h_addr_list[0], &(paddr_in->sin_addr), result->h_length);
      result->h_addr_list[1] = 0;
      freeaddrinfo(res);
      return result;
   #elif defined (P__WIN32)
      struct hostent* ret;
      static pthread_mutex_t GDS__gethostbyname_r_mux = 
         PTHREAD_MUTEX_INITIALIZER;
      pthread_mutex_lock (&GDS__gethostbyname_r_mux);
      ret = gethostbyname (name);
      if (!ret) {
         pthread_mutex_unlock (&GDS__gethostbyname_r_mux);
	 if (h_errnop) *h_errnop = errno;
         return NULL;
      }
      if (!copy_hent (result, ret, buffer, buflen)) {
         pthread_mutex_unlock (&GDS__gethostbyname_r_mux);
	 if (h_errnop) *h_errnop = ERANGE;
         return NULL;
      }
      pthread_mutex_unlock (&GDS__gethostbyname_r_mux);
      if (h_errnop) *h_errnop = 0;
      return result;
   #else
   #error define gethostbyname_r for this platform
   #endif 
   }


/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Internal Procedure Name: __gethostbyaddr_r				*/
/*            								*/
/*----------------------------------------------------------------------*/
   static struct hostent* __gethostbyaddr_r (const char* addr, 
                     int length, int type, struct hostent* result, 
                     char* buffer, int buflen, int* h_errnop)
   
   {
   #if defined (P__SOLARIS)
      return gethostbyaddr_r (addr, length, type, result, buffer, 
                              buflen, h_errnop);
   #elif defined (P__LINUX) || defined(P__DARWIN)
      *h_errnop = getnameinfo((const struct sockaddr*)addr, length, 
			   buffer, buflen, NULL, 0, 0);
      if (*h_errnop) return NULL;

      result->h_name      = strdup(buffer);
      result->h_aliases   = NULL;
      result->h_addrtype  = type;
      result->h_length    = length;
      result->h_addr_list = (char**)malloc(2*sizeof(char*));
      result->h_addr_list[0] = (char*)malloc(length);
      memcpy(result->h_addr_list[0], addr, length);
      result->h_addr_list[1] = 0;
      return result;
   #elif defined (P__WIN32)
      struct hostent* ret;
      static pthread_mutex_t GDS__gethostbyaddr_r_mux = 
         PTHREAD_MUTEX_INITIALIZER;
      pthread_mutex_lock (&GDS__gethostbyaddr_r_mux);
      ret = gethostbyaddr (addr, length, type);
      if (!ret) {
         pthread_mutex_unlock (&GDS__gethostbyaddr_r_mux);
	 if (h_errnop) *h_errnop = errno;
         return NULL;
      }
      if (!copy_hent (result, ret, buffer, buflen)) {
         pthread_mutex_unlock (&GDS__gethostbyaddr_r_mux);
	 if (h_errnop) *h_errnop = ERANGE;
         return NULL;
      }
      pthread_mutex_unlock (&GDS__gethostbyaddr_r_mux);
      if (h_errnop) *h_errnop = 0;
      return result;   
   #else
   #error define gethostbyaddr_r for this platform
   #endif
   }


/*----------------------------------------------------------------------*/
/*                                                         		*/
/* External Procedure Name: nslookup					*/
/*                                                         		*/
/* Procedure Description: looks up a hostname (uses DNS if necessary)	*/
/*                                                         		*/
/* Procedure Arguments: host address, address (return)			*/
/*                                                         		*/
/* Procedure Returns: 0 if successful, <0 if failed			*/
/*                                                         		*/
/*----------------------------------------------------------------------*/
   int nslookup (const char* host, struct in_addr* addr)
   {
      char		hostname[256];
      struct hostent	hostinfo;
      struct hostent* 	phostinfo;
      char		buf[2048];
      int		i;
   
      if (addr == NULL) {
         return -1;
      }
      if (host == NULL) {
         if (gethostname (hostname, sizeof (hostname)) < 0) {
            return -1;
         }
      }
      else {
         strncpy (hostname, host, sizeof (hostname));
      }
   
      phostinfo = __gethostbyname_r (hostname, &hostinfo, 
                                    buf, 2048, &i);
      if (phostinfo == NULL) {
#ifdef DEBUG_PRT
	 printf("Unable to get host address for %s\n", hostname);
#endif
         return -1;
      }
      else {
         memcpy (&addr->s_addr, hostinfo.h_addr_list[0], 
                sizeof (addr->s_addr));
#if defined (P__LINUX)
	 __destroy_hostent(&hostinfo);
#endif
         return 0;
      }
   }


/*----------------------------------------------------------------------*/
/*                                                         		*/
/* External Procedure Name: nsilookup					*/
/*                                                         		*/
/* Procedure Description: looks up an address (uses DNS if necessary)	*/
/*                                                         		*/
/* Procedure Arguments: address, host name (return)			*/
/*                                                         		*/
/* Procedure Returns: 0 if successful, <0 if failed			*/
/*                                                         		*/
/*----------------------------------------------------------------------*/
   int nsilookup (const struct in_addr* addr, char* hostname)
   {
      struct hostent* 	phostinfo;
      struct hostent	hostinfo;
      char		buf[2048];
      int		i;
      uint32_t		a;
   
      if (addr == NULL) {
         if (gethostname (hostname, 256) < 0) {
            return -1;
         }
         else {
            return 0;
         }
      }
   
      a = addr->s_addr;
      phostinfo = __gethostbyaddr_r ((char*)&a, sizeof (a),
                                    AF_INET, &hostinfo, 
                                    buf, 2048, &i);
      if (phostinfo == NULL) {
         return -1;
      }
      else {
         strncpy (hostname, hostinfo.h_name, 256);
#if defined (P__LINUX) || defined(P__DARWIN)
	 __destroy_hostent(&hostinfo);
#endif
         return 0;
      }
   }


/*----------------------------------------------------------------------*/
/*                                                         		*/
/* External Procedure Name: connectWithTimeout				*/
/*                                                         		*/
/* Procedure Description: same as connect, but with timeout		*/
/*                                                         		*/
/* Procedure Arguments: socket, address to connect, address size,	*/
/*                      timeout				      		*/
/*                                                         		*/
/* Procedure Returns: 0 if successful, <0 if failed			*/
/*                                                         		*/
/*----------------------------------------------------------------------*/
   int connectWithTimeout (int sock, struct sockaddr* name, 
			   int size, wait_time timeout)
   {
      int fileflags;
      int nset, sysrc;
      int sverrno;
  
#ifdef DEBUG_PRT
      printf("Connect socket %i to " , sock);
      unsigned char* p = (unsigned char*)name;
      for (nset=0; nset<16; nset++) printf("%i.", (int)p[nset]);
      printf("\n");
#endif

      /* set socket to non blocking */
      fileflags = fcntl (sock, F_GETFL, 0);
      if (fileflags == -1) {
         return -1;
      }
      if (fcntl (sock, F_SETFL, fileflags | O_NONBLOCK) == -1) {
         return -1;
      }

      /*  try to connect */
      sysrc   = connect (sock, name, size);
      sverrno = errno;
      fcntl (sock, F_SETFL, fileflags);
      errno = sverrno;

      /*  test if connection in progress */
      if (sysrc == 0) return 0;
      if (sverrno != EINPROGRESS) return -1;

#ifdef DEBUG_PRT
      printf("Connect to socket %i in progress... Wait %f seconds\n",
	     sock, timeout);
#endif

      /*  wait for connect to finish */
      errno = 0;
      nset = socketWait(sock, timeout, wm_write);
      sverrno = errno;

#ifdef DEBUG_PRT
      printf("Socket %i wait complete, nset = %i\n",
	     sock, nset);
      perror("return from select");
#endif

      /*  error during select */
      if (nset < 0) {
	 errno = sverrno;
         return -1;
      }

      /*  timed out */
      else if (nset == 0) {
         errno = ETIMEDOUT;
         return -1;
      }

      /*  connection attempted, now test if socket is open */
      else {
	 socklen_t bufsz;

	 /* Connection test has been replaced with the following 
	    getsockopt(SO_ERROR) code. As far as I can tell, this code
	    works on all available operating systems (Linux, OSX, Windows)
	    (JGZ: 2015.03.05) 
	 */
	 bufsz = sizeof(sverrno);
	 sysrc = getsockopt(sock, SOL_SOCKET, SO_ERROR, 
			    &sverrno, &bufsz);
	 if (sysrc < 0) {
	    sverrno = errno;
	    perror("error in getsockopt(SO_ERROR)");
	 }
	 errno = sverrno;
	 if (errno != 0) return -1;


#if 0
	 /* Original getpeername- based code */
         struct sockaddr name;
         bufsz = sizeof (name);
         /*  if we can get a peer the connect was successful */
         if (getpeername (sock, &name, &bufsz) < 0) {
            /* errno = ENOENT; */
            return -1;
         }
#endif
      }
      return 0;
   }


/*----------------------------------------------------------------------*/
/*                                                         		*/
/* External Procedure Name: ping					*/
/*                                                         		*/
/* Procedure Description: pings a server				*/
/*                                                         		*/
/* Procedure Arguments: host address, timeout				*/
/*                                                         		*/
/* Procedure Returns: true if successful				*/
/*                                                         		*/
/*----------------------------------------------------------------------*/
   int ping (const char* hostname, wait_time timeout)
   {
      int		sock;		/* socket handle */
      struct sockaddr_in name;		/* internet address */
      int 		nset;		/* # of selected descr. */
      int 		flags;		/* socket flags */
   
      if (timeout <= 0) {
         timeout = 10.;
      }
   
      /* create the socket */
      sock = socket (PF_INET, SOCK_STREAM, 0);
      if (sock == -1) {
         return 0;
      }
   
      /* bind socket */
      name.sin_family = AF_INET;
      name.sin_port = 0;
      name.sin_addr.s_addr = htonl (INADDR_ANY);
      if (bind (sock, (struct sockaddr*) &name, sizeof (name))) {
         close (sock);
         return 0;
      }
   
      /* make socket non blocking */
      if (((flags = fcntl (sock, F_GETFL, 0)) == -1) ||
         (fcntl (sock, F_SETFL, flags | FNDELAY))) {
         close (sock);
         return 0;
      }
   
      /* get the internet address */
      name.sin_family = AF_INET;
      name.sin_port = htons (_PING_PORT);
      if (nslookup (hostname, &name.sin_addr) < 0) {
         close (sock);
         return 0;
      }
   
      /* try to connect */
      if ((connect (sock, (struct sockaddr*) &name, 
         sizeof (struct sockaddr_in)) >= 0) || 
         (errno == ECONNREFUSED)) {
         /* connection open or refused */
         close (sock);
         return 1;
      }
      /* connection failed */
      if (errno != EINPROGRESS) {
         close (sock);
         return 0;
      }
      /* connection in progress: test completion with select */
      nset = socketWait(sock, timeout, wm_write);
      if (nset > 0) {
         /* connected */
         close (sock);
         return 1;
      }
      else {
         /* timed out */
         close (sock);
         return 0;
      }
   }

/*----------------------------------------------------------------------*/
/*                                                         		*/
/* External Procedure Name: setRcvBufLen				*/
/*                                                         		*/
/* Procedure Description: Set the receive buffer length                 */
/*                                                         		*/
/* Procedure Arguments: socket, length          	       		*/
/*                                                         		*/
/* Procedure Returns: 0 if successful				        */
/*                                                         		*/
/*----------------------------------------------------------------------*/
int setRcvBufLen(int sock, int len) {
    return setsockopt(sock, SOL_SOCKET, SO_RCVBUF, (char*)&len, sizeof(int));
}

/*----------------------------------------------------------------------*/
/*                                                         		*/
/* External Procedure Name: setSndBufLen				*/
/*                                                         		*/
/* Procedure Description: Set the send buffer length                    */
/*                                                         		*/
/* Procedure Arguments: socket, length          	       		*/
/*                                                         		*/
/* Procedure Returns: 0 if successful				        */
/*                                                         		*/
/*----------------------------------------------------------------------*/
int setSndBufLen(int sock, int len) {
    return setsockopt(sock, SOL_SOCKET, SO_SNDBUF, (char*)&len, sizeof(int));
}

/*----------------------------------------------------------------------*/
/*                                                         		*/
/* External Procedure Name: socketWait					*/
/*                                                         		*/
/* Procedure Description: Wait until a condition is satisfied  		*/
/*                                                         		*/
/* Procedure Arguments: socket, maximum time, condition	       		*/
/*                                                         		*/
/* Procedure Returns: true if successful				*/
/*                                                         		*/
/*----------------------------------------------------------------------*/
int socketWait(int sock, wait_time maxtime, enum wait_mode mode) {
#ifdef DEBUG_PRT
   printf("Wait %f seconds for socket %i ...", maxtime, sock);
#endif

   /*-------------------------------  Get the timeout specifier         */
   struct timeval t, *timeout;
   if (maxtime < 0) {
      timeout = NULL;
   } else {
      long nSec = (long) maxtime;
      t.tv_sec  = nSec;
      t.tv_usec = (long) ((maxtime - nSec) * 1000000.0);
      timeout = &t;
   }

   /*-------------------------------  Set up the socket mask            */
   fd_set fdmask;
   FD_ZERO (&fdmask);
   FD_SET (sock, &fdmask);

   /*-------------------------------  Wait for the condition            */
   int nset = 0;
   switch (mode) {
   case wm_read:
#if DEBUG_PRT > 1
      if (timeout) printf("read timeout %lis+%lius\n", t.tv_sec, t.tv_usec);
      else         printf("read wait\n");
#endif
      nset = select(sock+1, &fdmask, NULL, NULL, timeout);
      break;
   case wm_write:
#if DEBUG_PRT > 1
      if (timeout) printf("write timeout %lis+%lius\n", t.tv_sec, t.tv_usec);
      else         printf("write wait\n");
#endif
      nset = select(sock+1, NULL, &fdmask, NULL, timeout);
      break;
   case wm_ctrl:
#if DEBUG_PRT > 1
      if (timeout) printf("ctrl timeout %lis+%lius\n", t.tv_sec, t.tv_usec);
      else         printf("ctrl wait\n");
#endif
      nset = select(sock+1, NULL, NULL, &fdmask, timeout);
      break;
   default:
      nset = -1;
   }
#ifdef DEBUG_PRT
   switch (nset) {
   case 1:
      printf(" success!\n");
      break;
   case 0:
      printf(" timeout!\n");
      break;
   default:
      printf(" interrupted!\n");
   }
#endif
   return nset;
}
