/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef _OSMBUF_HH_
#define _OSMBUF_HH_
//
//   Shared memory output stream buffer class
//
#include "PConfig.h"
// #define OSMBUF_TRACE

#if defined(OSMBUF_TRACE)
  #include <string>
  #include <iostream>
#endif

#if defined(__GNU_STDC_OLD)
#include <streambuf.h>
#else
#include <iostream>
#include <streambuf>
#endif

class LSMP_PROD;

/**  This class interfaces the shared memory buffer manager to the output
  *  stream class. It is based on the standard template class streambuf,
  *  and implements the virtual buffer management functions of streambuf.
  *  @memo Shared memory output stream class.
  *  @author John Zweizig
  *  @version 1.2; Last Modified March 3, 2008
  *  @ingroup IO_lsmp
  */
class oSMbuf : public std::streambuf {
  public:
#if defined(__GNU_STDC_OLD)
    typedef std::ios::open_mode openmode_type;
    typedef std::ios::seek_dir  seekdir_type;
#else
    typedef std::ios::openmode openmode_type;
    typedef std::ios::seekdir  seekdir_type;
#endif

  public:

    /**  Create an unopened stream buffer.
      *  @memo Default creator.
      */
    oSMbuf();

    /**  A stream buffer is created and attached to a shared memory partition. 
      *  Attaching the stream buffer to the memory partition is described
      *  in the discussion of open().
      *  @brief Create and open a stream buffer.
      */
    oSMbuf(const char *partition, openmode_type mode=std::ios::out);

    /**  Detach (if necessary) and destroy a stream buffer object.
      */
    ~oSMbuf();

    /**  Attach a stream buffer object to a shared memory partition.
      *  The shared memory partition "partition" is found, or created if
      *  it does not already exist. A data producer is created for the 
      *  partition to be used by the stream buffering functions. Buffering 
      *  of the stream uses the shared memory directly.
      */
    oSMbuf* open(const char *partition, openmode_type mode=std::ios::out);

    /**  Close and release the output buffer
      */
    void close(void);

    /**  Implementation of virtual streambuf::doallocate.
      */
    int doallocate(void);

    /**  Implementation of virtual streambuf::overflow.
      */
    int overflow(int c);

    /**  Set the shared memory event id field
     */
    void set_id(int evtid);

    /**  Implementation of virtual streambuf::underflow.
      */
    int underflow() {return std::streambuf::traits_type::eof();}

    /**  Implementation of virtual streambuf::underflow(int).
      */
    int underflow(int c) {return std::streambuf::traits_type::eof();}

    /**  Implementation of virtual streambuf::seekoff.
      */
    std::streampos seekoff(std::streamoff off, seekdir_type dir, 
			   openmode_type mode);

    /**  Implementation of virtual streambuf::seekoff.
      */
    std::streampos seekpos(std::streampos pos, openmode_type mode);

    /**  Implementation of virtual streambuf::sync.
      */
    int sync(void);

#ifndef __GNU_STDC_OLD
  /** Set buffers for obsolete gnu API.
   */
  void setb(char* b, char* eb, int a=0) {}

  /** number of bytes of output data for obsolete gnu API.
   */
  int out_waiting(void) {return this->pptr() - this->pbase();}
#endif

  private:
    ///                   Point to the shared memory producer interface.
    LSMP_PROD *mProducer;
    ///                   Pointer to the buffer currently being written
    char      *mBuffer;
    ///                   Length of buffer currently being written.
    int        mLength;
#ifdef OSMBUF_TRACE
  public:
    void pTrace(void) const {mTrace.pTrace();}
  private:
    class trace {
    #define maxTrace 8
    public:
        trace() : nTrace(0) {}
        void tracepoint(const std::string& funct, int arg1=0, int arg2=0);
        long returns(long i);
        void pTrace(void) const;
        void clear(void);
    private:
        int nTrace;
        std::string mFuncName[maxTrace];
        long mArg1[maxTrace];
        long mArg2[maxTrace];
        long mRet[maxTrace];
    } mTrace;
#endif
};

#ifdef OSMBUF_TRACE
inline void 
oSMbuf::trace::tracepoint(const std::string& funct, int arg1, int arg2) {
    int inx = nTrace % maxTrace;
    mFuncName[inx] = funct;
    mArg1[inx] = arg1;
    mArg2[inx] = arg2;
    nTrace++;
}

inline void 
oSMbuf::trace::clear(void) {
    nTrace = 0;
}

inline long 
oSMbuf::trace::returns(long ret) {
  int inx = (nTrace-1) % maxTrace;
  mRet[inx] = ret;
  return ret;
}
#endif

#endif  //  _OSMBUF_HH_
