/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//   LSMP is the Ligo Shared Memory Partition object class
//
#ifndef LSMP_HH
#define LSMP_HH

#include "gds_shmem.hh"
#include <sys/types.h>
#include <string>
#include <iosfwd>

#define LSMP_LNAME  16                      // maximum partition name length

//---------------------------------------  Consumer flags.
#define READALL  0x0001
#define EVWAIT   0x0002
#define NOWAIT   0x0004
#define SHOWALL  0x0008

/** LSMP provides basic access to the shared memory partition. 
  * The data elements of the LSMP class keep track of partition status, and 
  * point to the different control tables within the partition. These are 
  * protected from the end user, but are available to the derived classes.
  * @memo Shared Memory Base Class.
  * @author   J. Zweizig
  * @version  1.3; Last Modified May 14, 2010
  * @ingroup IO_lsmp
  */
class LSMP {
  public:
    /**  Ligo shared memory error enumerator.
     */
    enum LSMP_error {
	NoError = 0,            ///< No errors detected in last operation
	NotAttached,            ///< Partition is not attached
	IncompatiblePartition,  ///< Partition size is incompatible with request
	CantGetSem,             ///< Failed to get a semaphore
	NoFreeID,               ///< No free IDs
	CantAttach,             ///< Failed to attach shm partition
	CantCreate,             ///< Failed to create shm partition.
	CantCreateSem,          ///< Failed to create a semaphore.
	NoPriviledge,           ///< No privilege for requested operation.
	BadVersion,             ///< Version ID mismatch
	NoPartition             ///< Named partition was not found.
    };

    /**  Event id data type used to extract specific events. For Ligo, the
      *  event identifiers are GPS time stamps.
      */
    typedef int eventid_type;

  public:
    ///  The last error to occur while accessing the partition.
    LSMP_error error;

    /**  Test that a partition is enabled for use.
      *  \brief Test if partition is attached.
      *  \return True if partition is attached.
      */
    bool valid() const {
	 return accessed;}

    /**  Test that a partition has been found.
      *  \brief Test if partition is attached.
      *  \return True if partition is attached.
      */
    bool attached() const;

    /**  Write a formatted dump of the current status of the partition to 
      *  the standard output stream.
      *  @param flg Flag indicating which portions of partition to dump
      *  @memo Print the partition status to cout
      */
    void dump(int flg=0xff) const;

    /**  Write a formatted dump of the current status of the partition to 
      *  the specified output stream. Individual sections of the dump may 
      *  be selected with \c flg. as follows:
      *  <table>
      *  <tr><td>flag</td><td>Data to be printed</td></tr>
      *   <tr><td>  1</td><td>Header (partition name)</td></tr>
      *   <tr><td>  2</td><td>Control/status data</td></tr>
      *   <tr><td>  4</td><td>Buffer information</td></tr>
      *   <tr><td>  8</td><td>Consumer information</td></tr>
      *   <tr><td>256</td><td>Verbose printouts</td></tr>
      *   <tr><td>512</td><td>Suppress headers</td></tr>
      *  </table>
      *  @memo Print the partition status to an output stream.
      *  @param Out output stream to which the status dump will be written.
      *  @param flg Or of flag bits indicating portions of dump to be included.
      */
    void dump(std::ostream& Out, int flg=0xff) const;

    /**  Turn On/Off request synchronization.
      *  \brief Set request synchronization flag
      *  \param on Use request synchronization if true.
      */
    void rqsynch(bool on);

    /**  Set the buffer allocation to a specified mode.
      *  <table>
      *    <tr><td>mode=0</td><td>A buffer is released (moved to free list) any
      *                           time is not reserved and not in use.</td></tr>
      *    <tr><td>mode=1</td><td>A buffer is released after it has been seen 
      *                           by at least one consumer and is no longer 
      *                           reserved or in use.</td></tr>
      *    <tr><td>mode=2</td><td>A buffer is not released except if it is 
      *                           unreserved, not in use and needed to fill 
      *                           a producer request.</td></tr>
      *  </table>
      *  \brief Set allocation mode.
      *  \param mode Buffer allocation mode.
      */
    void bufmode(int mode);

    /**  Set a flag to indicate whether the partition is to be deleted 
      *  when it is detached. If \a flag is true, the partition will not be
      *  deleted when it is detached and the use count falls to zero.
      *  @memo Keep (or destroy) the partition on release.
      *  \param flag Keep (true) or destroy (false) the partition.
      */
    void keep(bool flag=true);

    /**  Locks the shared memory partition into memory. In order for 
      *  lock to work, the process must be able to set the effective
      *  user ID to 0. 
      *  @memo Lock the shared memory partition into memory
      *  \param flag Lock (true) or unlock (false) the partition.
      *  \return true on error
      */
    bool lock(bool flag=true);

    /**  Get a description of the last error.
      *  \brief Latest error
      *  \return Constant pointer to string describing latest error.
      */
    const char *Error(void) const;

    /**  Get the length of the buffers in the attached partition.
      *  @memo Buffer length.
      *  \returns Maximum data length for each buffer.
      */
    int getBufferLength(void) const;

    /**  Get the number of buffers in the attached partition.
      *  @memo Buffer count.
      *  \returns Number of buffers in the attached partition.
      */
    int getBufferCount(void) const;

    /**  Get the share memory partition name string.
      *  @memo Partition name.
      *  \returns Partition name string.
      */
    std::string name(void) const;

    /**  The default constructor creates the LSMP data structure without 
      *  attaching it to a shared memory partition.
      *  @memo Default constructor. 
      */
    LSMP(void);

    /**  The copy constructor copies an existing partition handle and links 
      *  it into the process list. Note that no new partition is created.
      *  @memo Copy constructor.
      *  \param copy Partition to be copied.
      */
    LSMP(LSMP& copy);

    /** Attach and access an existing named partition.
      * @memo Access existing partition.
      * @param name Name of existing partition.
      */
    explicit LSMP(const char* name);

    /**  Create a new partition with a specified name, number of buffers 
      *  and buffer size (in bytes). This constructor will attach an 
      *  existing partition with the correct name if the number of buffers
      *  and the buffer size are greater than or equal to the specified
      *  values.
      *  \brief Partition creation constructor.
      *  \param name Name of partition to be created.
      *  \param nbuf Number of buffers in the partition to be created.
      *  \param bsize Size (in bytes) of each buffer in the partition.
      */
    LSMP(const char* name, int nbuf, int bsize);

    /**  Deaccess and detach a partition. The physical shared memory 
      *  partition and the semaphores used for synchronization are deleted
      *  if the use count falls to zero and the keep flag is not set.
      *  @memo The partition destructor.
      */
    virtual ~LSMP(void);

    /**  Get a shared memory parameter value. Valid parameters are:
      *  <ul>
      *    <li> \b exists: 1 if partition exists.</li>
      *    <li> \b name:   Partition name.</li>
      *    <li> \b shmid:  Shm identifier.</li>
      *    <li> \b version: Lsmp version number.</li>
      *    <li> \b nbuf:   Number of partition buffers.</li>
      *    <li> \b lbuf:   Length of partition buffers.</li>
      *    <li> \b ncons:  Number of active consumers.</li>
      *    <li> \b use_count: Number of processes using the partition.</li>
      *    <li> \b buffer_tot: Total buffers written to partition.</li>
      *    <li> \b maxcons: Number of consumer slots.</li>
      *    <li> \b semglobal: Global semaphore cluster ID.</li>
      *    <li> \b fullbuf: Number of buffers in the full queue.</li>
      *    <li> \b freebuf: Number of buffers in the free queue.</li>
      *    <li> \b usedbuf: Sum of buffer use counts.</li>
      *    <li> \b last_ID: ID of Last buffer in the full queue.</li>
      *    <li> \b vstart: Virtual start of data (last_ID - buffer_tot)</li>
      *  </ul>
      *  @memo Get a parameter.
      *  @param par parameter name.
      *  @return parameter value.
      */
     std::string getStat(const std::string& par) const;

    /**  Test a shared memory status flag
      *  @memo Get a flag value.
      *  @param x Flag number.
      *  @return Flag status.
      */
     bool testFlag(int x) const;

     /**  Get the lock count. The lock count is incremented each time gate(true)
       *  is called successfully.
       *  \brief Lock count.
       *  \return Number of times gate() has been called.
       */
     int getLockCount(void) const {return mLockCount;}

     /**  Clear the lock count.
       *  \brief Clear the lock count.
       */
     void clrLockCount(void) {mLockCount=0;}

  protected:
    /**  Release the buffers reserved or allocated by the specified consumer 
      *  and mark the consumer as free for allocation.
      *  @memo Release a specified consumer
      *  @param icon The number of the consumer to be released.
      *  \return -1 if error occurred in freeing consumer, otherwise zero.
      */
    int free_consumer(int icon);

  protected:
    ///  Pointer to the partition control area.
    struct LSMP_global* pointer;

    ///  Pointer to the partitions consumer database
    struct LSMP_consbk* conptr;

    ///  Pointer to the partitions buffer database.
    struct LSMP_buffer* bufptr;

    /**  Get the data start address of a specified buffer.
      *  \brief Get data address
      *  \param ibuf Number of buffer for which the address is requested.
      *  \return Data start address in specified buffer.
      */
    char* buffer_addr(int ibuf) const;

    /**  Get the current data length of a specified buffer.
      *  \brief Get data length.
      *  \param ibuf Number of buffer for which the length is requested.
      *  \return Number of data bytes currently in specified buffer.
      */
    int buffer_length(int ibuf) const;

    /**  Request or release exclusive write control over the partition
      *  control structure.
      *  \brief Request/free exclusive control of partition.
      *  \param lok Lock the gate if true, otherwise unlock
      *  \return Request succeeded if true.
      */
    bool gate(bool lok) const;

    /**  Return the global process ID used by this process for this partition.
      *  \brief Global process ID.
      *  \return Global process ID used by this partition.
      */
    pid_t my_procid(void) const;

    /**  Zero the use count and clean the keep flag for the named (not 
      *  attached) partition. The named partition will be like totally
      *  blown away after this.
      *  \brief Zero the partition use count and delete the partition.
      *  \param name Partition name string.
      */
    void Zeuss(const char* name);

public:
    ///  Access an attached partition.
    bool access();

    ///  DeAccess a partition.
    void deaccess();

    ///  Find a named partition and attach it.
    bool find(const std::string& name);

    ///  Release the partition.
    void release();

private:
    ///  Create a named partition with 'nbuf' buffers of length 'size'.
    bool make(const char *name, int nbuf, int size);

  private:
    /**  Shared memory partition.
     */
    gds_shmem  _partition;

    /**  Indicates that the partition has been accessed (i.e. the use count 
      *  has been incremented) by a derived class.
      */
    bool accessed;

    /**  Inhibits deleting the partition once it is released by the current 
      *  process.
      */
    bool kpflg;

    /**  The number of bytes in the attached partition.
      *  @memo Buffer length;
      */
    int NByte;

    /**  The number of buffers in the attached partition.
      *  @memo Buffer count.
      */
    int NBuff;

    ///  Count lock delays
    mutable int  mLockCount;
};

/**  LSMP_ACCESS provides an interface to the shared memory partition for 
  *  data management functions. The shared memory partition is attached by 
  *  the LSMP_ACCESS constructor.
  *  @memo Shared Memory Maintenance Class.
  */
class LSMP_ACCESS : public LSMP {

  //------------------------------------  Public data and methods
  public:
    /**  LSMP_ACCESS() constructs a null accessor, not connected to any 
      *  partition.
      *  @memo Default constructor.
      */
    LSMP_ACCESS();

    /**  Initialize a accessor structure and attach it to an existing 
      *  partition. If the named partition doesn't exist, no partition is
      *  attached.
      *  @memo Attach to a partition.
      *  \param part Partition name string.
      */
    LSMP_ACCESS(const char *part);

    /**  If a partition is attached, it is released (see LSMP::release).
      *  @memo Accessor destructor.
      */
    ~LSMP_ACCESS();

    /**  One or more consumers are tested for an existing process. If the 
      *  process doesn't exist, the consumer is deleted. Buffers are also 
      *  released if no consumers exist or if they are reserved by a 
      *  non-existent producer.
      *  @memo Repair a partition.
      *  \param ID Consumer ID or -1 for all consumers.
      */
    void Repair(int ID=-1);

    /**  The data in a specified buffer are copied to the specified output
      *  stream. No consumer synchronization is performed.
b      *  @memo Binary copy one buffer to an output stream.
      *  @param jBuf   ID iof the buffer to be spewed.
      *  @param Out    stream to receive the data.
      *  @param length Number of bytes to copy.
      */
    void Spew(int jBuf, std::ostream& Out, int length=0);

  private:
};

//======================================  LSMP inline accessors.
inline bool 
LSMP::attached() const {
    return _partition.is_attached();
}

inline int 
LSMP::getBufferLength(void) const {
    return NByte;
}

inline int 
LSMP::getBufferCount(void) const {
    return NBuff;
}

#endif // LSMP_HH
