/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef LMSG_TRANSPORTTCP_HH
#define LMSG_TRANSPORTTCP_HH

#include "lmsg/MsgTypes.hh"

namespace lmsg {
  class Buffer;
  class BufferPool;
  class Message;
  class MsgAddr;
  class MsgHeader;
  class Socket;
  class TCPSocket;

  /**  TransportTCP provides an API for transporting messages with the lmsg
    *  protocol. This API formats mesages with the proper headers and sends
    *  or receives the messages using the lmsg::Socket TCP/IP message API.
    *  @memo Message transport API.
    *  @author John Zweizig
    *  @version 1.1; Modified April 21, 2000.
    *  @ingroup IO_lmsg
    */
  class TransportTCP {
  public:

    /**  Open options.
      */
    enum openflags {o_none, o_client=1, o_async=2};

    /**  Construct a message transporter.
      *  @memo TransportTCP constructor.
      */
    TransportTCP(void);

    /**  Close any open socket and destroy the message transporter.
      *  @memo TransportTCP destructor.
      */
    ~TransportTCP(void);

    /**  Get the address of the current socket.
      *  @memo Get address.
      *  @return Address to which the socket is bound.
      */
    MsgAddr getAddr(void) const;

    /**  Test whether transport is a client port.
      *  @memo Test for client socket.
      *  @return True if socket is client.
      */
    bool    isClient(void) const;

    /**  Test whether transport is connected to a peer port.
      *  @memo Test for connected socket.
      *  @return True if socket is connected.
      */
    bool    isConnected(void) const;

    /**  Test whether transport has an open socket.
      *  @memo test for open socket.
      *  @return True if socket is open.
      */
    bool    isOpen(void) const;

    /**  Get the current debugging level.
      *  @memo Get debug level.
      *  @return Current debugging level.
      */
    index_type getDebug(void) const;

    /**  Open a socket with the specified option flags. An address to which 
      *  the socket will be bound may be specified if desired. If no address
      *  is specified, the socket will be bound to an unused IP address.
      *  @memo Open a socket.
      *  @return Lmsg error codes (OK or SysError).
      *  @param flags Bit mask specifying socket handling options.
      *  @param addr  Optional IP address to which the socket is to be bound.
      */
    error_type open(mask_type flags, const MsgAddr* addr=0);

    /**  Return the socket to the appropriate pool.
      *  @memo Close the socket.
      */
    void    close(void);

    /**  Make a connection with the specified IP address. Once a connection 
      *  has been made messages may only be sent to or received from the 
      *  specified partner socket.
      *  @memo Connect to an IP partner address.
      *  @return Lmsg error codes (OK or SysError).
      *  @param  addr Address of partner to which the connection will be made.
      */
    error_type connect(const MsgAddr& addr);

    /**  Disconnect from a partner.
      *  @memo Disconnect from a partner.
      *  @return Lmsg error codes (OK or SysError).
      */
    error_type disconnect(void);

    /**  A message is received. The header of the message is copied to 'from' 
      *  and the message text (up to the specified maximum length) is copied 
      *  to 'b'. The buffer pointer may be coded as zero if the maximum 
      *  length is zero. receive returns with a TimeOut error code if no 
      *  message is received within 'timeout' seconds. If 'timeout' < 0,  
      *  receive will not return until either a message is received, or the 
      *  the wait is interrupted by a caught signal.
      *  @memo Receive a message.
      *  @return Lmsg error codes (OK, TimeOut, Continue, SysError).
      *  @param from    Received message header.
      *  @param b       Pointer to the buffer to receive the message text;
      *  @param maxlen  Length of buffer 'b'.
      *  @param timeout Maximum time in seconds to wait for a message.
      */
    error_type receive(MsgHeader& from, char* b, size_type maxlen, 
		    wtime_type timeout=-1.0);

    /**  A message is received. The header of the message is copied to 'from' 
      *  and the message text (up to the specified maximum length) is copied 
      *  to 'msg'. receive returns with a TimeOut error code if no  message 
      *  is received within 'timeout' seconds. If 'timeout' < 0, receive will 
      *  not return until either a message is received, or the the wait is 
      *  interrupted by a caught signal.
      *  @memo Receive a message.
      *  @return Lmsg error codes (OK, TimeOut, Continue, SysError).
      *  @param from    Received message header.
      *  @param msg     Message buffer to receive the message text;
      *  @param timeout Maximum time in seconds to wait for a message.
      */
    error_type receive(MsgHeader& from, Message& msg, wtime_type timeout=-1.0);

    /**  A message is received. The messge header and text are returned in a 
      *  Buffer, the address of which is stored in 'Data'. The Buffer must
      *  be returned to the appropriate buffer pool with Data->Return()
      *  once the buffer is no longer needed. receive returns with a TimeOut 
      *  error code if no message is received within 'timeout' seconds. If 
      *  'timeout' < 0, receive will not return until either a message is 
      *  received, or it is interrupted by a caught signal.
      *  @memo Receive a message.
      *  @return Lmsg error codes (OK, TimeOut, Continue, SysError).
      *  @param Data Pointer to receive the filled buffer address.
      *  @param timeout Maximum time in seconds to wait for a message.
      */
    error_type receive(Buffer** Data, wtime_type timeout=-1.0);

    /**  A message is sent. The header of the message to be sent is supplied
      *  by 'to' which must contain the destination address, the message 
      *  type and the length of the message text. The message text is 
      *   contained in 'b'.
      *  @memo Send a message.
      *  @return Lmsg error codes (OK, Continue, SysError).
      *  @param to Message header.
      *  @param b  Pointer to the buffer containing the message text;
      */
    error_type send(const MsgHeader& to, const char* b);

    /**  A message is sent. The header of the message to be sent is supplied
      *  by 'to' which must contain the destination address and the 
      *  transaction and block IDs. The message text is contained in 'b'.
      *  @memo Send a message.
      *  @return Lmsg error codes (OK, Continue, SysError).
      *  @param to Message header.
      *  @param b  Message to be sent.
      */
    error_type send(const MsgHeader& to, const Message& b);

    /**  Wait for a message or a timeout.
      *  @memo Wait for a message.
      *  @return Lmsg error codes (OK, Continue, SysError).
      *  @param Timeout Timeout time.
      */
    error_type waitMsg(wtime_type Timeout);

    /**  Delete a socket. This method is to be used only for sockets that 
      *  are broken in some way e.g. unexpected messages have been received
      *  over the buffer.
      */
    void Purge(void);

    /**  Specify the pool from which buffers will be allocated to hold sent 
      *  or received messages.
      *  @memo Specify buffer pool.
      *  @param pool Pointer to the buffer pool.
      */
    void setBufferPool(BufferPool* pool);

    /**  Set the debugging level.
      *  @brief set the debug level
      *  @param level Debug level.
      */
    void setDebug(index_type level);

  private:
    error_type send(Buffer* b);

  private:
    mask_type      mFlags;
    TCPSocket*  mSocket;
    TCPSocket*  mCSocket;
    BufferPool* mBuffPool;
    index_type     mDebug;
  };
} // namespace lmsg

inline bool
lmsg::TransportTCP::isOpen(void) const {
    return (mSocket != 0);
}

inline bool
lmsg::TransportTCP::isClient(void) const {
    return (mFlags & o_client);
}

inline lmsg::index_type
lmsg::TransportTCP::getDebug(void) const {
    return mDebug;
}

#endif // LMSG_TRANSPORTTCP_HH
