/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef LMSG_GRINDER_HH
#define LMSG_GRINDER_HH
#include "lmsg/MsgTypes.hh"
#include <string.h>

namespace lmsg {

    /**  The Grinder class is used for machine-dependent conversion to the
      *  standard exchange format (e.g. byte-swapping).
      *  @memo Conversion to exchange format.
      *  @author J. Zweizig
      *  @version 2.0; Last modified may 18, 2007
      *  @ingroup IO_lmsg
      */
    class Grinder {
    public:
	/**  Construct a grinder for use in converting to and from the host
	  *  data format.
	  *  @brief Constructor
	  */
	Grinder(void);

	/**  Destry a grinder.
	  *  @brief Destructor.
	  */
	~Grinder(void);

	/**  Test whether the data byte ordering needs to be swapped.
	  *  @brief Test if swapping is needed.
	  *  @return True if swapping is needed.
	  */
	bool need_swap(void) const;

	/**  Swap byte ordering in place in memory of the specified data 
	  *  element.
	  *  @brief Swap data bytes in memory.
	  *  @param _data Data storage address
	  */
	template<typename T> void SwapT(T* _data);

	/**  Swap byte ordering of N data words in the _in array, storing the
	  *  result in the _out array. If the byte ordering of the current host
	  *  is the same as the standard byte ordering, this is equivalent to a
	  *  copy.
	  *  @brief Swap data bytes in memory.
	  *  @param _in  Pointer to data stored with standard byte ordering.
	  *  @param _out Data array to receive data with host ordering.
	  *  @param N    Number of data words to copy and swap.
	  */
	template<typename T> void SwapIn(const char* _in, T* _out, size_type N);

	/**  Swap byte ordering of N data words in the _data array, storing the
	  *  result in the out array. If the byte ordering of the current host
	  *  is the same as the standard byte ordering, this is equivalent to a
	  *  copy.
	  *  @brief Swap data bytes in memory.
	  *  @param _in Pointer to data stored with the host's byte ordering.
	  *  @param out Data array to receive data in stabdard byte order.
	  *  @param N   Number of data words to copy and swap.
	  */
	template<typename T> void SwapOut(const T* _in, char* out, size_type N);
    private:
	bool   mBigEnd;
    };   // class Grinder

    //==================================  Need swap
    inline bool
    Grinder::need_swap(void) const {
	return !mBigEnd;
    }

    //==================================  Swap templates
    template<typename T> 
    inline void 
    Grinder::SwapT(T* _data) {
	char* fp = reinterpret_cast<char*>(_data);
	char* bp = fp + sizeof(T);
	while (fp < bp) {
	    char t = *(--bp);
	    *bp = *fp;
	    *(fp++) = t;
	}
    }
	
    template<typename T> 
    inline void 
    Grinder::SwapIn(const char* _in, T* _out, size_type N) {
	memcpy(_out, _in, N*sizeof(T));
	if (!mBigEnd) {
	    for (size_type _i=0; _i<N; ++_i) SwapT(_out+_i);
	}
    }
	
    template<typename T> 
    inline void 
    Grinder::SwapOut(const T* _in, char* _out, size_type N) {
	memcpy(_out, _in, N*sizeof(T));
	if (!mBigEnd) {
	    T* _op = reinterpret_cast<T*>(_out);
	    for (size_type _i=0; _i<N; ++_i) SwapT(_op+_i);
	}
    }

    extern Grinder export_format_grinder;

}  // namespace lmsg

#endif // LMSG_GRINDER_HH
