/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "scandir.hh"
#include <cstdlib>
#include <cstring>
#include <sys/types.h>
#include <dirent.h>
#include <iostream>
#include <sys/stat.h>
#include <unistd.h>

using namespace std;

//======================================  Recast the handle
inline DIR*
cast_handle(void* x) {
    return reinterpret_cast<DIR*>(x);
}

//======================================  Default constructor
scandir::scandir(void)
  : _dirHandle(NULL), _dirPtr(NULL), _frameEntry(false)
{
    _dirEntry = new dirent;
}

//======================================  Open constructor
scandir::scandir(const std::string& dir)
  : _dirHandle(NULL), _dirPtr(NULL), _frameEntry(false)
{
    _dirEntry = new dirent;
    open(dir);
}

//======================================  Destructor
scandir::~scandir(void) {
    close();
    delete _dirEntry;
}

//======================================  Open a directory path
bool 
scandir::open(const std::string& dir) {
    _dirName   = dir;
    _dirHandle = ::opendir(_dirName.c_str());
    return is_open();
}

//======================================  Open a directory path
void 
scandir::close(void) {
    ::closedir(cast_handle(_dirHandle));
    _dirHandle = NULL;
}

//======================================  Get file name
std::string 
scandir::file_name(void) const {
    if (_dirPtr) return _dirPtr->d_name;
    return "";
}

//======================================  Get file path
std::string 
scandir::file_path(void) const {
    string rs;
    if (!_dirPtr) return rs;
    string::size_type l = _dirName.size() + strlen(_dirPtr->d_name) +2;
    rs.reserve(l);
    rs = _dirName; 
    rs += "/";
    rs += _dirPtr->d_name;
    return rs;
}

//======================================  Test if entry is a directory
bool 
scandir::is_dir(void) const {
    if (!_dirPtr) return false;
    if (_dirPtr->d_type == DT_DIR) return true;
    return false;
}

//======================================  Test if entry is a file
bool 
scandir::is_file(void) const {
    if (!_dirPtr) return false;
#ifdef _DIRENT_HAVE_D_TYPE
    if (_dirPtr->d_type == DT_REG)     return true;
    if (_dirPtr->d_type == DT_UNKNOWN) return true;
#else
    string path = file_path();
    struct stat f_stat;
    if (lstat(path.c_str(), &f_stat)) return false;
    if (S_ISREG(f_stat.st_mode)) return true;
#endif
    return false;
}

//======================================  Test if entry is a frame file
bool 
scandir::is_frame(void) {
    //cout << "Test frame: " << file_name() << " ... ";
    if (!is_file()) {
	//cout << "not file" << endl;
 	return false;
    }
    if (_frameEntry) {
	//cout << "already tested" << endl;
        return true;
    }

    //----------------------------------  Get the file name
    string name;
    file_name().swap(name);
    if (name.empty()) {
	//cout << " empty name" << endl;
	return false;
    }

    //----------------------------------  Find/check/remove the suffix
    string::size_type inx = name.rfind('.');
    if (inx == string::npos) {
	//cout << "no ext." << endl;
	return false;
    }
    if (name.substr(inx) != ".gwf") {
	//cout << "ext not .gwf" << endl;
	return false;
    }
    name.erase(inx);

    //----------------------------------  Find/convert the duration
    char* nxptr;
    inx = name.rfind('-');
    if (inx == string::npos) {
	//cout << "no -dt" << endl;
	return false;
    }
    _frameDt = ::strtol(name.c_str()+inx+1, &nxptr, 0);
    if (*nxptr || !_frameDt) {
	//cout << "dt not end or invalid" << endl;
	return false;
    }
    name.erase(inx);

    //----------------------------------  Find/convert the GPS
    inx = name.rfind('-');
    if (inx == string::npos) {
	//cout << "no -gps" << endl;
	return false;
    }
    _frameGPS = ::strtol(name.c_str()+inx+1, &nxptr, 0);
    if (*nxptr || !_frameGPS) {
        //cout << "gps not end or invalid" << endl;
 	return false;
    }

    _framePrefix = name.substr(0, inx);

    //??????????????????????????????????  Check for site name?
    // name.erase(inx);

    //----------------------------------  It's a frame!
    _frameEntry = true;
    //cout << "OK" << endl;
    return _frameEntry;
}

//======================================  Test if entry is a symlink
bool 
scandir::is_symlink(void) const {
    if (!_dirPtr) return false;
    if (_dirPtr->d_type == DT_LNK) return true;
    return false;
}

//======================================  Read the next entry.
bool 
scandir::next_entry(void) {
    _frameEntry = false;
    _dirPtr     = NULL;
    if (!is_open()) return false;
    int rc = ::readdir_r(cast_handle(_dirHandle), _dirEntry, &_dirPtr);
    if (rc || !_dirPtr) return false;
    return true;
}

//======================================  Test for a directory.
bool 
scandir::test_directory(const std::string& path) {
    struct stat f_stat;
    if (stat(path.c_str(), &f_stat) || !S_ISDIR(f_stat.st_mode)) return false;
    return true;
}
