/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef FRAMEF_HH
#define FRAMEF_HH

#include "fferrors.hh"
#include <string>
#include <iostream>
#include <inttypes.h>

/**  Class FrameF is a fast frame file reader. It understands the structure
  *  headers in a frame file and keeps track of the current structure and
  *  where it ends.
  *  @memo Fast frame file reader.
  *  @author John Zweizig
  *  @version 1.2; Modified July 27, 2000 
  *  @ingroup IO_futils
  */
class FrameF {
public:

    /** 8-byte integer data type.
      */
    typedef int64_t int8_t;

    /** 4-byte integer data type.
      */
    typedef int32_t int4_t;

    /** 2-byte integer data type.
      */
    typedef int16_t int2_t;


public:
    /**  Frame structure header.
      */
    struct StrHdr {
	/** Length of structure.
	  */
	long  length;
	/** CheckSum type code.
	  */
	short chksm;
	/** Structure type code.
	  */
	short ID;
	/**  Structure instance.
	 */
	int   Instance;
    };

    /** Link structure.
      */
    struct StrLink {
	/** Structure type.
	  */
	int2_t ID;
	/** Structure instance.
	  */
	int4_t Instance;
};

public:
  /**  Construct a Frame file reader and attach it to an existing stream. 
    *  The stream is specified.
    *  @memo File constructor.
    */
  FrameF(std::istream& in);

  /**  Destructor.
   */
  ~FrameF(void);

  //------------------------------------  Accessors
  /**  Test whether the input data stream is ready for reading.
    *  @memo Test input stream.
    *  @return true if input data stream is ready to read from.
    */
  bool   isOK(void) const;

  /**  Get the current structure ID.
    *  @memo Get the current structure ID.
    *  @return the current structure ID.
    */
  short  getID(void) const;

  /**  Get the current structure instance number.
    *  @memo Get the current structure instance number.
    *  @return the current structure instance number.
    */
  int    getInstance(void) const;

  /**  Get the current structure length. The length returned includes the
    *  structure header length.
    *  @memo Get the current structure length.
    *  @return Current structure length.
    */
  long   getLength(void) const;

  /**  Get the frame patch number.
    *  @memo Get the frame patch number.
    *  @return the frame patch number.
    */
  int    getPatch(void) const;

  /**  Get the structure header length for the version of the frame standard 
    *  to which this file conforms.
    *  @memo Get the frame version.
    *  @return the frame version.
    */
  int    getStrucHdrLen(void) const;

  /**  Get the version of the frame standard to which this file conforms.
    *  @memo Get the frame version.
    *  @return the frame version.
    */
  int    getVersion(void) const;

  //------------------------------------  Mutators
  /**  Read the frame header record.
    *  @brief Read the frame header record.
    */
  void   ReadHeader(void) throw (BadFile);

  /**  Advance the file position to the start of the next structure.
    *  @brief Skip to the next structure.
    */
  bool   NxStruct(void) throw (BadFile);

  /**  Read a single character data item.
    *  @brief Get a double.
    *  @return Double data word.
    */
  char   getChar(void) throw (BadFile);

  /**  Read a short integer data item. The byte ordering is swapped 
    *  to allow use by the host.
    *  @brief Get a double.
    *  @return Double data word.
    */
  short  getShort(void) throw (BadFile);

  /**  Read a float data item. The byte ordering is swapped 
    *  to allow use by the host.
    *  @brief Get a double.
    *  @return Double data word.
    */
  float  getFloat(void) throw (BadFile);

  /**  Read an integer data item. The byte ordering is swapped 
    *  to allow use by the host.
    *  @brief Get a double.
    *  @return Double data word.
    */
  int    getInt(void) throw (BadFile);

  /**  Read a structure link data item.
    *  @brief Get a double.
    *  @return Double data word.
    */
  void   getLink(StrLink& link) throw(BadFile);

  /**  Read a long integer data item. The byte ordering is swapped 
    *  to allow use by the host.
    *  @brief Get a double.
    *  @return Double data word.
    */
  long   getLong(void) throw (BadFile);

  /**  Read a string data item. The string is stored in the frame as a
    *  byte-count followed by the data.
    *  @brief Get a double.
    *  @return Double data word.
    */
  std::string getString(void) throw (BadFile);

  /**  Read a double precision float data item. The byte ordering is swapped 
    *  to allow use by the host.
    *  @brief Get a double.
    *  @return Double data word.
    */
  double getDouble(void) throw (BadFile);

  /**  Set the header OK flag to flase. This will force reading in the header
    *  before the next I?O operation.
    *  @brief Reset the header valid flag.
    */
  void   resetHeader(void);

  /**  Seek to the specified file position.
    *  @brief Seek file position.
    *  @param off  Offset in bytes.
    *  @param mode starting position.
    */
  void   Seek(int off, std::ios::seekdir mode);

  /**  Skip a specified number of bytes in the frame file.
    *  @brief skip data.
    *  @param off Number of bytes to skip.
    */
  void   Skip(int off);

  /**  Test whether the current byte offset is outside the current structure.
    *  @brief Test offset in structure range.
    *  @return true if offset is outside the structure.
    */
    bool testOffset(void) const;
     
private:
  std::istream& mIn;
  bool     mSwap;
  bool     mHeaderOK;
  char     mData[40];
  int      mVersion;
  StrHdr   mSHdr;
  long     mOffset;
  long     mHdrSz;
};

inline int 
FrameF::getVersion(void) const {
    return mVersion;
}

inline int 
FrameF::getPatch(void) const {
    return mData[6];
}

inline bool
FrameF::isOK(void) const {
    return mIn.good();
}

inline short
FrameF::getID(void) const {
    return mSHdr.ID;
}

inline int 
FrameF::getInstance(void) const {
    return mSHdr.Instance;
}

inline long
FrameF::getLength(void) const {
    return mSHdr.length;
}

inline int
FrameF::getStrucHdrLen(void) const {
    return mHdrSz;
}

inline void
FrameF::resetHeader(void) {
    mHeaderOK = false;
}

inline bool
FrameF::testOffset(void) const {
    return mOffset < 0 || mOffset > mSHdr.length;
}
#endif
