#ifndef DAQSOCKET_HH
#define DAQSOCKET_HH
//
//    This is a C++ interface for frame data.
//
#include <vector>
#include <string>
#include <map>
#include "gmutex.hh"
#include "sockutil.h"

#define DAQD_PORT 8088

//  Protect against undefined MAX_CHNNAME_SIZE
#ifndef MAX_CHNNAME_SIZE
#define MAX_CHNNAME_SIZE 60
#endif


/** \page Network Data Access API (C++)
    This API provides a client interface for the "classic" NDS server. It
    allows the user to get channel data over the network. The syntax
    implemented by this interface is:
    <table>
      <tr>
        <td>Function</td>
	<td>DAQSocket method</td>
	<td>NDS command</td>
      </tr>
      <tr>
        <td>Get data from specified time</td>
	<td>RequestData</td>
	<td>start net-writer \<start\> \<duration\> [{"channel" ...} | all];</td>
      </tr>
      <tr>
        <td>Get file names</td>
	<td>RequestNames</td>
	<td>start name-writer all;</td>
      </tr>
      <tr>
        <td>Get available channel list</td>
	<td>Available</td>
	<td>status channels [2];</td>
      </tr>
      <tr>
        <td>Get fast online data</td>
	<td>RequestOnlineData</td>
	<td>start fast-writer [{"channel" ...} | all];</td>
      </tr>
      <tr>
        <td>Get slow online data</td>
	<td>RequestOnlineData</td>
	<td>start net-writer [{"channel" ...} | all];</td>
      </tr>
      <tr>
        <td>Get trend data</td>
	<td>RequestTrendData</td>
	<td>start trend [60] net-writer \<start\> \<duration\> 
	    [{"channel" ...} | all];</td>
      </tr>
      <tr>
        <td>Stop a net-writer</td>
	<td>StopWriter</td>
	<td>kill net-writer nnnnnnnn;</td>
      </tr>
    </table>

    The data trasfer protocol is as follows. After connecting to the nds 
    server, the client sends a command. The commands corresponding to each
    DAQSocket method are listed above. The syntax of the comands are:
    \verbatim
       <command> [{<channel-1> [<rate-1>] [... <channel-n> [<rate-n>]]}];
    \endverbatim
    Each command is terminated by a semi-colon and a new-line character. 
    Upon receipt of a valid command, the servers replies with a 4-hex-digit 
    ascii response code. a non-zero response code indicates that no further 
    data will be sent for the command. After each writer (frame-writer, 
    name-writer, [trend] net-writer) is started, a writer id is returned as 
    an 8-hex digit ascii string. The writer ID is used by the StopWriter 
    method to terminate the writer. Following this, the writer sends an 
    online/offline mode flag as a binary 4-byte integer valie. A zero value 
    indicates an online writer.

    A typical online NDS client would do the following:
    \verbatim
    //---------  Construct a DAQD socket 
    DAQSocket nds;

    //---------  Open a socket to the specified server port.
    const char* servername = "nds-server:port";
    nds.open(servername);
    if (!nds.TestOpen()) fail("Open failed!");

    //---------  Specify the channel to be read.
    const char* chan = "channel name";
    if (!nds.AddChannel(chan)) fail("Add channel failed!");
    if (nds.RequestOnlineData()) fail("Data Request failed");

    //---------  Specify the channel to be read.
    float* Samples = new float[data_len];
    while (1) {
        int nData = nds.GetData((char*) Samples, data_len);
	if (nData <= 0) break;
        ...  Process data ...
    }
    \endverbatim
    @memo Access channel data through the network data server
    @author John Zweizig
    @version 0.1; Last modified March 5, 2008
    @ingroup IO_daqs
************************************************************************/

/*@{*/

/**  The channel database contains a description of all the available 
  *  channels including their names, sampling rates and group numbers.
  *  @brief Channel data-base entry.
  *  @author John Zweizig
  *  @version 1.0; last modified March 5, 2008
  *  @ingroup IO_daqs
  */
   struct DAQDChannel {
   ///                  The channel name
      char mName[MAX_CHNNAME_SIZE + 10];
   ///                  The channel group number
      int  mGroup;
   ///                  The channel sample rate
      int  mRate;
   ///                  The channel or testpoint number
      int  mNum;
   ///                  The channel bytes-per-sample value
      int  mBPS;
   ///                  The channel data type
      int  mDatatype;
   ///                  The channel front-end gain
      float mGain;
   ///                  The channel slope
      float mSlope;
   ///                  The channel offset
      float mOffset;
   ///                  The channel unit
      char mUnit[40];
   };

/**  The DAQD header record is sent before each block of data.
  *  @brief DAQD header record.
  *  @ingroup IO_daqs
  */
   struct DAQDRecHdr {
   ///            Data block length (in bytes) excluding this word.
      int Blen;
   ///            Data length in seconds.
      int Secs;
   ///            GPS time (in seconds) of the start of this data.
      int GPS;
   ///            Time offset of the data from the GPS second (in ns).
      int NSec;
   ///            Data block sequence number (first reply to a request is 0).
      int SeqNum;
   };

/**  DAQSocket provides a client interface to the Network Data Server.
  *  The server provides data in the CDS proprietary format or as standard
  *  frames. The interface may also be used to list channel names, or to
  *  specify the channels to be read in.
  *  @brief The DAQD socket class.
  *  @author John Zweizig and Daniel Sigg
  *  @version 1.2; last modified March 5, 2008
  *  @ingroup IO_daqs
  */
   class DAQSocket {
   public:
      /// pair representing data rate (in Hz) & byte per sample
      typedef std::pair<int, int> rate_bps_pair;

      /// list of channels: map between channel name and channel info
      typedef std::map<std::string, DAQDChannel> channellist;

      /// channel list iterator
      typedef channellist::iterator Channel_iter;

     ///  List of channel names to be read.
     channellist mChannel;
   
     /**  Construct an unopened socket.
       */
     DAQSocket();
   
     /**  Create a socket and connect it to the specified server. The system
       *  receive buffer fot the socket is set to the specified length. The 
       *  network address argument has the same syntax as that passed to
       *  DAQSocket::open().
       *  @memo Create a socket and connect it to a server.
       *  @param ipaddr Server IP address
       *  @param ipport Server IP port number
       *  @param RcvBufferLen System receive buffer lengtr.
       */
     explicit DAQSocket(const char* ipaddr, int ipport = DAQD_PORT,
                        long RcvBufferLen = 1048576);
   
     /**  Disconnect and close a socket.
       */
      ~DAQSocket();
   
     /**  The argument, ipaddr, specifies the IP address of the node on which 
       *  the network data server is running. It may be specified either as 
       *  a symbolic name or as four numeric fields separated by dots. open()
       *  \brief Open a socket and connect it to a server.
       *  \param ipaddr Server IP address (see above)
       *  \param ipport Server IP port number.
       *  \param RcvBufferLen Minimum length of system receive buffer for 
       *         this socket.
       *  \return Zero if successful, a positive non-zero error code if one 
       *          was returned by DAQD or -1.
       */
     int  open (const char* ipaddr, int ipport = DAQD_PORT, 
                long RcvBufferLen = 1048576);
   
     /**  Disconnect and close a socket.
       *  @memo Close the client socket.
       */
     void close();
   
     /**  Flushes the input data from the socket.
       *  @memo Flush input data.
       */
     void flush();
   
     /**  Test whether the socket was opened successfully.
       *  @memo Test if socket is open.
       *  @return true if socket is open and connected
       */
     bool isOpen() const {
       return mOpened;
     }
   
     /**  Test is request has been sent and nds writer was started. Note 
       *  that a subsequent closure or failure of the socket may not be
       *  detected.
       *  @memo Test if server is sending data. 
       *  @return True if request was alreday sent
       */
     bool isOn() const {
       return mWriterType != NoWriter;
     }
   
     /**  This function effectively countermands the RequestXXX() functions.
       *  StopWriter returns either the server response code or -1 if no 
       *  writer has been requested.
       *  @memo Stop a data writer.
       *  \return Zero if successful, a DAQD response code or -1.
       */
     int  StopWriter(void);
   
     /**  The network data server is requested to start a frame writer task.
       *  Only channels explicitly specified by AddChannel() will be written.
       *  \brief Start reading frame data.
       *  \return Zero if successful, a DAQD response code or -1.
       */
     int  RequestFrames(void);
   
     /**  The network data server is requested to start a net-writer task.
       *  Only channels explicitly specified by AddChannel() will be written.
       *  \brief Start reading online CDS data.
       *  \param fast If true, request 16Hz data segments.
       *  \param timeout Timeout period in seconds or -1 to disable timeouts.
       *  \return Zero if successful, a DAQD response code or -1.
       */
     int  RequestOnlineData (bool fast = false, long timeout = -1);
   
     /**  The network data server is requested to start a net-writer task
       *  for past data. Start time and duration are given in GPS seconds. 
       *  Only channels explicitly specified by AddChannel() will be written.
       *  \brief Start reading CDS data.
       *  \param start GPS start time.
       *  \param duration Length of requested data in seconds.
       *  \param timeout Timeout period in seconds or -1 to disable timeouts.
       *  \return Zero if successful, a DAQD response code or -1.
       */
     int  RequestData (unsigned long start, unsigned long duration, 
                       long timeout = -1);
   
     /**  Start reading CDS trend data.
       *  The network data server is requested to start a trend net-writer 
       *  task. Start time and duration are given in GPS seconds. 
       *  Only channels explicitly specified by AddChannel() will be written.
       *  \brief Read CDS trend data.
       *  \param start GPS start time.
       *  \param duration Length of requested data in seconds.
       *  \param mintrend Select minute trends (true) or second trends (false).
       *  \param timeout Timeout period in seconds or -1 to disable timeouts.
       *  \return Zero if successful, a DAQD response code or -1.
       */
     int  RequestTrend (unsigned long start, unsigned long duration,
                        bool mintrend = false, long timeout = -1);
   
     /**  The network data server is requested to start a name writer task.
       *  \brief Start reading file names.
       *  \param timeout Timeout period in seconds or -1 to disable timeouts.
       *  \return Zero if successful, a DAQD response code or -1.
       */
     int  RequestNames(long timeout = -1);
   
     /**  Execution is blocked until data are available to the socket. This
       *  can be used to wait for data after a request has been made. The
       *  calling function can then e.g. allocate a buffer before calling 
       *  GetData(), GetName() or GetFrame(). If poll is true the function
       *  returns immediately with 1 if data is ready, or 0 if not.
       *  \brief Wait for data to arrive.
       *  \param poll Select polling mode.
       *  \return Data are available (1) or not (0).
       */
     int  WaitforData (bool poll = false);
   
     /**  Add the channel specified by a DAQDChannel structure to the 
       *  request list.
       *  \brief Add a channel to the request list.
       *  \param chns Channel specifier.
       *  \return Zero on success or error code.
       */
     int AddChannel (const DAQDChannel& chns);
   
     /**  Add a channel with the specified name and rate/sample size to 
       *  the request list. All channels may be added by specifying "all" 
       *  instead of a channel name.
       *  \brief Add a channel to the request list.
       *  \param chan Channel name.
       *  \param rb   Sample rate and size encoded as a rate_bps_pair.
       *  \return Zero on success or error code.
       */
     int  AddChannel(const char* chan, 
                     rate_bps_pair rb = rate_bps_pair (0, 0));
   
     /**  Remove a channel from the request list.
       *  \brief Remove channel
       *  \param chan Channel name
       */
     void RmChannel(const char* chan);

     /**  A single block of data (including the header) is received and copied
       *  into the specified buffer. 
       *  \brief Receive block of data in the CDS proprietary format.
       *  \param buf Preallocated buffer to receive data.
       *  \param len Length of output buffer.
       *  \param timeout Timeout period in seconds or -1 to disable timeouts.
       *  \return data length (excluding the header), -1 on error is, 
       *          or 0 if an End-Of-File record is received.
       */
     int GetData(char* buf, long len, long timeout = -1);
   
   /**  A single block of data (including the header) is received and copied
     *  into a user buffer. A buffer of the correct length is allocated and 
     *  its address is returned in the specified pointer. The caller is 
     *  reponsible for deallocating the buffer using delete[].
     *  \brief Receive block of data.
     *  \param buf Address of pointer to receive buffer address.
     *  \param timeout Timeout period in seconds or -1 to disable timeouts.
     *  \return The data length (excluding the header), -1 if an error is 
     *          detected or 0 if an end-of file record is found.
     */
     int GetData (char** buf, long timeout = -1);

     /**  The next frame file name written by the NDS is copied to buf and the 
       *  data length is returned. The GetName returns -1 if a name-writer 
       *  hasn't been started, if the data buffer is too short, or if an error
       *  occurs in reading the data. GetData waits for a new message if one
       *  is not available from the socket.
       *  \brief Receive a file name.
       *  \param buf Preallocated buffer to receive frame name.
       *  \param len Length of output buffer.
       *  \return Zero if successful or -1.
       */
     long GetName(char *buf, long len);
   
     /**  A single data frame is received and copied to the specified buffer.
       *  The length of the Frame data is returned. GetFrame() returns -1 in
       *  case of an error or 0 if a trailer (end of file) block is received.
       *  \brief Receive a data frame.
       *  \param buf buffer to recieve frame data.
       *  \param len Length of buffer to receive frame data.
       *  \return Frame length, zero on end of file or -1.
       */
     long GetFrame(char *buf, long len);

     /**  The names, sample rates, etc. of all channels known by the server 
       *  are copied into the channel list. The list is preallocated by the 
       *  caller with N entries. Available() returns the number of entries 
       *  found or -1 if an error occurred. If the number of entries is 
       *  greater than N, only the first N are copied to list;
       *  \brief Get a list all known channels.
       *  \param list Pre-allocated channel list
       *  \param N    Number of entries in list.
       *  \return Number of channels or -1 if an error occurred.
       */
     int  Available (DAQDChannel list[], long N);
   
     /**  The names, sample rates, etc. of all channels known by the server 
       *  are copied into the channel vector. Available() returns the number
       *  of entries found or -1 if an error occurred.
       *  \brief List all known channels.
       *  \param list Vector to receive channel list
       *  \return Number of channels or -1 if an error occurred.
       */
     int  Available (std::vector<DAQDChannel>& list);
   
     /**  The network data server is requested to return start time and 
       *  duration of the data stored on disk.
       *  \brief Known time intervals.
       *  \param start    Variable to receive GPS start time.
       *  \param duration Variable to receive duration (in seconds)
       *  \param timeout Timeout period in seconds or -1 to disable timeouts.
       *  \return Zero on success or -1.
       */
     int  Times (unsigned long& start, unsigned long& duration, 
                 long timeout = -1);

     /**  The network data server is requested to return start time and 
       *  duration of the trend data stored on disk.
       *  \brief Known time intervals of trend data.
       *  \param start    Variable to receive GPS start time.
       *  \param duration Variable to receive duration (in seconds)
       *  \param mintrend Select minute trends (true) or second trends (false).
       *  \param timeout Timeout period in seconds or -1 to disable timeouts.
       *  \return Zero on success or -1.
       */
      int  TimesTrend (unsigned long& start, unsigned long& duration, 
                      bool mintrend = false, long timeout = -1);
   
   /**  Setting debug mode to true causes the following to be printed to cout: 
     *  all request text, the status code and reply text for each request, 
     *  the header of each data block received and the length of each data/text
     *  block received and its buffer size.
     *  \brief Set debug mode.
     *  \param debug Debug mode flag
     */
      void setDebug(bool debug=true) {
         mDebug = debug;}
   
     /**  If the abort "button" is used, recv/send will periodically check if
       *  it has become true and if yes, abort the transaction. Make sure
       *  to set abort to false before calling a method that receives data
       *  or sends a request.
       *  \brief Set abort button.
       *  \param abort Pointer to abort button status variable.
       */
     void setAbort (bool* abort) {
         mAbort = abort;}
   
     /**  This is the only means of testing whether the creator was able to
       *  connect to a socket.
       *  \brief Test that connection is open.
       *  \return True is the socket is open.
       */
     bool TestOpen(void) {
       return mOpened;}
   
   /**  The version and revision numbers of the server software are returned 
     *  in a single float as (Version + 0.01*Revision).
     *  \brief Get the server version ID.
     *  \return Server version as (Version + 0.01*Revision).
     */
      float Version(void) {
         return mVersion + 0.01*mRevision;}
   
   private:
     /**  SendRequest sends a request specified by 'text' to the server and
       *  reads the reply status and optionally the reply text. 'reply' is 
       *  a preallocated buffer into which the reply is copied and 'length' 
       *  is the size of the reply buffer. If 'reply' is omitted or coded 
       *  as 0, the status will not be read. The reply status is returned by 
       *  SendRequest and the length of the reply message is returned in 
       *  *Size if Size is non-zero. Only the first message is read into 
       *  reply, so in general *Size != length.
       *  \brief Send a request.
       *  \return Zero if successful, the DAQD response code or -1;
       */
     int SendRequest(const char* text, char *reply=0, long length=0, 
		     long *Size=0, wait_time maxwait=-1);
   
   /**  Receive data from the socket.
    *  Up to 'len' bytes are read from the socket into '*buf'. If readall
    *  is true, RecvRec will perform as many reads as is necessary to 
    *  fill 'buf'.
    */
      int RecvRec(char *buf, long len, bool readall=false, 
                 wait_time maxwait=-1);
   
   /**  Send a record data to the socket.
    *  Up to 'len' bytes are written to the socket from '*buf'.
    */
      int SendRec(const char *buffer, long length, wait_time maxwait = -1);
   
   /**  Receive a data header and data block.
    *  Up to 'len' bytes are read from the socket into '*buf'. If keep
    *  is true, RecvData will put the header record into the buffer, if it 
    *  is false, the header is discarded.
    */
      int RecvData(char *buf, long len, DAQDRecHdr* hdr=0, long timeout = -1);
   
   /**  Receive a data header and data block.
    *  A memory buffer of correct length is allocated automatically. The header 
    *  record will be put at the beginning of the buffer. Returns the number
    *  of data bytes, or <0 on error.
    */
      int RecvData(char **buf, long timeout = -1);
   
   private:
   /**  mutex to protect object.
    */   
      mutable thread::recursivemutex	mux;
   
   /**  Socket is open and connected.
    */
      bool  mOpened;
   
   /**  Debug mode was specified.
    */
      bool  mDebug;
   
   /**  Socket number.
    */
      int   mSocket;
   
   /**  Socket receiving buffer length.
    */
      long   mRcvBuffer;
   
   /**  Set if 'all' channels was specified.
    *  The channel list is ignored if this flag is set.
    */
      bool  mGetAll;
   
   /**  Set if bytes must be reordered.
    */
      bool  mReorder;
   
   /**  ID number of the data writer currently in use.
    */
      char  mWriter[8];
   
   /**  Writer type enumeration.
    *  This enumerator specifies the type of data requested through the 
    *  socket. It is set when a data writer is started and tested by the 
    *  data read methods when they are called.
    */
      enum {NoWriter, NameWriter, DataWriter, FrameWriter, FastWriter} 
      mWriterType;
   
   /**  Offline flag sent when the writer is started.
    */
      int   mOffline;
   
   /**  Server version number.
    */
      int   mVersion;
   
   /**  Server revision number.
    */
      int   mRevision;
   
   /**  Abort "button".
    */
      bool*   mAbort;
   
   };

/*@}*/

#endif  //  DAQSOCKET_HH
