/**  @name root Macros
  *  @memo Macros to display LIGO data under root.
  */
//@{

/** Plot two time series with different color traces on the same axes.
  * @memo Plot two time series.
  */
  void ABPlot(const TSeries& ts1, const TSeries& ts2, const char *opts=0);

/** Plot two spectra with different color traces on the same axes.
  * @memo Plot two spectra.
  */
  void ABSpectrum(const Spectrum& s1, const Spectrum& ts2, const char *opts=0);

/** Generate a Bode plot of a frequency series.
  * @memo Bode Plot
  */
  void Bode(const FSeries& fs, const char *opts=0);

/** Generate a Bode plot of a frequency series.
  * @memo Bode Plot
  */
  void Bode(const FSeries* fs, const char *opts=0);

/**  Generate a constantly updating strip chart of the Band limited RMS of
  *  a set of bands defined by a configuration file. Each channel 
  *  displays the power produced on a single channel in a limited band
  *  width. By default, the scale is logarithmic with a factor of 10 
  *  corresponding to half the pen spacing. The Channel configuration 
  *  for the strip chart is specified by an ascii file passed as an 
  *  argument to Chart. The format of the file is as follows:
  *  \begin{verbatim}

   Line 1: <nPens> <tAccum> <write> <nSteps>
   Line 2: <Title>
   Line 3: <output>
   ...   : <channel> <fLow> <fHigh>

   \end{verbatim}
   * The fields on the first 3 lines have the following meaning:
   *  \begin{verbatim}
   <nPens>     The number of pens to be displayed.
   <tAccum>    The accumulation period in seconds. Data from each
               specified channel will be accumulated for the period 
               specified by <tAccum>
   <write>     Specifies whether data are to be written to a log file.
               <write=0 indicates that no log file is to be written.
   <nSteps>    The number of time intervals to be displayed on the strip 
               chart.
   <Title>     A title for the chart. The title is displayed on the 
               strip chart window.
   <output>    a fully qualified file name for the strip chart log.
   \end{verbatim}
   *  Following these lines comes a description of the contents of each pen.
   *  One line must be specified for each pen containing:
   *  \begin{verbatim}
   <channel>      The name of the channel from which the data are derived.
   <fLow>         the minimum fraquency.
   <fHigh>        The maximum frequency.
   \end{verbatim}
   *  @memo Multi-Channel band limited RMS stripchart.
   *  @param Config Name of a configuration file defining the char parameters.
   ***********************************************************************/
void Chart(const char* Config);

/**  Evaluate a function of various input channels and display the result
  *  in either the time or frequency domain.
  *  @memo  Display a function of several input channels.
  *  @param File Optional file name containing a starting configuration.
  */
void eval(const char* file=0);

/**  pTrend plots the minimum maximum and average values from a trend 
  *  file. If the file name is not specified, the default data accessor
  *  (In) will be used. If the starting time is not specified, the first 
  *  available data are plotted. If the time intervval is not specified,
  *  the data from a single frame are plitted.
  *  @memo Plot a channel from a trend file.
  *  @param Channel Name of channel to be plotted.
  *  @param File    Input frame file name.
  *  @param t0      Starting time.
  *  @param dT      Length of time to plot.
  */
void pTrend(const char* Channel, const char* File=0, Time t0=0, 
	    Interval dT=0.0);

/**  Replay data from the output file produced by Chart.
  *  @memo replay a Multi-Channel Strip-chart.
  *  @param Chart Name of a Chart output file to be replayed.
  */
void Replay(const char* Chart);

/**  A root macro to plot the magnitude-squared of an FSeries on a log-log 
  *  scale. The current root TCanvas is overwritten.
  *  @memo Plot an FSeries spectrum.
  *  @param series Reference to the series to be plotted.
  *  @param Opts Options for TH1.
  */
void Spectrum(const FSeries& series, const char* Opts=0);

/**  A root macro to plot the magnitude-squared of an FSeries on a log-log 
  *  scale. The current root TCanvas is overwritten.
  *  @memo Plot an FSeries spectrum.
  *  @param series Pointer to the series to be plotted.
  *  @param Opts Options for TH1.
  */
void Spectrum(const FSeries* series, const char* Opts=0);

/**  A root macro to plot an FSpectrum on a log-log scale. The current root 
  *  TCanvas is overwritten.
  *  @memo Plot an FSpectrum.
  *  @param spectrum FSpectrum to be plotted.
  *  @param Opts TH1 options.
  */
void Spectrum(const FSpectrum& spectrum, const char* Opts=0);

/**  A root macro to histogram the contents of a TSeries. The amplitude 
  *  of each sample in the TSeries is plotted as an entry in the histogram.
  *  @memo Histogram sample amplitudes in a TSeries.
  *  @param series Pointer to the series to be histogrammed.
  *  @param Opts Options for TH1.
  */
void THist(const TSeries* series, const char* Opts=0);

/**  A root macro to histogram the sample ampliudes for a specified channel.
  *  The channel must have been read in using the ``In'' Dacc object. The
  *  number of samples histogrammed is defined by the channel sample rate
  *  and the time interval specified when fill data with In.fillData(). 
  *  @memo Histogram sample amplitudes for a channel.
  *  @param channel Name of the channel to be histogrammed.
  *  @param Opts Options for TH1.
  */
void THist(const char* channel, const char* Opts=0);

/**  A root macro to plot the contents of a TSeries. The amplitude 
  *  of each sample in the TSeries is plotted versus time using a root 
  *  histogram (TH1) object.
  *  @memo Plot sample amplitude vs time in a TSeries.
  *  @param series Pointer to the series to be histogrammed.
  *  @param Opts Options for TH1.
  */
void TPlot(const TSeries* series, const char* Opts=0);

/**  Plot the sample amplitudes of a specified channel vs. time. The channel 
  *  must have been read in using the ``In'' Dacc object. The time interval 
  *  plotted is specified when filling data with In.fillData(). 
  *  @memo Plot sample amplitudes vs. time for a channel.
  *  @param channel Name of the channel to be plotted.
  *  @param Opts Options for TH1.
  */
void TPlot(const char* channel, const char* Opts=0);

/**  Plot the contents of a TSeries vesus another TSeries. The amplitude 
  *  of each sample in #ty# is plotted versus the same time point in #tx#.
  *  @memo Plot sample amplitude vs time in a TSeries.
  *  @param tx Series providing the x coordinates of the points to be plotted.
  *  @param ty Series providing the y coordinates of the points to be plotted.
  *  @param Opts Options for TH2.
  */
void TXYPlot(const TSeries* tx, const TSeries* ty, const char* Opts=0);

/**  Plot the spectrum of a Time integrated Frequency series. The XSpectrum
  *  macro is useful in cases such as seismometers where the physically
  *  meaningful quantity (x-displacement) is the integral of the measured
  *  quantity (velocity).
  *  @memo Plot the time integral spectrum.
  *  @param fs   FSeries to be integrated/plotted.
  *  @param Opts Options for TH1.
  */
void XSpectrum(const FSeries& fs, const char* Opts=0);


//@}
