#ifndef DAQ_ERR_FILTER_HH
#define DAQ_ERR_FILTER_HH

#include "Pipe.hh"

/**  This class simulates DAQ errors. At present the error types included 
  *  are:
  *   - \em Word \em swapping: during S1 several 16-bit word variables had
  *       the order each sample pair swapped. This mode operates only on 
  *       2-byte data time series.
  *   - \em LSB \em errors: errors occurred randomly in the Least Significant
  *       Byte (LSB) of 16-byte data transfers during science runs S1 - S3. 
  *       The simulation code generates errors
  *       by jamming \c 0xff into the low order byte of random words. Fixed 
  *       or random width clusters are generated at random intervals as
  *       specified by the setClusterWidth(), setAvgWidth() and setErrRate()
  *       methods. The cluster start time is calculated by generating a
  *       random time offset from an exponential distribution with average
  *       time as specified by setRate(). The time is adjusted to the 
  *       previous 1/16th second boundary. If an average cluster width was 
  *       specified, a random cluster width is generated from a Poisson
  *       distribution with the specified average. If the average with
  *       is zero, the specified constant cluster width is used. If a width
  *       specified or generated to be zero, no cluster is generated at the
  *       calculated start time.
  *
  *  The types of errors to be simulated are set by either the constructor
  *  argument or by the setSubTypes method. The type string is a comma 
  *  separated list of error types with one or more of "swap" or "lsb".
  *  @author J. Zweizig
  *  @version 1.0; Last modified: May 5, 2004
  */
class DaqErrFilter : public Pipe {
public:
  /**  Daq error enumerator.
   */
  enum type_enum {
    tSwap, ///< Data byte-swap error
    tLSB,  ///< Least significant byte errors
    tMax   ///< Number of enumerated error syndromes
  };

public:
  using Pipe::apply;
  using Pipe::dataCheck;

  /**  Daq error filter constructor. The argument is a list of error types 
    *  to be simulated with syntax as specified in the setSubTypes method.
    *  @memo Daq error constructor.
    *  @param subtype List of errors to be simulated.
    */
  DaqErrFilter(const char* subtype);

  /**  Daq error destructor.
    */
  ~DaqErrFilter(void);

  //------------------------------------  Set parameters
  /**  Return a string containing a list of error types to be simulated.
    *  \brief Simulation error type list.
    *  \return List of enabled error simulation types
    */
  std::string getTypeString(void) const;

  /**  Enable simulation of the specified error type.
    *  \brief Enable error type
    *  \param s Error type name.
    *  \return true if specified type was successfully selected.
    */
  bool selectType(const std::string& s);

  /**  Specify an average number of errors to be generated in a cluster of 
    *  LSB errors. By default the average is ?? 
    *  @memo Set average cluster width
    *  @param avgWid Average number of errors in an LSB error cluster
    */
  void setAvgWidth(double avgWid);

  /**  Specify a fixed number of errors in a cluster. Note that to specify a 
    *  fixed cluster width, the average cluster width must be zero.
    *  @memo Set Fixed cluster width.
    *  @param wid Number of errors in a cluster.
    */
  void setClusterWidth(long wid);

  /**  Set the error rate
    *  @memo Error rate.
    *  @param F Average error rate in Hz.
    */
  void setErrRate(double F);

  /**  Set the error types to be simulated by this filter. The error types
    *  are specified as a comma separated list of error names. At present 
    *  the valid type names are "swap" and "lsb".
    *  @memo Set subtypes.
    *  @param s Subtype name string.
    */
  void setSubTypes(const std::string& s);

  //--------------------------------  Pipe methods
  /**  Apply the DaqErr filter to the specified time series.
    *  @memo Filter time series.
    *  @param x Input time series.
    *  @return Filtered time series.
    */
  TSeries apply(const TSeries& x);

  /**  Create an identically configured copy of the filter.
    *  @memo Clone the filter.
    *  @return Pointer to the newly created copy.
    */
  DaqErrFilter* clone(void) const;

  /**  Check that the input series has valid data.
    *  @memo Check for valid data.
    *  @param x Input time series.
    *  @exception runtime_error Thrown if data are not valid.
    */
  void dataCheck(const TSeries& x) const;

  /**  Get time of first data filtered.
    *  @memo Get first data time.
    *  @return %Time of first data filtered.
    */
  Time getStartTime(void) const;

  /**  Get time of most recent data filtered.
    *  @memo Get most recent data time.
    *  @return %Time of most recent data filtered.
    */
  Time getCurrentTime(void) const;

  /**  Test whether filter is in use. The filter is defined to be in use if
    *  it has been used to filter data since the last reset.
    *  @memo Test in use.
    *  @return True if in use.
    */
  bool inUse(void) const;

  /** Reset the filter. Mark it as not in use and zero out history.
   *  @memo Reset the filter.
   */
  void reset(void);

private:
  void roll(void);
  bool testType(type_enum t) const;

private:
  long   mSubTypes;
  long   mClustWidth;
  double mAvgWidth;
  double mErrRate;
  Time   mLsbErrTime;
  Time   mStartTime;
  Time   mCurrentTime;
};

//======================================  Inline methods
inline void 
DaqErrFilter::dataCheck(const TSeries& x) const {
}

inline Time
DaqErrFilter::getCurrentTime() const {
    return mCurrentTime;
}

inline Time
DaqErrFilter::getStartTime() const {
    return mStartTime;
}

inline bool
DaqErrFilter::inUse(void) const {
    return mStartTime != Time(0);
}

inline bool 
DaqErrFilter::testType(type_enum t) const {
    return ((1<<t) & mSubTypes) != 0;
}

#endif // !defined(DAQ_ERR_FILTER_HH)
