#ifndef MULTISTREAM_HH
#define MULTISTREAM_HH

#ifndef __CINT__
#include "MultiDacc.hh"
#include "SigFlag.hh"
#include "TrigRaw.hh"
#endif //  !defined(__CINT__)

/**  <h2> Introduction </h2>
  *  The data processing base class defines the basic routines necessary 
  *  for a data processing application. The %MultiStream class is designed to
  *  provide a similar environment to the DatEnv class, but with the support
  *  of several paralell input data streams. The functions
  *  provided by the environment are:
  *  <ul>
  *    <li> Parse common command line arguments.
  *    <li> Read data from one or more synchronized frame streams.
  *    <li> Copy requested data into time series (\c TSeries ) containers.
  *    <li> Handling attention interrupts.
  *    <li> Clean up on termination.
  *  </ul>
  *
  *  <h2> Writing a %MultiStream Monitor </h2>
  *  The strategy for writing a monitor program is to create a class 
  *  (\c MyClass ) based on %MultiStream including a constructor, destructor
  *  and \c ProcessData method. An attention handler may also be provided if
  *  desired. The operations to be performed by each of these methods
  *  are:
  *
  *  <b> Constructor (MyClass(int argc, const char* argv[])) </b>
  *  <ul>
  *    <li> Parse user specific arguments
  *    <li> Define parameters (see \c TrigPar class).
  *    <li> Configure the process.
  *    <li> Request data channels with \c MultiDacc::addChannel.
  *  </ul>
  *
  *  <b> Destructor (~MyClass(void)) </b>
  *  <ul>
  *    <li> Save modified parameters
  *    <li> Print results and statistics.
  *  </ul>
  *
  *  <b> Epoch processing (void ProcessData(void)) </b>
  *  <ul>
  *    <li> %Process data from one epoch.
  *    <li> Generate triggers if necessary.
  *    <li> Keep statistics.
  *  </ul>
  *
  *  <b> Interrupt handler (void Attention(void)) </b>
  *  <ul>
  *    <li> Monitor dependent interrupt processing.
  *  </ul>
  *
  *  The \c EXECDAT(myclass) macro is used to generate the main
  *  function. Definition and implementation templates are available 
  *  from DatTemplate.hh and DatTemplate.cc.
  *
  *  The \c EXEROOT(myclass) macro is used to generate a main function
  *  with \c root graphics. Note that the \c "TROOT.h" and \c "TApplication.h"
  *  header files must be included in your monitor implementation file 
  *  if \c EXEROOT is used.
  *
  *  <h2> Running the Monitor </h2>
  *  The base class constructor recognizes the following command line 
  *  arguments: 
  *  <table>
     <tr><td>-debug \<lvl\>      </td>
         <td> Set debug level to \<lvl\></td></tr>
     <tr><td>-inlist \<list\>    </td>
         <td> Define a single data input stream and set it to read  data from
	      the file path(s) listed in \<list\>. Several -inlist arguments
	      may be specified in which case each argument will define a new 
	      independent stream.</td></tr>
     <tr><td>-inlists \<llist\>  </td>
         <td> Define multiple streams from list files in \<llist\>. Each line
	      in \<llist\> specifies the path to a list file that in turn
	      contains the paths of the frames to be read by one stream.
	      </td></tr>
     <tr><td>-maxframes \<nFr\>  </td>
         <td> Stop after reading \<nFr\> frames</td></tr>
     <tr><td>-partition \<part\> </td>
         <td> Read from shared memory partition \<part\>.</td></tr>
     </table>
  *  If no input option is specified, a single stream is defined and set to
  *  read frames from the path(s) specified by the \c DMTINPUT environment
  *  variable. If DMTINPUT is also not defined, a single stream will be 
  *  defined to read data from the shared memory partition defined by 
  *  LIGOSMPART. If LIGOSMPART is also not specified, themonitor will terminte.
  *  Shared memory partitions may be included in any of the data streams 
  *  by using the pseudo-file system \e /online/\<partition\>.
  *
  *  When running under ROOT, the base class definition and implementation 
  *  files must be loaded before the monitor can be run. If these files 
  *  aren't provided automatically by the start-up script, they may be 
  *  loaded with: \verbatim
       gROOT->LoadMacro("MultiStream.hh");
       gROOT->LoadMacro("MultiStream.cc"); \endverbatim
  *  The monitor is run with the \c .x command or by loading it and calling 
  *  it as a function. See the note entitled "Online monitoring of LIGO Data
  *  Using ROOT" for more details.
  *  @memo Data processing monitor base class.
  *  @author John G. Zweizig
  *  @version $Id: MultiStream.hh 6684 2012-07-06 08:56:46Z john.zweizig@LIGO.ORG $
  */
class MultiStream {
  public:
    /**  The class constructor gets all the command line arguments in the 
      *  usual Unix/C argument list format. It sets up the environment 
      *  for processing and opens the frame input stream as directed by 
      *  the command line arguments.
      *  @memo Class constructor.
      *  @param argc Number of arguments specified.
      *  @param argv Pointers to arguments.
      */
    MultiStream(int argc, const char *argv[]);

    /**  Close the input stream and go away.
      *  @memo Class destructor.
      */
    virtual ~MultiStream();

    /**  MainLoop reads in data and invokes \c ProcessData for each data 
      *  epoch successfully read in. MainLoop exits when a \c SIGINT or 
      *  \c SIGTERM signal is caught, or when finish is called by a subsidiary 
      *  function (usually by \c ProcessData). This function is provided by 
      *  the base class.
      *  @memo Loop over data epochs.
      */
    void MainLoop(void);

    /**  ProcessData performs the monitor specific processing data from
      *  a single epoch. It must be defined as part of the derived monitor 
      *  class implementation.
      *  @memo %Process one epoch (supplied by user).
      */
    virtual void ProcessData(void)=0;

    /**  Attention performs the monitor specific attention interrupt 
      *  processing. Its implementation is optional. Execution of 
      *  the attention handler is deferred until the processing of the 
      *  current epoch has completed. This means that there may be a 
      *  significant delay between the delivery of the SIGUSR1 signal and
      *  the invocation of the handler. The ROOT version uses the ROOT 
      *  event handler as a source of When used with ROOT, the default 
      *  action is to terminate the monitor. When used in the background, 
      *  it defaults to no action. The execution of the attention handler is 
      *  synchronized to times between calls to ProcessData.
      *  @memo Attention interrupt handler (supplied by user).
      */
    virtual void Attention(void);

    /**  finish stops processing by setting a flag that breaks the main 
      *  loop. Note that finish returns to the calling function (generally 
      *  the user's ProcessData) which must exit normally for the process 
      *  to terminate.
      *  @memo Stop data processing and go away cleanly.
      */
    void finish(void) {mActive = false;}

    /**  Debug returns the debug level as specified by the "-debug nnn" 
      *  argument. If "-debug" isn't specified in the command line the 
      *  debug level defaults to 0.
      *  @memo Get the debug level.
      *  @return Debug printout level
      */
    int Debug(void) const {return mDebug;}

    /**  Return a constant reference to the data accessor.
      *  @memo Get the data accessor.
      *  @return Constant reference to the MultiDacc instance for this monitor.
      */
    const MultiDacc& getDacc(void) const {return mIn;}

    /**  Get number of defined input streams (data accessors).
      *  @memo Number of streams
      *  @return Number of defined input streams
      */
    unsigned int getNStream(void) {return mIn.getNDacc();}

    /**  Get specified individual data accessor
      *  @memo Get specfied %Dacc
      *  @param i Index of data accessor
      *  @return Pointer to the specified %Dacc instance.
      */
    Dacc* getDacc(unsigned int i) {return mIn.getDacc(i);}

    /**  Test for a %MultiStream command line argument. All %MultiStream 
      *  command line  arguments take a value in the following argument
      *  position.
      *  @memo Test if string is a %MultiStream command line argument.
      *  @param arg Argument string.
      *  @return true if the specified string is a %MultiStream argument.
      */
    bool isMultiStreamArg(const char* arg) const;

    /**  Align the time stride to an integer multiple of seconds.
      *  @memo set stride alignment.
      *  @param mult boundary alignment.
      *  @param off  boundary offset
      */
    void setStrideAlignment(unsigned long mult, double off);

    /**  Test whether the specified signal has been caught.
      *  @memo Test for a specified signal.
      *  @param sig Signal number.
      *  @return true if the specified signal was caught.
      */
    bool testAttnSignal(int sig) const;

  //------------------------------------  Private methods
  private:
    /**  Test if a termination interrupt occurred.
      *  @memo Test for termination.
      */
    bool testTerm(void) const;

    /**  Test if an attention interrupt occurred.
      *  @memo Test for attention.
      */
    bool testAttn(void);

  protected:
    /**  Return a reference to the data accessor.
      *  @memo Get the data accessor.
      *  @return Reference to the MultiDacc instance for this monitor.
      */
    MultiDacc& getDacc(void) {return mIn;}

    ///  Frame Input object
    MultiDacc  mIn;

#ifndef __CINT__
    /**  A raw trigger data identifier is encoded for each data epoch read.
      *  This field is omitted in the ROOT version.
      *  @memo Raw trigger data identifier for current epoch.
      *  @return Raw data identifier.
      */
    trig::TrigRaw  mRawID;
#endif  // __CINT__

  private:
    ///  Internal run flag
    bool     mActive;

#ifndef __CINT__
    ///  Termination signal handler
    SigFlag mTerm;

    ///  Attention signal handler
    SigFlag mAttn;
#endif // !def(__CINT__)
  
    ///  Debug Level
    int  mDebug;

    /**  Frame read error.
      *  mEOF is set when a frame read error occurs.
      */
    bool mEOF;

    /**  Attention signals.
      *  mAttnMask is set to a or of the bits corresponding to the Signals 
      *  that have been caught.
      */
    long mAttnMask;

    long mMaxFrames;

    long mAlignMult;
    double mAlignOff;
};

inline bool 
MultiStream::testAttnSignal(int sig) const {
    return ((1 << sig) & mAttnMask) != 0;
}

//
//   The EXECDAT macro generates the main function to execute a monitor
//   based on MultiStream. The argument is whatever name you have chosen
//   for your class.
//
#define EXECDAT(myclass) \
int main(int argc, const char *argv[]) { \
    myclass MyObject(argc,argv); \
    MyObject.MainLoop(); \
    return 0;}

//
//   The EXEROOT macro generates the main function to execute a monitor
//   based on MultiStream. This macro differs from EXECDAT in that it also 
//   sets up the root object structure which allows the derived class to 
//   use any root classes. The argument is whatever name you have chosen
//   for your class.
//
#define EXEROOT(myclass) \
TROOT root("MultiStream", "Graphical background monitor"); \
int main(int argc, const char* argv[]) { \
    int bc(argc); const char** bv=new const char*[bc]; \
    for (int i=0 ; i<bc ; i++) bv[i]=argv[i]; \
    TApplication theApp("App", &bc, const_cast<char**>(bv)); \
    myclass MyObject(argc,argv); \
    MyObject.MainLoop(); \
    return 0;}

#endif     //  MULTISTREAM_HH
